package com.estimote.cloud_plugin.secure.resolvers

import com.estimote.android_ketchup.kotlin_goodness.toHexString
import com.estimote.internal_plugins_api.cloud.secure.ResolvedEstimoteLocation
import com.estimote.internal_plugins_api.cloud.secure.SecureCloud
import com.estimote.internal_plugins_api.scanning.EstimoteLocation
import com.estimote.internal_plugins_api.scanning.EstimoteSecure
import com.estimote.internal_plugins_api.scanning.MacAddress
import com.estimote.internal_plugins_api.secure.SecurePacketResolver
import io.reactivex.Observable
import io.reactivex.Single
import io.reactivex.functions.BiFunction
import java.util.concurrent.TimeUnit

/**
 * Packet resolver that tracks received encrypted payloads and tries to resolve them in Cloud.
 * @author Estimote Inc. (contact@estimote.com)
 */
internal class EstimoteLocationSecurePacketResolver(private val secureCloud: SecureCloud<ResolvedEstimoteLocation>,
                                                    private val payloadTimeoutValue: Long,
                                                    private val payloadTimeoutUnit: TimeUnit) : SecurePacketResolver<EstimoteLocation> {

    override fun resolveEstimoteLocation(encryptedPacketObservable: Observable<EstimoteSecure>): Observable<EstimoteLocation> {
        return encryptedPacketObservable
                .groupBy { it.payload.toHexString() }
                .map {
                    val encryptedPacketsScan = it
                            .completeWhenThereIsNoMorePacketsWithThisPayloadAround()
                            .share()

                    val resolvePacketInCloud = encryptedPacketsScan
                            .firstOrError()
                            .tryResolvingInCloud()
                            .onErrorReturn { MaybeResolvedEstimoteLocation(null) }
                            .toObservable()

                    encryptedPacketsScan
                            .withLatestFrom(resolvePacketInCloud, joinResolvedPacketWithScannedFrame())
                            .filterOnlyResolvedPackets()
                }.flatMap { it }
    }

    private fun joinResolvedPacketWithScannedFrame() = BiFunction { encrypted: EstimoteSecure, maybeResolved: MaybeResolvedEstimoteLocation ->
        with(maybeResolved) {
            when (estimoteLocation) {
                is ResolvedEstimoteLocation -> {
                    MaybeEstimoteLocation(EstimoteLocationPacket(
                            encrypted.macAddress,
                            encrypted.rssi,
                            encrypted.timestamp,
                            estimoteLocation.deviceId,
                            estimoteLocation.channel,
                            estimoteLocation.protocolVersion,
                            estimoteLocation.measuredPower))
                }
                else -> MaybeEstimoteLocation(null)
            }
        }
    }


    private fun Single<EstimoteSecure>.tryResolvingInCloud() =
            flatMap { secureCloud.resolvePacket(it).map { MaybeResolvedEstimoteLocation(it) } }

    private fun Observable<EstimoteSecure>.completeWhenThereIsNoMorePacketsWithThisPayloadAround() =
            timeout(payloadTimeoutValue, payloadTimeoutUnit, Observable.empty())

    private fun Observable<MaybeEstimoteLocation>.filterOnlyResolvedPackets() =
            filter { it.estimoteLocation != null }.map { it.estimoteLocation!! }

    private data class MaybeResolvedEstimoteLocation(val estimoteLocation: ResolvedEstimoteLocation?)
    private data class MaybeEstimoteLocation(val estimoteLocation: EstimoteLocation?)
    private data class EstimoteLocationPacket(override val macAddress: MacAddress,
                                              override val rssi: Int,
                                              override val timestamp: Long,
                                              override val deviceId: String,
                                              override val channel: Int,
                                              override val protocolVersion: Int,
                                              override val measuredPower: Int) : EstimoteLocation
}