package com.estimote.cloud_plugin.analytics

import com.estimote.cloud_plugin.analytics.model.*
import com.estimote.cloud_plugin.analytics.rest.AnalyticsCloudRestApi
import com.estimote.internal_plugins_api.analytics.proximity.ProximityAnalyticsEvent
import com.estimote.internal_plugins_api.cloud.analytics.AnalyticsCloud
import com.estimote.internal_plugins_api.scanning.Duration
import com.estimote.internal_plugins_api.scanning.EstimoteTelemetryFull
import io.reactivex.Completable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers
import java.math.BigDecimal
import java.util.concurrent.TimeUnit

/**
 * @author Estimote Inc. (contact@estimote.com)
 */
internal class EstimoteAnalyticsCloud(private val analyticsCloudRestApi: AnalyticsCloudRestApi) : AnalyticsCloud {

    override fun postProximityAnalyticsEvents(events: List<ProximityAnalyticsEvent>): Completable =
            analyticsCloudRestApi
                    .postAnalyticsEvents(CloudAnalyticsEventsBatch(events.convertToCloudModels()))
                    .executeInBackground()
                    .notifyOnMainThread()

    override fun postProximityUsageMetrics(deviceId: String): Completable =
            analyticsCloudRestApi.getZonesBounds(deviceId)
                    .executeInBackground()
                    .notifyOnMainThread()

    override fun postTelemetry(telemetryFull: EstimoteTelemetryFull, timestampMillis: Long): Completable =
            analyticsCloudRestApi
                    .postTelemetry(CloudTelemetryBatch(listOf(telemetryFull.toCloudModel(timestampMillis))))
                    .executeInBackground()
                    .notifyOnMainThread()

    private fun List<ProximityAnalyticsEvent>.convertToCloudModels() =
            map {
                CloudAnalyticsEvent(
                        it.eventType.name.toLowerCase(),
                        it.packetType.name.toLowerCase(),
                        it.packetIdentifier,
                        it.expectedDistance,
                        it.isInForeground,
                        it.distinctId,
                        it.timestampMillis.toSeconds())
            }

    private fun EstimoteTelemetryFull.toCloudModel(timestampMillis: Long) =
            CloudTelemetryModel(
                    identifier.toLowerCase(),
                    Vector(acceleration.xAxis, acceleration.yAxis, acceleration.zAxis),
                    Vector(magnetometer.xAxis, magnetometer.yAxis, magnetometer.zAxis),
                    Motion(motionState, currentMotionDuration.toMillis(), previousMotionDuration.toMillis()),
                    temperatureInCelsiusDegrees.toInt(),
                    ambientLightInLux.toInt(),
                    batteryVoltageInMilliVolts,
                    uptime.toMillis(),
                    timestampMillis,
                    Warnings(false, false)
            )

    private fun Completable.executeInBackground(): Completable =
            this.subscribeOn(Schedulers.io())

    private fun Completable.notifyOnMainThread(): Completable =
            this.observeOn(AndroidSchedulers.mainThread())

    private fun Duration.toMillis() = timeUnit.convert(value.toLong(), TimeUnit.MILLISECONDS)

    private fun Long.toSeconds(): String = BigDecimal(this / 1000.0).setScale(2, BigDecimal.ROUND_HALF_UP).toString()

}