package com.estimote.cloud_plugin.management.internal

import com.estimote.cloud_plugin.management.internal.dagger.model.MeshId
import com.estimote.cloud_plugin.management.internal.rest.ManagementCloudRestApi
import com.estimote.internal_plugins_api.cloud.management.*
import io.reactivex.Completable
import io.reactivex.Maybe
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers

internal class EstimoteManagementCloud(private val restApiManagement: ManagementCloudRestApi) : ManagementCloud {

    override fun getMeshMessageForDevice(deviceId: CloudDeviceId): Observable<CloudMeshMessage> =
            getMeshIdForDevice(deviceId)
                    .completeIfThereIsNoMeshConfigured()
                    .mapToMeshMessage()
                    .completeIfThereIsNoMessageForDevice()
                    .executeInBackground()
                    .notifyOnMainThread()

    override fun updateSettingsVersion(deviceId: String, cloudMessage: CloudMeshMessage): Observable<CloudMeshMessage> {
        return restApiManagement.updateSettingsVersion(deviceId, MeshSettingsVersionUpdateFactory().create(deviceId, cloudMessage.settingsVersion))
                .map { cloudMessage }
                .executeInBackground()
                .notifyOnMainThread()
    }


    override fun putMeshMessageBuffer(cloudMessage: CloudMeshMessage): Completable {
        return restApiManagement.putMeshMessageBuffer(cloudMessage.meshId, BufferConfirmationMessage(cloudMessage.data.toHexString()))
                .executeInBackground()
                .notifyOnMainThread()
    }

    override fun getCommandsForDevice(deviceId: CloudDeviceId): Observable<CloudCommands> {
        return restApiManagement.getCommandsForDevice(deviceId.id)
                .executeInBackground()
                .notifyOnMainThread()
    }

    override fun deleteCommands(deviceId: CloudDeviceId): Completable {
        return restApiManagement.deleteCommands(deviceId.id)
                .executeInBackground()
                .notifyOnMainThread()
    }

    override fun getMeshesList(): Observable<List<CloudMeshConfig>> {
        return restApiManagement.getMeshesList()
                .map { it.cloudMeshConfigs }
                .executeInBackground()
                .notifyOnMainThread()
    }

    override fun getMeshSettingsVersion(meshId: Long): Observable<CloudMeshSettingsVersion> {
        return restApiManagement.getMeshSettingsVersion(meshId)
                .executeInBackground()
                .notifyOnMainThread()
    }

    override fun confirmMeshSettings(confirmationRequests: List<MeshSettingsConfirmationRequest>): Observable<MeshSettingsConfirmationResponse> {
        return restApiManagement.confirmMeshSettings(confirmationRequests)
                .executeInBackground()
                .notifyOnMainThread()
    }

    private fun getMeshIdForDevice(deviceId: CloudDeviceId): Observable<MeshId> {
        val observable = restApiManagement.getDeviceMeshId(deviceId.id)
        return observable
    }

    private fun Observable<MeshId>.completeIfThereIsNoMeshConfigured(): Observable<MeshId> =
            this.flatMapMaybe { meshId -> Maybe.create<MeshId>{
                    emitter -> if(meshId.meshId == null || meshId.meshId == 0L) emitter.onComplete() else emitter.onSuccess(meshId) } }

    private fun Observable<MeshId>.mapToMeshMessage(): Observable<CloudMeshMessage> {
        return this.flatMap { meshId  -> restApiManagement.getMeshMessage(meshId.meshId!!).map { CloudMeshMessage(meshId.meshId!!, it.data, it.settingsVersion) } }
    }

    private fun Observable<CloudMeshMessage>.completeIfThereIsNoMessageForDevice(): Observable<CloudMeshMessage> =
            this.flatMapMaybe { message -> Maybe.create<CloudMeshMessage> {
                    emitter -> if (message.data.isEmpty()) emitter.onComplete() else emitter.onSuccess(message) } }

    private fun <T> Observable<T>.executeInBackground(): Observable<T> =
            this.subscribeOn(Schedulers.io())

    private fun Completable.executeInBackground(): Completable =
            this.subscribeOn(Schedulers.io())

    private fun <T> Observable<T>.notifyOnMainThread(): Observable<T> =
            this.observeOn(AndroidSchedulers.mainThread())

    private fun Completable.notifyOnMainThread():Completable =
            this.observeOn(AndroidSchedulers.mainThread())
    
    private val HEX_CHARS = "0123456789abcdef".toCharArray()
    fun ByteArray.toHexString() : String {
        var result = ""
        forEach {
            val octet = it.toInt()
            val firstIndex = (octet and 0xF0).ushr(4)
            val secondIndex = octet and 0x0F
            result += ("${HEX_CHARS[firstIndex]}${HEX_CHARS[secondIndex]}")
        }
        return result
    }
}