package com.estimote.cloud_plugin.proximity

import com.estimote.cloud_plugin.proximity.rest.ProximityCloudRestApi
import com.estimote.internal_plugins_api.cloud.proximity.ProximityAttachmentModel
import com.estimote.internal_plugins_api.cloud.proximity.ProximityCloud
import io.reactivex.Flowable
import io.reactivex.Single
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers

internal class EstimoteProximityCloud(private val proximityCloudRestApi: ProximityCloudRestApi) : ProximityCloud {


    override fun getDeviceIdToProximityAttachmentMapping(keyValuePairs: List<Pair<String, String>>) =
            Single.just(keyValuePairs.convertToRequestParam())
            .requestAttachments()
            .filterInvalidIdentifiers()
            .filterNullAttachments()
            .proximityAttachmentsToMap()
            .executeInBackground()
            .notifyOnMainThread()

    private fun List<Pair<String, String>>.convertToRequestParam() = map{"{\"" + it.first + "\":\"" +  it.second + "\"}"}

    private fun Single<List<String>>.requestAttachments() = flatMap {
        Single.concat(it.map { proximityCloudRestApi.getProximityAttachmentsForPayload(it) })
                .flatMap { Flowable.fromIterable(it.attachmentModels) }
                .distinct { it.getDeviceId() }
                .toList()
    }

    private fun Single<List<ProximityAttachmentModel>>.proximityAttachmentsToMap() = map { it.associateBy({ it.getDeviceId() }, { it }) }

    private fun Single<List<ProximityAttachmentModel>>.filterInvalidIdentifiers() = map { it.filter { it.getDeviceId().isNotEmpty() } }

    private fun Single<List<ProximityAttachmentModel>>.filterNullAttachments() = map { it.filter { it.hasAttachment() } }

    private fun <T> Single<T>.executeInBackground(): Single<T> =
            this.subscribeOn(Schedulers.io())

    private fun <T> Single<T>.notifyOnMainThread(): Single<T> =
            this.observeOn(AndroidSchedulers.mainThread())
}