package com.estimote.cloud_plugin.internal

import com.estimote.cloud_plugin.internal.dagger.model.MeshId
import com.estimote.cloud_plugin.internal.rest.CloudRestApi
import com.estimote.internal_plugins_api.cloud.CloudDeviceId
import com.estimote.internal_plugins_api.cloud.CloudManager
import com.estimote.internal_plugins_api.cloud.CloudMeshMessage
import io.reactivex.Completable
import io.reactivex.Maybe
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers

internal class EstimoteCloudManager(private val restApi: CloudRestApi) : CloudManager {

    override fun getMeshMessageForDevice(deviceId: CloudDeviceId): Observable<CloudMeshMessage> =
            restApi.getDeviceMeshId(deviceId.id)
                    .mapToMeshMessage()
                    .completeIfThereIsNoMessageForDevice()
                    .executeInBackground()
                    .notifyOnMainThread()

    override fun updateSettingsVersion(deviceId: String, settingsVersion: Long): Completable {
        return restApi.updateSettingsVersion(deviceId, MeshSettingsVersionUpdateFactory().create(deviceId, settingsVersion))
                .executeInBackground()
                .notifyOnMainThread()
    }

    private fun Observable<MeshId>.mapToMeshMessage(): Observable<CloudMeshMessage> {
        return this.flatMap { restApi.getMeshMessage(it.meshId!!) }
    }

    private fun Observable<CloudMeshMessage>.completeIfThereIsNoMessageForDevice(): Observable<CloudMeshMessage> =
            this.flatMapMaybe { message -> Maybe.create<CloudMeshMessage> {
                    emitter -> if (message.data.isEmpty()) emitter.onComplete() else emitter.onSuccess(message) } }

    private fun Observable<CloudMeshMessage>.executeInBackground(): Observable<CloudMeshMessage> =
            this.subscribeOn(Schedulers.io())

    private fun Completable.executeInBackground(): Completable =
            this.subscribeOn(Schedulers.io())

    private fun Observable<CloudMeshMessage>.notifyOnMainThread(): Observable<CloudMeshMessage> =
            this.observeOn(AndroidSchedulers.mainThread())

    private fun Completable.notifyOnMainThread():Completable =
            this.observeOn(AndroidSchedulers.mainThread())
}