/*
 * Decompiled with CFR 0.152.
 */
package com.erudika.para.server.persistence;

import com.erudika.para.core.App;
import com.erudika.para.core.ParaObject;
import com.erudika.para.core.annotations.Locked;
import com.erudika.para.core.persistence.DAO;
import com.erudika.para.core.utils.Pager;
import com.erudika.para.core.utils.Para;
import com.erudika.para.core.utils.Utils;
import com.erudika.para.server.persistence.AWSDynamoUtils;
import jakarta.inject.Singleton;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.dynamodb.DynamoDbClient;
import software.amazon.awssdk.services.dynamodb.model.AttributeValue;
import software.amazon.awssdk.services.dynamodb.model.ConditionalCheckFailedException;
import software.amazon.awssdk.services.dynamodb.model.GetItemResponse;
import software.amazon.awssdk.services.dynamodb.model.KeysAndAttributes;
import software.amazon.awssdk.services.dynamodb.model.UpdateItemRequest;
import software.amazon.awssdk.services.dynamodb.model.WriteRequest;

@Singleton
public class AWSDynamoDAO
implements DAO {
    private static final Logger logger = LoggerFactory.getLogger(AWSDynamoDAO.class);
    private static final int MAX_ITEMS_PER_WRITE = 25;
    private static final int MAX_KEYS_PER_READ = 100;

    public AWSDynamoDAO() {
        if (!AWSDynamoUtils.existsTable(Para.getConfig().getRootAppIdentifier())) {
            AWSDynamoUtils.createTable(Para.getConfig().getRootAppIdentifier());
        }
    }

    DynamoDbClient client() {
        return AWSDynamoUtils.getClient();
    }

    public <P extends ParaObject> String create(String appid, P so) {
        if (so == null) {
            return null;
        }
        if (StringUtils.isBlank((CharSequence)so.getId())) {
            so.setId(Utils.getNewId());
        }
        if (so.getTimestamp() == null) {
            so.setTimestamp(Long.valueOf(Utils.timestamp()));
        }
        so.setAppid(appid);
        this.createRow(so.getId(), appid, AWSDynamoUtils.toRow(so, null));
        logger.debug("DAO.create() {}->{}", (Object)appid, (Object)so.getId());
        return so.getId();
    }

    public <P extends ParaObject> P read(String appid, String key) {
        if (StringUtils.isBlank((CharSequence)key)) {
            return null;
        }
        Object so = AWSDynamoUtils.fromRow(this.readRow(key, appid));
        logger.debug("DAO.read() {}->{}", (Object)appid, (Object)key);
        return (P)(so != null ? so : null);
    }

    public <P extends ParaObject> void update(String appid, P so) {
        if (so != null && so.getId() != null) {
            so.setUpdated(Long.valueOf(Utils.timestamp()));
            boolean updated = this.updateRow(so.getId(), appid, AWSDynamoUtils.toRow(so, Locked.class));
            if (so.getVersion() != null && so.getVersion() > 0L) {
                so.setVersion(Long.valueOf(updated ? so.getVersion() + 1L : -1L));
            } else {
                so.setVersion(Long.valueOf(0L));
            }
            logger.debug("DAO.update() {}->{}", (Object)appid, (Object)so.getId());
        }
    }

    public <P extends ParaObject> void delete(String appid, P so) {
        if (so != null && so.getId() != null) {
            this.deleteRow(so.getId(), appid);
            logger.debug("DAO.delete() {}->{}", (Object)appid, (Object)so.getId());
        }
    }

    private String createRow(String key, String appid, Map<String, AttributeValue> row) {
        if (StringUtils.isBlank((CharSequence)key) || StringUtils.isBlank((CharSequence)appid) || row == null || row.isEmpty()) {
            return null;
        }
        String table = AWSDynamoUtils.getTableNameForAppid(appid);
        try {
            key = AWSDynamoUtils.getKeyForAppid(key, appid);
            this.setRowKey(key, row);
            this.client().putItem(b -> b.tableName(table).item(row));
        }
        catch (Exception e) {
            logger.error("Could not write row to DB - table={}, appid={}, key={}:", new Object[]{table, appid, key, e});
            AWSDynamoUtils.throwIfNecessary(e);
        }
        return key;
    }

    private boolean updateRow(String key, String appid, Map<String, AttributeValue> row) {
        if (StringUtils.isBlank((CharSequence)key) || StringUtils.isBlank((CharSequence)appid) || row == null || row.isEmpty()) {
            return false;
        }
        String table = AWSDynamoUtils.getTableNameForAppid(appid);
        try {
            UpdateItemRequest.Builder updateRequest = UpdateItemRequest.builder();
            StringBuilder updateExpression = new StringBuilder("SET ");
            HashMap<Object, String> names = new HashMap<Object, String>(row.size() + 1);
            HashMap<Object, AttributeValue> values = new HashMap<Object, AttributeValue>(row.size() + 1);
            boolean isLockingEnabledForRow = false;
            AttributeValue version = row.remove("version");
            if (version == null || version.n() == null) {
                version = (AttributeValue)AttributeValue.builder().n("0").build();
            }
            if (Long.parseLong(version.n()) > 0L) {
                isLockingEnabledForRow = true;
            }
            for (Map.Entry<String, AttributeValue> attr : row.entrySet()) {
                String name = "#" + attr.getKey();
                String value = ":" + attr.getKey();
                updateExpression.append(name).append("=").append(value).append(",");
                names.put(name, attr.getKey());
                values.put(value, attr.getValue());
            }
            updateExpression.setLength(updateExpression.length() - 1);
            if (isLockingEnabledForRow) {
                names.put("#version", "version");
                values.put(":version", version);
                values.put(":plusOne", (AttributeValue)AttributeValue.builder().n("1").build());
                updateRequest.conditionExpression("#version = :version");
                updateExpression.append(" ADD #").append("version").append(" :plusOne");
            }
            updateRequest.tableName(table);
            updateRequest.key(this.rowKey(key, appid));
            updateRequest.expressionAttributeNames(names);
            updateRequest.expressionAttributeValues(values);
            updateRequest.updateExpression(updateExpression.toString());
            this.client().updateItem((UpdateItemRequest)updateRequest.build());
            return true;
        }
        catch (ConditionalCheckFailedException ex) {
            logger.warn("Item not updated - versions don't match. table={}, appid={}, key={}.", new Object[]{table, appid, key});
        }
        catch (Exception e) {
            logger.error("Could not update row in DB - table={}, appid={}, key={}:", new Object[]{table, appid, key, e});
            AWSDynamoUtils.throwIfNecessary(e);
        }
        return false;
    }

    private Map<String, AttributeValue> readRow(String key, String appid) {
        if (StringUtils.isBlank((CharSequence)key) || StringUtils.isBlank((CharSequence)appid)) {
            return null;
        }
        Map row = null;
        String table = AWSDynamoUtils.getTableNameForAppid(appid);
        try {
            GetItemResponse res = this.client().getItem(b -> b.tableName(table).key(this.rowKey(key, appid)));
            if (res != null && res.item() != null && !res.item().isEmpty()) {
                row = res.item();
            }
        }
        catch (Exception e) {
            logger.error("Could not read row from DB - table={}, appid={}, key={}:", new Object[]{table, appid, key, e});
        }
        return row == null || row.isEmpty() ? null : row;
    }

    private void deleteRow(String key, String appid) {
        if (StringUtils.isBlank((CharSequence)key) || StringUtils.isBlank((CharSequence)appid)) {
            return;
        }
        String table = AWSDynamoUtils.getTableNameForAppid(appid);
        try {
            this.client().deleteItem(b -> b.tableName(table).key(this.rowKey(key, appid)));
        }
        catch (Exception e) {
            logger.error("Could not delete row from DB - table={}, appid={}, key={}:", new Object[]{table, appid, key, e});
            AWSDynamoUtils.throwIfNecessary(e);
        }
    }

    public <P extends ParaObject> void createAll(String appid, List<P> objects) {
        if (objects == null || objects.isEmpty() || StringUtils.isBlank((CharSequence)appid)) {
            return;
        }
        LinkedHashMap<String, WriteRequest> reqs = new LinkedHashMap<String, WriteRequest>(objects.size());
        int batchSteps = 1;
        if (objects.size() > 25) {
            batchSteps = objects.size() / 25 + (objects.size() % 25 > 0 ? 1 : 0);
        }
        Iterator<P> it = objects.iterator();
        String tableName = AWSDynamoUtils.getTableNameForAppid(appid);
        int j = 0;
        for (int i = 0; i < batchSteps; ++i) {
            while (it.hasNext() && j < 25) {
                ParaObject object = (ParaObject)it.next();
                if (StringUtils.isBlank((CharSequence)object.getId())) {
                    object.setId(Utils.getNewId());
                }
                if (object.getTimestamp() == null) {
                    object.setTimestamp(Long.valueOf(Utils.timestamp()));
                }
                object.setAppid(appid);
                Map<String, AttributeValue> row = AWSDynamoUtils.toRow(object, null);
                this.setRowKey(AWSDynamoUtils.getKeyForAppid(object.getId(), appid), row);
                reqs.put(object.getId(), (WriteRequest)WriteRequest.builder().putRequest(b -> b.item(row)).build());
                ++j;
            }
            AWSDynamoUtils.batchWrite(Collections.singletonMap(tableName, reqs.values().stream().collect(Collectors.toList())), 1);
            reqs.clear();
            j = 0;
        }
        logger.debug("DAO.createAll() {}->{}", (Object)appid, (Object)objects.size());
    }

    public <P extends ParaObject> Map<String, P> readAll(String appid, List<String> keys, boolean getAllColumns) {
        if (keys == null || keys.isEmpty() || StringUtils.isBlank((CharSequence)appid)) {
            return new LinkedHashMap();
        }
        TreeSet<String> keySet = new TreeSet<String>(keys);
        if (keySet.size() < keys.size() && !keySet.isEmpty()) {
            logger.debug("Duplicate keys found - readAll({})", keys);
        }
        LinkedHashMap results = new LinkedHashMap(keySet.size(), 0.75f, true);
        ArrayList<Map<String, AttributeValue>> keyz = new ArrayList<Map<String, AttributeValue>>(100);
        String table = AWSDynamoUtils.getTableNameForAppid(appid);
        try {
            int batchSteps = 1;
            if (keySet.size() > 100) {
                batchSteps = keySet.size() / 100 + (keySet.size() % 100 > 0 ? 1 : 0);
            }
            Iterator it = keySet.iterator();
            int j = 0;
            for (int i = 0; i < batchSteps; ++i) {
                while (it.hasNext() && j < 100) {
                    String key = (String)it.next();
                    results.put(key, null);
                    if (StringUtils.isBlank((CharSequence)key)) continue;
                    keyz.add(this.rowKey(key, appid));
                    ++j;
                }
                KeysAndAttributes.Builder kna = KeysAndAttributes.builder().keys(keyz);
                if (!getAllColumns) {
                    kna.attributesToGet(Arrays.asList("id", "key", "type"));
                }
                AWSDynamoUtils.batchGet(Collections.singletonMap(table, (KeysAndAttributes)kna.build()), results, 1);
                keyz.clear();
                j = 0;
            }
            logger.debug("DAO.readAll({}) {}", keySet, (Object)results.size());
        }
        catch (Exception e) {
            logger.error("Failed to readAll({}), table={}:", new Object[]{keys, table, e});
        }
        return results;
    }

    public <P extends ParaObject> List<P> readPage(String appid, Pager pager) {
        if (StringUtils.isBlank((CharSequence)appid)) {
            return Collections.emptyList();
        }
        if (pager == null) {
            pager = new Pager();
        }
        List<Object> results = new LinkedList();
        String table = AWSDynamoUtils.getTableNameForAppid(appid);
        try {
            results = AWSDynamoUtils.isSharedAppid(appid) ? AWSDynamoUtils.readPageFromSharedTable(appid, pager) : AWSDynamoUtils.readPageFromTable(appid, pager);
            pager.setCount(pager.getCount() + (long)results.size());
        }
        catch (Exception e) {
            logger.error("Failed to readPage({}), table={}:", new Object[]{appid, table, e});
        }
        return results;
    }

    public <P extends ParaObject> void updateAll(String appid, List<P> objects) {
        if (objects != null) {
            for (ParaObject object : objects) {
                if (object == null || object.getId() == null) continue;
                object.setUpdated(Long.valueOf(Utils.timestamp()));
                boolean updated = this.updateRow(object.getId(), appid, AWSDynamoUtils.toRow(object, Locked.class));
                if (object.getVersion() != null && object.getVersion() > 0L) {
                    object.setVersion(Long.valueOf(updated ? object.getVersion() + 1L : -1L));
                    continue;
                }
                object.setVersion(Long.valueOf(0L));
            }
        }
        logger.debug("DAO.updateAll() {}", (Object)(objects == null ? 0 : objects.size()));
    }

    public <P extends ParaObject> void deleteAll(String appid, List<P> objects) {
        if (objects == null || objects.isEmpty() || StringUtils.isBlank((CharSequence)appid)) {
            return;
        }
        LinkedHashMap<String, WriteRequest> reqs = new LinkedHashMap<String, WriteRequest>(objects.size());
        int batchSteps = 1;
        if (objects.size() > 25) {
            batchSteps = objects.size() / 25 + (objects.size() % 25 > 0 ? 1 : 0);
        }
        Iterator<P> it = objects.iterator();
        String tableName = AWSDynamoUtils.getTableNameForAppid(appid);
        int j = 0;
        for (int i = 0; i < batchSteps; ++i) {
            while (it.hasNext() && j < 25) {
                ParaObject object = (ParaObject)it.next();
                if (object == null) continue;
                reqs.put(object.getId(), (WriteRequest)WriteRequest.builder().deleteRequest(b -> b.key(this.rowKey(object.getId(), appid))).build());
                ++j;
            }
            AWSDynamoUtils.batchWrite(Collections.singletonMap(tableName, reqs.values().stream().collect(Collectors.toList())), 1);
            reqs.clear();
            j = 0;
        }
        logger.debug("DAO.deleteAll() {}", (Object)objects.size());
    }

    private void setRowKey(String key, Map<String, AttributeValue> row) {
        if (row.containsKey("key")) {
            logger.warn("Attribute name conflict:  attribute {} will be overwritten! {} is a reserved keyword.", (Object)"key");
        }
        row.put("key", (AttributeValue)AttributeValue.builder().s(key).build());
    }

    private Map<String, AttributeValue> rowKey(String key, String appid) {
        return Collections.singletonMap("key", (AttributeValue)AttributeValue.builder().s(AWSDynamoUtils.getKeyForAppid(key, appid)).build());
    }

    public <P extends ParaObject> String create(P so) {
        return this.create(Para.getConfig().getRootAppIdentifier(), so);
    }

    public <P extends ParaObject> P read(String key) {
        return this.read(Para.getConfig().getRootAppIdentifier(), key);
    }

    public <P extends ParaObject> void update(P so) {
        this.update(Para.getConfig().getRootAppIdentifier(), so);
    }

    public <P extends ParaObject> void delete(P so) {
        this.delete(Para.getConfig().getRootAppIdentifier(), so);
    }

    public <P extends ParaObject> void createAll(List<P> objects) {
        this.createAll(Para.getConfig().getRootAppIdentifier(), objects);
    }

    public <P extends ParaObject> Map<String, P> readAll(List<String> keys, boolean getAllColumns) {
        return this.readAll(Para.getConfig().getRootAppIdentifier(), keys, getAllColumns);
    }

    public <P extends ParaObject> List<P> readPage(Pager pager) {
        return this.readPage(Para.getConfig().getRootAppIdentifier(), pager);
    }

    public <P extends ParaObject> void updateAll(List<P> objects) {
        this.updateAll(Para.getConfig().getRootAppIdentifier(), objects);
    }

    public <P extends ParaObject> void deleteAll(List<P> objects) {
        this.deleteAll(Para.getConfig().getRootAppIdentifier(), objects);
    }

    static {
        App.addAppCreatedListener(app -> {
            if (app != null && !app.isSharingTable()) {
                AWSDynamoUtils.createTable(app.getAppIdentifier());
            }
        });
        App.addAppDeletedListener(app -> {
            if (app != null) {
                if (app.isSharingTable()) {
                    final String appid = app.getAppIdentifier();
                    Para.asyncExecute((Runnable)new Runnable(){

                        @Override
                        public void run() {
                            logger.info("Async deleteAllFromSharedTable({}) started.", (Object)appid);
                            AWSDynamoUtils.deleteAllFromSharedTable(appid);
                            logger.info("Finished deleteAllFromSharedTable({}).", (Object)appid);
                        }
                    });
                } else {
                    AWSDynamoUtils.deleteTable(app.getAppIdentifier());
                }
            }
        });
    }
}

