package com.enterprisemath.utils.image;

import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.imageio.ImageIO;

import org.apache.commons.lang3.builder.ToStringBuilder;

import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;

/**
 * Implementation of image animation which creates frames from the list of file images.
 *
 * @author radek.hecl
 *
 */
public class FileListImageAnimation implements ImageAnimation {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Frame width.
         */
        private Integer frameWidth;

        /**
         * Frame height.
         */
        private Integer frameHeight;

        /**
         * Frame duration in milliseconds.
         */
        private Integer frameDuration;

        /**
         * Paths to the frame files.
         */
        private List<String> framePaths = new ArrayList<String>();

        /**
         * Sets frame width.
         *
         * @param frameWidth frame width
         * @return this instance
         */
        public Builder setFrameWidth(Integer frameWidth) {
            this.frameWidth = frameWidth;
            return this;
        }

        /**
         * Sets frame height.
         *
         * @param frameHeight frame height
         * @return this instance
         */
        public Builder setFrameHeight(Integer frameHeight) {
            this.frameHeight = frameHeight;
            return this;
        }

        /**
         * Sets frame duration in milliseconds.
         *
         * @param frameDuration frame duration in milliseconds
         * @return this instance
         */
        public Builder setFrameDuration(Integer frameDuration) {
            this.frameDuration = frameDuration;
            return this;
        }

        /**
         * Sets frame image paths.
         *
         * @param framePaths frame image paths
         * @return this instance
         */
        public Builder setFramePaths(List<String> framePaths) {
            this.framePaths = DomainUtils.softCopyList(framePaths);
            return this;
        }

        /**
         * Adds pat to the frame image. This results with another frame.
         *
         * @param framePath frame image path
         * @return this instance
         */
        public Builder addFramePath(String framePath) {
            framePaths.add(framePath);
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public FileListImageAnimation build() {
            return new FileListImageAnimation(this);
        }
    }

    /**
     * Frame width.
     */
    private Integer frameWidth;

    /**
     * Frame height.
     */
    private Integer frameHeight;

    /**
     * Frame duration in milliseconds.
     */
    private Integer frameDuration;

    /**
     * Paths to the frame files.
     */
    private List<String> framePaths;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public FileListImageAnimation(Builder builder) {
        frameWidth = builder.frameWidth;
        frameHeight = builder.frameHeight;
        frameDuration = builder.frameDuration;
        framePaths = Collections.unmodifiableList(DomainUtils.softCopyList(builder.framePaths));
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardPositiveInt(frameWidth, "frameWidth must be positive");
        ValidationUtils.guardPositiveInt(frameHeight, "frameHeight must be positive");
        ValidationUtils.guardPositiveInt(frameDuration, "frameDuration must be positive");
        ValidationUtils.guardNotEmptyStringInCollection(framePaths, "framePaths cannot have empty element");
    }

    @Override
    public int getFrameWidth() {
        return frameWidth;
    }

    @Override
    public int getFrameHeight() {
        return frameHeight;
    }

    @Override
    public int getNumFrames() {
        return framePaths.size();
    }

    @Override
    public int getFrameDuration() {
        return frameDuration;
    }

    @Override
    public RenderedImage getFrame(int index) {
        try {
            BufferedImage img = ImageIO.read(new File(framePaths.get(index)));
            ValidationUtils.guardEquals(frameWidth, img.getWidth(), "image width does not match the frameWidth: " +
                    "frameWidth = " + frameWidth + "; " +
                    "imageWidth = " + img.getWidth() + "; " +
                    "path = " + framePaths.get(index));
            ValidationUtils.guardEquals(frameHeight, img.getHeight(), "image height does not match the frameHeight: " +
                    "frameHeight = " + frameHeight + "; " +
                    "imageHeight = " + img.getHeight() + "; " +
                    "path = " + framePaths.get(index));
            return img;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
