package com.enterprisemath.utils.engine;

import com.enterprisemath.utils.ValidationUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Defines task progress.
 *
 * @author radek.hecl
 */
public class TaskProgress {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Progress. Must be number between [0-1].
         */
        private Double progress;

        /**
         * Status message.
         */
        private String message = "";

        /**
         * Sets progress. Must be in interval [0, 1].
         *
         * @param progress progress, must be in interval [0, 1]
         * @return this instance
         */
        public Builder setProgress(Double progress) {
            this.progress = progress;
            return this;
        }

        /**
         * Sets message.
         *
         * @param message message
         * @return this instance
         */
        public Builder setMessage(String message) {
            this.message = message;
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public TaskProgress build() {
            return new TaskProgress(this);
        }
    }

    /**
     * Progress. Must be number between [0-1].
     */
    private Double progress;

    /**
     * Status message.
     */
    private String message;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public TaskProgress(Builder builder) {
        progress = builder.progress;
        message = builder.message;
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotNull(progress, "progress cannot be null");
        ValidationUtils.guardGreaterOrEqualDouble(progress, 0, "progress must be in interval [0, 1]");
        ValidationUtils.guardGreaterOrEqualDouble(1, progress, "progress must be in interval [0, 1]");
        ValidationUtils.guardNotNull(message, "message cannot be null");
    }

    /**
     * Returns progress. This is a number in interval [0, 1].
     *
     * @return progress as a number in [0, 1]
     */
    public Double getProgress() {
        return progress;
    }

    /**
     * Returns status message.
     *
     * @return status message
     */
    public String getMessage() {
        return message;
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.reflectionHashCode(this);
    }

    @Override
    public boolean equals(Object obj) {
        return EqualsBuilder.reflectionEquals(this, obj);
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    /**
     * Creates new instance.
     *
     * @param progress progress, interval [0, 1].
     * @return created progress
     */
    public static TaskProgress create(double progress) {
        return new TaskProgress.Builder().
                setProgress(progress).
                build();
    }

    /**
     * Creates new instance.
     *
     * @param progress progress, interval [0, 1].
     * @param message status message
     * @return created progress
     */
    public static TaskProgress create(double progress, String message) {
        return new TaskProgress.Builder().
                setProgress(progress).
                setMessage(message).
                build();
    }
}
