package com.enterprisemath.utils.engine;

import java.util.Map;

/**
 * Defines interface for task management.
 *
 * <p>
 * <strong>Note:</strong> Adding functions to this interface will <strong>NOT</strong> be
 * considered as breaking binary compatibility.
 * </p>
 *
 * @author radek.hecl
 */
public interface TaskManager {

    /**
     * Returns task identification code.
     *
     * @return task identification code
     */
    public String getCode();

    /**
     * Returns whether task is done. It doesn't matter in which way the end happened.
     * Task may end up by returning the result, throwing exception or get stopped.
     *
     * @return whether task is done or not
     */
    public boolean isDone();

    /**
     * Returns task result. Result satisfies the following conditions.
     * <ul>
     * <li>If task finished normally by returning from the run method, then result is ALWAYS not null (empty map if run method returned null)</li>
     * <li>If task is still running, then result is null</li>
     * <li>If task thrown an exception, then result is null</li>
     * <li>If task has been stopped, then result is null</li>
     * </ul>
     *
     * @return task result or null if result is not available
     */
    public Map<String, String> getResult();

    /**
     * Returns details about the exception. This method returns not null always and only if task
     * execution ended up by exception thrown from the run method. In all other cases returns null.
     *
     * @return details about the exception or null if there was no exception
     */
    public Map<String, String> getExceptionDetails();

    /**
     * Stops task execution. Once this method returns, then task is stopped and it cannot produce any result any more.
     * If task already finished or has been stopped, then this method has no effect.
     */
    public void stop();

    /**
     * Waits until task is done. This method blocks calling thread until task is done by any way.
     * Immediate call to isDone method will always return true.
     */
    public void waitTillDone();
}
