package com.enterprisemath.utils.engine;

import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;
import java.util.Date;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Defines status of the engine task.
 *
 * @author radek.hecl
 */
public class EngineTaskStatus {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Identification code.
         */
        private String code;

        /**
         * Task class.
         */
        private String taskClass;

        /**
         * Timestamp when task started.
         */
        private Date startTimestamp;

        /**
         * Progress.
         */
        private TaskProgress progress;

        /**
         * Sets identification code.
         *
         * @param code identification code
         * @return this instance
         */
        public Builder setCode(String code) {
            this.code = code;
            return this;
        }

        /**
         * Sets task class.
         *
         * @param taskClass task class
         * @return this instance
         */
        public Builder setTaskClass(String taskClass) {
            this.taskClass = taskClass;
            return this;
        }

        /**
         * Sets timestamp when task started.
         *
         * @param startTimestamp timestamp when task started
         * @return this instance
         */
        public Builder setStartTimestamp(Date startTimestamp) {
            this.startTimestamp = DomainUtils.copyDate(startTimestamp);
            return this;
        }

        /**
         * Sets progress.
         *
         * @param progress progress
         * @return this instance
         */
        public Builder setProgress(TaskProgress progress) {
            this.progress = progress;
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public EngineTaskStatus build() {
            return new EngineTaskStatus(this);
        }
    }

    /**
     * Identification code.
     */
    private String code;

    /**
     * Task class.
     */
    private String taskClass;

    /**
     * Timestamp when task started.
     */
    private Date startTimestamp;

    /**
     * Progress.
     */
    private TaskProgress progress;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public EngineTaskStatus(Builder builder) {
        code = builder.code;
        taskClass = builder.taskClass;
        startTimestamp = DomainUtils.copyDate(builder.startTimestamp);
        progress = builder.progress;
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotEmpty(code, "code cannot be empty");
        ValidationUtils.guardNotEmpty(taskClass, "taskClass cannot be empty");
        ValidationUtils.guardNotNull(startTimestamp, "startTimestamp cannot be null");
        ValidationUtils.guardNotNull(progress, "progress cannot be null");
    }

    /**
     * Returns identification code.
     *
     * @return identification code
     */
    public String getCode() {
        return code;
    }

    /**
     * Returns task class.
     *
     * @return task class
     */
    public String getTaskClass() {
        return taskClass;
    }

    /**
     * Returns timestamp when task started.
     *
     * @return timestamp when task started
     */
    public Date getStartTimestamp() {
        return DomainUtils.copyDate(startTimestamp);
    }

    /**
     * Returns progress.
     *
     * @return progress
     */
    public TaskProgress getProgress() {
        return progress;
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.reflectionHashCode(this);
    }

    @Override
    public boolean equals(Object obj) {
        return EqualsBuilder.reflectionEquals(this, obj);
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
