"use strict";
/**
 * Built-in authentication functions.
 *
 * @example
 * var authLib = require('/lib/xp/auth');
 *
 * @module auth
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.modifyRole = exports.createRole = exports.findUsers = exports.modifyProfile = exports.getProfile = exports.getIdProviderConfig = exports.deletePrincipal = exports.findPrincipals = exports.removeMembers = exports.addMembers = exports.modifyGroup = exports.createGroup = exports.modifyUser = exports.createUser = exports.getMembers = exports.getMemberships = exports.getPrincipal = exports.changePassword = exports.generatePassword = exports.hasRole = exports.getUser = exports.logout = exports.login = void 0;
function checkRequired(obj, name) {
    if (obj == null || obj[name] == null) {
        throw "Parameter '".concat(String(name), "' is required");
    }
}
function checkRequiredValue(value, name) {
    if (typeof value === 'undefined' || value === null) {
        throw "Parameter '".concat(String(name), "' is required");
    }
}
var isLoginParams = function (params) {
    return typeof params === 'object' &&
        params !== null &&
        'user' in params &&
        ('password' in params || 'skipAuth' in params);
};
/**
 * Login a user with the specified idProvider, userName, password and scope.
 *
 * @example-ref examples/auth/login.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.user Name of user to log in.
 * @param {string} [params.idProvider] Name of id provider where the user is stored. If not specified it will try all available id providers, in alphabetical order.
 * @param {string} [params.password] Password for the user. Ignored if skipAuth is set to true, mandatory otherwise.
 * @param {('SESSION'|'REQUEST'|'NONE')} [params.scope=SESSION] The scope of this login. Two values are valid. SESSION logs the user in and creates a session in XP for use in future requests. REQUEST logs the user in but only for this particular request and thus does not create a session.
 * @param {boolean} [params.skipAuth=false] Skip authentication.
 * @param {number} [params.sessionTimeout] Session timeout (in seconds). By default, the value of session.timeout from com.enonic.xp.web.jetty.cfg
 * @returns {LoginResult} Information for logged-in user.
 */
function login(params) {
    var _a;
    checkRequired(params, 'user');
    var _b = params !== null && params !== void 0 ? params : {}, user = _b.user, idProvider = _b.idProvider, _c = _b.scope, scope = _c === void 0 ? 'SESSION' : _c, sessionTimeout = _b.sessionTimeout;
    var bean = __.newBean('com.enonic.xp.lib.auth.LoginHandler');
    bean.setUser(user);
    bean.setScope(scope);
    bean.setSessionTimeout(__.nullOrValue(sessionTimeout));
    if (isLoginParams(params)) {
        var skipAuth = (_a = params.skipAuth) !== null && _a !== void 0 ? _a : false;
        var password = params.password;
        if (skipAuth) {
            bean.setSkipAuth(true);
        }
        else {
            checkRequiredValue(password, 'password');
            bean.setPassword(password);
        }
    }
    if (idProvider) {
        bean.setIdProvider([].concat(idProvider));
    }
    return __.toNativeObject(bean.login());
}
exports.login = login;
/**
 * Logout an already logged-in user.
 *
 * @example-ref examples/auth/logout.js
 */
function logout() {
    var bean = __.newBean('com.enonic.xp.lib.auth.LogoutHandler');
    bean.logout();
}
exports.logout = logout;
/**
 * Returns the logged-in user. If not logged-in, this will return *undefined*.
 *
 * @example-ref examples/auth/getUser.js
 *
 * @param {object} [params] JSON parameters.
 * @param {boolean} [params.includeProfile=false] Include profile.
 *
 * @returns {User} Information for logged-in user.
 */
function getUser(params) {
    var _a = (params !== null && params !== void 0 ? params : {}).includeProfile, includeProfile = _a === void 0 ? false : _a;
    var bean = __.newBean('com.enonic.xp.lib.auth.GetUserHandler');
    bean.setIncludeProfile(includeProfile);
    return __.toNativeObject(bean.getUser());
}
exports.getUser = getUser;
/**
 * Checks if the logged-in user has the specified role.
 *
 * @example-ref examples/auth/hasRole.js
 *
 * @param {string} role Role name to check for.
 * @returns {boolean} True if the user has specified role, false otherwise.
 */
function hasRole(role) {
    var bean = __.newBean('com.enonic.xp.lib.auth.HasRoleHandler');
    bean.setRole(__.nullOrValue(role));
    return bean.hasRole();
}
exports.hasRole = hasRole;
/**
 * Generates a secure password.
 *
 * @example-ref examples/auth/generatePassword.js
 *
 * @returns {string} A secure generated password.
 */
function generatePassword() {
    var bean = __.newBean('com.enonic.xp.lib.auth.GeneratePasswordHandler');
    return __.toNativeObject(bean.generatePassword());
}
exports.generatePassword = generatePassword;
/**
 * Changes or clears password for specified user.
 *
 * @example-ref examples/auth/changePassword.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.userKey Key for user to change password.
 * @param {string} [params.password] The new password to set. If the value is null, the password will be cleared.
 */
function changePassword(params) {
    var bean = __.newBean('com.enonic.xp.lib.auth.ChangePasswordHandler');
    checkRequired(params, 'userKey');
    var _a = params !== null && params !== void 0 ? params : {}, userKey = _a.userKey, password = _a.password;
    bean.setUserKey(userKey);
    bean.setPassword(__.nullOrValue(password));
    bean.changePassword();
}
exports.changePassword = changePassword;
/**
 * Returns the principal with the specified key.
 *
 * @example-ref examples/auth/getPrincipal.js
 *
 * @param {string} principalKey Principal key to look for.
 * @returns {User | Group | Role} the principal specified, or null if it doesn't exist.
 */
function getPrincipal(principalKey) {
    checkRequiredValue(principalKey, 'principalKey');
    var bean = __.newBean('com.enonic.xp.lib.auth.GetPrincipalHandler');
    bean.setPrincipalKey(principalKey);
    return __.toNativeObject(bean.getPrincipal());
}
exports.getPrincipal = getPrincipal;
/**
 * Returns a list of principals the specified principal is a member of.
 *
 * @example-ref examples/auth/getMemberships.js
 *
 * @param {string} principalKey Principal key of user or group to retrieve memberships for.
 * @param {boolean} [transitive=false] Retrieve transitive memberships.
 * @returns {Array<Group | Role>} Returns the list of groups and roles.
 */
function getMemberships(principalKey, transitive) {
    if (transitive === void 0) { transitive = false; }
    checkRequiredValue(principalKey, 'principalKey');
    var bean = __.newBean('com.enonic.xp.lib.auth.GetMembershipsHandler');
    bean.setPrincipalKey(principalKey);
    bean.setTransitive(transitive);
    return __.toNativeObject(bean.getMemberships());
}
exports.getMemberships = getMemberships;
/**
 * Returns a list of principals that are members of the specified principal.
 *
 * @example-ref examples/auth/getMembers.js
 *
 * @param {string} principalKey Principal key to retrieve members for.
 * @returns {Array<User | Group>} Returns the list of users and groups.
 */
function getMembers(principalKey) {
    checkRequiredValue(principalKey, 'principalKey');
    var bean = __.newBean('com.enonic.xp.lib.auth.GetMembersHandler');
    bean.setPrincipalKey(principalKey);
    return __.toNativeObject(bean.getMembers());
}
exports.getMembers = getMembers;
/**
 * Creates user from passed parameters.
 *
 * @example-ref examples/auth/createUser.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.idProvider Key for id provider where user has to be created.
 * @param {string} params.name User login name to set.
 * @param {string} [params.displayName] User display name.
 * @param {string} [params.email] User email.
 */
function createUser(params) {
    checkRequired(params, 'name');
    checkRequired(params, 'idProvider');
    var bean = __.newBean('com.enonic.xp.lib.auth.CreateUserHandler');
    bean.setIdProvider(params.idProvider);
    bean.setName(params.name);
    bean.setDisplayName(__.nullOrValue(params.displayName));
    bean.setEmail(__.nullOrValue(params.email));
    return __.toNativeObject(bean.createUser());
}
exports.createUser = createUser;
/**
 * Retrieves the user specified and updates it with the changes applied.
 *
 * @example-ref examples/auth/modifyUser.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.key Principal key of the user to modify.
 * @param {function} params.editor User editor function to apply to user.
 * @returns {User} the updated user or null if a  user not found.
 */
function modifyUser(params) {
    checkRequired(params, 'key');
    checkRequired(params, 'editor');
    var bean = __.newBean('com.enonic.xp.lib.auth.ModifyUserHandler');
    bean.setPrincipalKey(params.key);
    bean.setEditor(__.toScriptValue(params.editor));
    return __.toNativeObject(bean.modifyUser());
}
exports.modifyUser = modifyUser;
/**
 * Creates a group.
 *
 * @example-ref examples/auth/createGroup.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.idProvider Key for id provider where group has to be created.
 * @param {string} params.name Group name.
 * @param {string} [params.displayName] Group display name.
 * @param {string} [params.description] as principal description .
 */
function createGroup(params) {
    checkRequired(params, 'idProvider');
    checkRequired(params, 'name');
    var bean = __.newBean('com.enonic.xp.lib.auth.CreateGroupHandler');
    bean.setIdProvider(params.idProvider);
    bean.setName(params.name);
    bean.setDisplayName(__.nullOrValue(params.displayName));
    bean.setDescription(__.nullOrValue(params.description));
    return __.toNativeObject(bean.createGroup());
}
exports.createGroup = createGroup;
/**
 * Retrieves the group specified and updates it with the changes applied.
 *
 * @example-ref examples/auth/modifyGroup.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.key Principal key of the group to modify.
 * @param {function} params.editor Group editor function to apply to group.
 * @returns {Group} the updated group or null if a group not found.
 */
function modifyGroup(params) {
    checkRequired(params, 'key');
    checkRequired(params, 'editor');
    var bean = __.newBean('com.enonic.xp.lib.auth.ModifyGroupHandler');
    bean.setPrincipalKey(params.key);
    bean.setEditor(__.toScriptValue(params.editor));
    return __.toNativeObject(bean.modifyGroup());
}
exports.modifyGroup = modifyGroup;
/**
 * Adds members to a principal (user or role).
 *
 * @example-ref examples/auth/addMembers.js
 *
 * @param {string} principalKey Key of the principal to add members to.
 * @param {Array<string>} members Keys of users and groups to add.
 */
function addMembers(principalKey, members) {
    checkRequiredValue(principalKey, 'principalKey');
    var bean = __.newBean('com.enonic.xp.lib.auth.AddMembersHandler');
    bean.setPrincipalKey(principalKey);
    bean.setMembers([].concat(members));
    bean.addMembers();
}
exports.addMembers = addMembers;
/**
 * Removes members from a principal (user or role).
 *
 * @example-ref examples/auth/removeMembers.js
 *
 * @param {string} principalKey Key of the principal to remove members from.
 * @param {string} members Keys of the principals to remove.
 */
function removeMembers(principalKey, members) {
    checkRequiredValue(principalKey, 'principalKey');
    var bean = __.newBean('com.enonic.xp.lib.auth.RemoveMembersHandler');
    bean.setPrincipalKey(principalKey);
    bean.setMembers([].concat(members));
    bean.removeMembers();
}
exports.removeMembers = removeMembers;
/**
 * Search for principals matching the specified criteria.
 *
 * @example-ref examples/auth/findPrincipals.js
 *
 * @param {object} params JSON parameters.
 * @param {string} [params.type] Principal type to look for, one of: 'user', 'group' or 'role'. If not specified all principal types will be included.
 * @param {string} [params.idProvider] Key of the id provider to look for. If not specified all id providers will be included.
 * @param {number} [params.start=0] First principal to return from the search results. It can be used for pagination.
 * @param {number} [params.count=10] A limit on the number of principals to be returned.
 * @param {string} [params.name] Name of the principal to look for.
 * @param {string} [params.searchText] Text to look for in any principal field.
 * @returns {FindPrincipalsResult} The "total" number of principals matching the search, the "count" of principals included, and an array of "hits" containing the principals.
 */
function findPrincipals(params) {
    var bean = __.newBean('com.enonic.xp.lib.auth.FindPrincipalsHandler');
    var _a = params !== null && params !== void 0 ? params : {}, type = _a.type, idProvider = _a.idProvider, _b = _a.start, start = _b === void 0 ? 0 : _b, _c = _a.count, count = _c === void 0 ? 10 : _c, name = _a.name, searchText = _a.searchText;
    bean.setType(__.nullOrValue(type));
    bean.setIdProvider(__.nullOrValue(idProvider));
    bean.setStart(__.nullOrValue(start));
    bean.setCount(__.nullOrValue(count));
    bean.setName(__.nullOrValue(name));
    bean.setSearchText(__.nullOrValue(searchText));
    return __.toNativeObject(bean.findPrincipals());
}
exports.findPrincipals = findPrincipals;
/**
 * Deletes the principal with the specified key.
 *
 * @example-ref examples/auth/deletePrincipal.js
 *
 * @param {string} principalKey Principal key to delete.
 * @returns {boolean} True if deleted, false otherwise.
 */
function deletePrincipal(principalKey) {
    checkRequiredValue(principalKey, 'principalKey');
    var bean = __.newBean('com.enonic.xp.lib.auth.DeletePrincipalHandler');
    bean.setPrincipalKey(principalKey);
    return __.toNativeObject(bean.deletePrincipal());
}
exports.deletePrincipal = deletePrincipal;
/**
 * This function returns the ID provider configuration.
 * It is meant to be called from an ID provider controller.
 *
 * @example-ref examples/auth/getIdProviderConfig.js
 *
 * @returns {object} The ID provider configuration as JSON.
 */
function getIdProviderConfig() {
    var bean = __.newBean('com.enonic.xp.lib.auth.GetIdProviderConfigHandler');
    return __.toNativeObject(bean.execute());
}
exports.getIdProviderConfig = getIdProviderConfig;
/**
 * This function retrieves the profile of a user.
 *
 * @example-ref examples/auth/getProfile.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.key Principal key of the user.
 * @param {string} [params.scope] Scope of the data to retrieve.
 * @returns {object} The extra data as JSON
 */
function getProfile(params) {
    checkRequired(params, 'key');
    var bean = __.newBean('com.enonic.xp.lib.auth.GetProfileHandler');
    bean.setKey(params.key);
    bean.setScope(__.nullOrValue(params.scope));
    return __.toNativeObject(bean.execute());
}
exports.getProfile = getProfile;
/**
 * This function retrieves the profile of a user and updates it.
 *
 * @example-ref examples/auth/modifyProfile.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.key Principal key of the user.
 * @param {string} [params.scope] Scope of the data to retrieve and update.
 * @param {function} params.editor Profile editor function to apply.
 * @returns {object} The extra data as JSON
 */
function modifyProfile(params) {
    checkRequired(params, 'key');
    checkRequired(params, 'editor');
    var bean = __.newBean('com.enonic.xp.lib.auth.ModifyProfileHandler');
    bean.setKey(params.key);
    bean.setScope(__.nullOrValue(params.scope));
    bean.setEditor(__.toScriptValue(params.editor));
    return __.toNativeObject(bean.execute());
}
exports.modifyProfile = modifyProfile;
/**
 * Search for users matching the specified query.
 *
 * @example-ref examples/auth/findUsers.js
 *
 * @param {object} params JSON with the parameters.
 * @param {number} [params.start=0] Start index (used for paging).
 * @param {number} [params.count=10] Number of contents to fetch.
 * @param {string} params.query Query expression.
 * @param {string} [params.sort] Sorting expression.
 * @param {boolean} [params.includeProfile=false] Include profile.
 *
 * @returns {FindPrincipalsResult} Result of query.
 */
function findUsers(params) {
    var _a = params !== null && params !== void 0 ? params : {}, _b = _a.start, start = _b === void 0 ? 0 : _b, _c = _a.count, count = _c === void 0 ? 10 : _c, query = _a.query, sort = _a.sort, _d = _a.includeProfile, includeProfile = _d === void 0 ? false : _d;
    var bean = __.newBean('com.enonic.xp.lib.auth.FindUsersHandler');
    bean.setStart(start);
    bean.setCount(count);
    bean.setQuery(__.nullOrValue(query));
    bean.setSort(__.nullOrValue(sort));
    bean.setIncludeProfile(includeProfile);
    return __.toNativeObject(bean.execute());
}
exports.findUsers = findUsers;
/**
 * Creates a role.
 *
 * @example-ref examples/auth/createRole.js
 *
 * @param {string} params.name Role name.
 * @param {string} [params.displayName] Role display name.
 * @param {string} [params.description] as principal description .
 */
function createRole(params) {
    checkRequired(params, 'name');
    var bean = __.newBean('com.enonic.xp.lib.auth.CreateRoleHandler');
    bean.setName(params.name);
    bean.setDisplayName(__.nullOrValue(params.displayName));
    bean.setDescription(__.nullOrValue(params.description));
    return __.toNativeObject(bean.createRole());
}
exports.createRole = createRole;
/**
 * Retrieves the role specified and updates it with the changes applied.
 *
 * @example-ref examples/auth/modifyRole.js
 *
 * @param {object} params JSON parameters.
 * @param {string} params.key Principal key of the role to modify.
 * @param {function} params.editor Role editor function to apply to role.
 * @returns {Role} the updated role or null if a role not found.
 */
function modifyRole(params) {
    checkRequired(params, 'key');
    checkRequired(params, 'editor');
    var bean = __.newBean('com.enonic.xp.lib.auth.ModifyRoleHandler');
    bean.setPrincipalKey(params.key);
    bean.setEditor(__.toScriptValue(params.editor));
    return __.toNativeObject(bean.modifyRole());
}
exports.modifyRole = modifyRole;
