package com.easefun.polyvsdk.sub.vlms.main;

import android.content.Context;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.easefun.polyvsdk.sub.auxilliary.CheckThrows;
import com.easefun.polyvsdk.sub.auxilliary.RetrofitHelper;
import com.easefun.polyvsdk.sub.auxilliary.cache.auxiliary.Md5Util;
import com.easefun.polyvsdk.sub.vlms.listener.PolyvVlmsApiListener;

import java.util.HashMap;
import java.util.Map;

import okhttp3.ResponseBody;
import retrofit2.Call;

/**
 * 网校接口管理类
 */
public class PolyvVlmsManager {

    public PolyvVlmsManager(Context context) {
        RetrofitHelper.init(context);
    }

    private String generateAccessTokenSign(String api_id, String school_id, long timestamp, String key) {
        return generateAccessTokenSign(api_id, null, school_id, timestamp, key);
    }

    private String generateAccessTokenSign(String api_id, String refresh_token, String school_id, long timestamp, String key) {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("api_id=").append(api_id);
        if (!TextUtils.isEmpty(refresh_token))
            stringBuilder.append("&refresh_token=").append(refresh_token);
        stringBuilder.append("&school_id=").append(school_id).append("&timestamp=")
                .append(timestamp).append("&key=").append(key);
        return Md5Util.getMd5(stringBuilder.toString()).toUpperCase();
    }

    /**
     * 获取accessToken，有效期为两个小时
     *
     * @param api_id 网校的学校api_id，获取方式参考{@link PolyvVlmsTestData}
     * @param school_id 网校的学校id，获取方式参考{@link PolyvVlmsTestData}
     * @param key 网校的key，获取方式参考{@link PolyvVlmsTestData}
     * @param listener 监听器
     * @return 请求对象，可能为null
     */
    public @Nullable Call<ResponseBody> getAccessToken(String api_id, String school_id, String key, PolyvVlmsApiListener.GetAccessTokenListener listener) {
        try {
            return getAccessToken_t(api_id, school_id, key, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> getAccessToken_t(String api_id, String school_id, String key, PolyvVlmsApiListener.GetAccessTokenListener listener) throws Exception {
        CheckThrows.checkNotNull("api_id", api_id);
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("key", key);
        long timestamp = System.currentTimeMillis();
        Map<String, Object> map = new HashMap<>();
        map.put("api_id", api_id);
        map.put("school_id", school_id);
        map.put("timestamp", timestamp);
        map.put("sign", generateAccessTokenSign(api_id, school_id, timestamp, key));
        Call<ResponseBody> getAccessTokenCall = RetrofitHelper.getPolyvVlmsApi()
                .getAccessToken(map);
        getAccessTokenCall.enqueue(new PolyvVlmsApiListener.GetAccessTokenCallback(listener));
        return getAccessTokenCall;
    }

    /**
     * 刷新accessToken，有效期暂定7天
     *
     * @param api_id
     * @param refresh_token
     * @param school_id
     * @param key
     * @param listener
     * @return
     */
    public Call<ResponseBody> refreshAccessToken(String api_id, String refresh_token, String school_id, String key, PolyvVlmsApiListener.GetAccessTokenListener listener) {
        try {
            return refreshAccessToken_t(api_id, refresh_token, school_id, key, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> refreshAccessToken_t(String api_id, String refresh_token, String school_id, String key, PolyvVlmsApiListener.GetAccessTokenListener listener) throws Exception {
        CheckThrows.checkNotNull("api_id", api_id);
        CheckThrows.checkNotNull("refresh_token", refresh_token);
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("key", key);
        long timestamp = System.currentTimeMillis();
        Map<String, Object> map = new HashMap<>();
        map.put("api_id", api_id);
        map.put("refresh_token", refresh_token);
        map.put("school_id", school_id);
        map.put("timestamp", timestamp);
        map.put("sign", generateAccessTokenSign(api_id, refresh_token, school_id, timestamp, key));
        Call<ResponseBody> refreshAccessTokenCall = RetrofitHelper.getPolyvVlmsApi()
                .refreshAccessToken(map);
        refreshAccessTokenCall.enqueue(new PolyvVlmsApiListener.GetAccessTokenCallback(listener));
        return refreshAccessTokenCall;
    }

    public Call<ResponseBody> getCourses(String school_id, int page, int page_size, String access_token, PolyvVlmsApiListener.GetCoursesListener listener) {
        return getCourses(school_id, null, null, null, page, page_size, access_token, listener);
    }

    /**
     * 获取课程
     *
     * @param school_id
     * @param category_id
     * @param is_free      是否免费课程：Y(是)、N(否)
     * @param type         课程类型：RECOMMEND(推荐课程)、POPULAR(热门课程)、DEFAULT(默认课程，按最新创建时间降序排列)
     * @param page
     * @param page_size
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> getCourses(String school_id, String category_id, String is_free, String type, int page, int page_size, String access_token, PolyvVlmsApiListener.GetCoursesListener listener) {
        try {
            return getCourses_t(school_id, category_id, is_free, type, page, page_size, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> getCourses_t(String school_id, String category_id, String is_free, String type, int page, int page_size, String access_token, PolyvVlmsApiListener.GetCoursesListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkRange_lessequal0("page", page);
        CheckThrows.checkRange_lessequal0("page_size", page_size);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String, Object> map = new HashMap<>();
        map.put("school_id", school_id);
        if (category_id != null)
            map.put("category_id", category_id);
        if (is_free != null)
            map.put("is_free", is_free);
        if (type != null)
            map.put("type", type);
        map.put("page", page);
        map.put("page_size", page_size);
        map.put("access_token", access_token);
        Call<ResponseBody> getCoursesCall = RetrofitHelper.getPolyvVlmsApi()
                .getCourses(map);
        getCoursesCall.enqueue(new PolyvVlmsApiListener.GetCoursesCallback(listener));
        return null;
    }

    public Call<ResponseBody> getCourseDetail(String school_id, String courseId, String access_token, PolyvVlmsApiListener.GetCourseDetailListener listener) {
        return getCourseDetail(school_id, courseId, null, access_token, listener);
    }

    /**
     * 获取课程详情
     *
     * @param school_id
     * @param courseId
     * @param is_detail
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> getCourseDetail(String school_id, String courseId, String is_detail, String access_token, PolyvVlmsApiListener.GetCourseDetailListener listener) {
        try {
            return getCourseDetail_t(school_id, courseId, is_detail, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> getCourseDetail_t(String school_id, String courseId, String is_detail, String access_token, PolyvVlmsApiListener.GetCourseDetailListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("courseId", courseId);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String, Object> map = new HashMap<>();
        map.put("school_id", school_id);
        if (is_detail != null)
            map.put("is_detail", is_detail);
        map.put("access_token", access_token);
        Call<ResponseBody> getCourseDetailCall = RetrofitHelper.getPolyvVlmsApi()
                .getCourseDetail(courseId, map);
        getCourseDetailCall.enqueue(new PolyvVlmsApiListener.GetCourseDetailCallback(listener));
        return getCourseDetailCall;
    }

    /**
     * 登录
     *
     * @param school_id
     * @param account
     * @param password
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> login(String school_id, String account, String password, String access_token, PolyvVlmsApiListener.LoginListener listener) {
        try {
            return login_t(school_id, account, password, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> login_t(String school_id, String account, String password, String access_token, PolyvVlmsApiListener.LoginListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("account", account);
        CheckThrows.checkNotNull("password", password);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String, Object> map = new HashMap<>();
        map.put("school_id", school_id);
        map.put("account", account);
        map.put("password", password);
        map.put("access_token", access_token);
        Call<ResponseBody> loginCall = RetrofitHelper.getPolyvVlmsApi()
                .login(map);
        loginCall.enqueue(new PolyvVlmsApiListener.LoginCallback(listener));
        return loginCall;
    }

    public Call<ResponseBody> getCurriculum(String school_id, String course_id, String access_token, PolyvVlmsApiListener.GetCurriculumListener listener) {
        return getCurriculum(school_id, course_id, null, access_token, listener);
    }

    /**
     * 获取课程的课时
     *
     * @param school_id
     * @param course_id
     * @param user_id 需要登录才能获取
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> getCurriculum(String school_id, String course_id, String user_id, String access_token, PolyvVlmsApiListener.GetCurriculumListener listener) {
        try {
            return getCurriculum_t(school_id, course_id, user_id, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> getCurriculum_t(String school_id, String course_id, String user_id, String access_token, PolyvVlmsApiListener.GetCurriculumListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("course_id", course_id);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String, Object> map = new HashMap<>();
        map.put("school_id", school_id);
        map.put("course_id", course_id);
        if (user_id != null)
            map.put("user_id", user_id);
        map.put("access_token", access_token);
        Call<ResponseBody> getCurriculumCall = RetrofitHelper.getPolyvVlmsApi()
                .getCurriculum(map);
        getCurriculumCall.enqueue(new PolyvVlmsApiListener.GetCurriculumCallback(listener));
        return getCurriculumCall;
    }

    public Call<ResponseBody> addOrder(String school_id, String user_id, String course_id, String payment_type, String access_token, PolyvVlmsApiListener.AddOrderListener listener) {
        return addOrder(school_id, user_id, course_id, payment_type, null, access_token, listener);
    }

    /**
     * 添加订单(前提条件：已登录(未登录会返回access_token错误))
     *
     * @param school_id
     * @param user_id
     * @param course_id
     * @param payment_type WEIXIN(微信支付)、BALANCE(余额支付)、ALIPAY(支付宝)、FREE(免费课程)
     * @param trading_note
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> addOrder(String school_id, String user_id, String course_id, String payment_type, String trading_note, String access_token, PolyvVlmsApiListener.AddOrderListener listener) {
        try {
            return addOrder_t(school_id, user_id, course_id, payment_type, trading_note, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> addOrder_t(String school_id, String user_id, String course_id, String payment_type, String trading_note, String access_token, PolyvVlmsApiListener.AddOrderListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("user_id", user_id);
        CheckThrows.checkNotNull("course_id", course_id);
        CheckThrows.checkNotNull("payment_type", payment_type);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String, Object> map = new HashMap<>();
        map.put("school_id", school_id);
        map.put("user_id", user_id);
        map.put("course_id", course_id);
        map.put("payment_type", payment_type);
        if (trading_note != null)
            map.put("trading_note", trading_note);
        map.put("access_token", access_token);
        Call<ResponseBody> addOrderCall = RetrofitHelper.getPolyvVlmsApi()
                .addOrder(map);
        addOrderCall.enqueue(new PolyvVlmsApiListener.AddOrderCallback(listener));
        return addOrderCall;
    }

    /**
     * 获取问题
     *
     * @param school_id
     * @param course_id
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> getQuestion(String school_id, String course_id, String access_token, PolyvVlmsApiListener.GetQuestionListener listener) {
        try {
            return getQuestion_t(school_id, course_id, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> getQuestion_t(String school_id, String course_id, String access_token, PolyvVlmsApiListener.GetQuestionListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("course_id", course_id);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String, Object> map = new HashMap<>();
        map.put("school_id", school_id);
        map.put("course_id", course_id);
        map.put("access_token", access_token);
        Call<ResponseBody> getQuestionCall = RetrofitHelper.getPolyvVlmsApi()
                .getQuestion(map);
        getQuestionCall.enqueue(new PolyvVlmsApiListener.GetQuestionCallback(listener));
        return getQuestionCall;
    }

    /**
     * 获取回复
     *
     * @param school_id
     * @param question_id
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> getAnswer(String school_id, String question_id, String access_token, PolyvVlmsApiListener.GetAnswerListener listener) {
        try {
            return getAnswer_t(school_id, question_id, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> getAnswer_t(String school_id, String question_id, String access_token, PolyvVlmsApiListener.GetAnswerListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("question_id", question_id);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String, Object> map = new HashMap<>();
        map.put("school_id", school_id);
        map.put("question_id", question_id);
        map.put("access_token", access_token);
        Call<ResponseBody> getAnswerCall = RetrofitHelper.getPolyvVlmsApi()
                .getAnswer(map);
        getAnswerCall.enqueue(new PolyvVlmsApiListener.GetAnswerCallback(listener));
        return getAnswerCall;
    }

    /**
     * 添加新问题(前提条件：已登录(未登录会返回access_token错误)，已购买(未购买会返回添加失败))
     *
     * @param school_id
     * @param course_id
     * @param user_id
     * @param title
     * @param content
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> addNewQuestion(String school_id, String course_id, String user_id, String title, String content, String access_token, PolyvVlmsApiListener.AddNewQuestionListener listener) {
        try {
            return addNewQuestion_t(school_id, course_id, user_id, title, content, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> addNewQuestion_t(String school_id, String course_id, String user_id, String title, String content, String access_token, PolyvVlmsApiListener.AddNewQuestionListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("course_id", course_id);
        CheckThrows.checkNotNull("user_id", user_id);
        CheckThrows.checkNotNull("title", title);
        CheckThrows.checkNotNull("content", content);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String, Object> map = new HashMap<>();
        map.put("school_id", school_id);
        map.put("course_id", course_id);
        map.put("user_id", user_id);
        map.put("title", title);
        map.put("content", content);
        map.put("access_token", access_token);
        Call<ResponseBody> addNewQuestionCall = RetrofitHelper.getPolyvVlmsApi()
                .addNewQuestion(map);
        addNewQuestionCall.enqueue(new PolyvVlmsApiListener.AddNewQuestionCallback(listener));
        return addNewQuestionCall;
    }

    /**
     * 添加新回复(前提条件：已登录(未登录会返回access_token错误)，已购买(未购买会返回添加失败))
     * @param school_id
     * @param course_id
     * @param question_id
     * @param user_id
     * @param content
     * @param access_token
     * @param listener
     * @return
     */
    public Call<ResponseBody> addNewAnswer(String school_id, String course_id, String question_id, String user_id, String content, String access_token, PolyvVlmsApiListener.AddNewAnswerListener listener) {
        try {
            return addNewAnswer_t(school_id, course_id, question_id, user_id, content, access_token, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> addNewAnswer_t(String school_id, String course_id, String question_id, String user_id, String content, String access_token, PolyvVlmsApiListener.AddNewAnswerListener listener) throws Exception {
        CheckThrows.checkNotNull("school_id", school_id);
        CheckThrows.checkNotNull("course_id", course_id);
        CheckThrows.checkNotNull("question_id", question_id);
        CheckThrows.checkNotNull("user_id", user_id);
        CheckThrows.checkNotNull("content", content);
        CheckThrows.checkNotNull("access_token", access_token);
        Map<String,Object> map = new HashMap<>();
        map.put("school_id", school_id);
        map.put("course_id", course_id);
        map.put("question_id", question_id);
        map.put("user_id", user_id);
        map.put("content", content);
        map.put("access_token", access_token);
        Call<ResponseBody> addNewAnswerCall = RetrofitHelper.getPolyvVlmsApi()
                .addNewAnswer(map);
        addNewAnswerCall.enqueue(new PolyvVlmsApiListener.AddNewAnswerCallback(listener));
        return addNewAnswerCall;
    }
}
