package com.easefun.polyvsdk.sub.auxilliary;

import android.content.Context;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.os.storage.StorageManager;

import java.io.File;
import java.lang.reflect.Method;

public class SDCardUtil {

    // 获取缓存路径
    public static File getDiskCacheDir(Context context, String path) {
        return new File(getDiskCacheDir(context), path);
    }

    // 获取缓存路径
    public static File getDiskCacheDir(Context context) {
        return (Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState())
                || !Environment.isExternalStorageRemovable()) ? context.getExternalCacheDir() : context.getCacheDir();
    }

    // 创建路径
    public static String createPath(Context context, String path) {
        String[] allPaths = SDCardUtil.getVolumePaths(context);
        String fitPath = null;
        String internalPath = null;
        long maxTotal = 0;
        for (String parentPath : allPaths) {
            boolean isMounted = SDCardUtil.checkMounted(context, parentPath);
            if (isMounted) {
                long total = SDCardUtil.getSDTotalSize(context, parentPath);
                if (total > maxTotal) {
                    maxTotal = total;
                    fitPath = parentPath;
                }
                if (!SDCardUtil.isRemovableSD(context, parentPath)) {
                    internalPath = parentPath;
                }
            }
        }
        File creataFile = new File(fitPath, path);
        if (SDCardUtil.isRemovableSD(context, fitPath)) {
            context.getExternalFilesDir(null);
            if (!creataFile.exists())
                creataFile.mkdirs();
            if (!creataFile.exists()) {
                StringBuilder stringBuilder = new StringBuilder();
                stringBuilder.append(fitPath).append(File.separator).append("Android").append(File.separator).append("data")
                        .append(File.separator).append(context.getPackageName()).append(File.separator).append(path);
                creataFile = new File(stringBuilder.toString());
                if (!creataFile.exists())
                    creataFile.mkdirs();
                if (!creataFile.exists() && internalPath != null) {
                    creataFile = new File(internalPath, path);
                    if (!creataFile.exists())
                        creataFile.mkdirs();
                }
            }
        } else {
            if (!creataFile.exists())
                creataFile.mkdirs();
        }
        return creataFile.exists() ? creataFile.getAbsolutePath() : null;
    }

    /**
     * 判断是否是外置sd卡
     */
    public static boolean isRemovableSD(Context context, String sdPath) {
        String eExternal = Environment.getExternalStorageDirectory().getAbsolutePath();
        if (isInternalSD(context, eExternal))
            return sdPath.startsWith(eExternal) ? false : true;
        else
            return sdPath.startsWith(getInternalSDPath(context)) ? false : true;
    }

    /**
     * 获取外置sd卡的路径
     */
    public static String getExtenalSDPath(Context context) {
        String[] allPaths = getVolumePaths(context);
        if (allPaths != null && allPaths.length > 0) {
            for (int i = 0; i < allPaths.length; i++) {
                if (isRemovableSD(context, allPaths[i]))
                    if (checkMounted(context, allPaths[i]))
                        return allPaths[i];
            }
        }
        return " ";
    }

    /**
     * 判断外置sd卡是否挂载
     */
    public static boolean externalSDIsMounted(Context context) {
        if (getExtenalSDPath(context).equals(" "))
            return false;
        return true;
    }

    /**
     * 判断有没内置sd卡,没挂载时return false
     */
    public static boolean internalSDIsMounted(Context context) {
        if (getInternalSDPath(context).equals(" "))
            return false;
        return true;
    }

    /**
     * 获取内置sd卡的路径
     */
    public static String getInternalSDPath(Context context) {
        String[] allPaths = getVolumePaths(context);
        if (allPaths != null && allPaths.length > 0) {
            for (int i = 0; i < allPaths.length; i++) {
                if (isInternalSD(context, allPaths[i]))
                    if (checkMounted(context, allPaths[i]))
                        return allPaths[i];
            }
        }
        return " ";
    }

    /**
     * 判断是否是内置sd卡
     */
    private static boolean isInternalSD(Context context, String sdPath) {
        if (context == null)
            return true;
        Class<?> mStorageVolume = null;
        StorageManager mStorageManager = null;
        Method mGetVolumeListMethod = null;
        Method mIsRemovableMethod = null;
        Object[] mStorageVolumeList = null;
        try {
            mStorageManager = (StorageManager) context.getSystemService(Context.STORAGE_SERVICE);
            mStorageVolume = Class.forName("android.os.storage.StorageVolume");
            mGetVolumeListMethod = mStorageManager.getClass().getMethod("getVolumeList", new Class[0]);
            mIsRemovableMethod = mStorageVolume.getMethod("isRemovable", new Class[0]);
            mStorageVolumeList = (Object[]) mGetVolumeListMethod.invoke(mStorageManager, new Object[0]);
            if (mStorageVolumeList != null && mStorageVolumeList.length > 0) {
                for (int i = 0; i < mStorageVolumeList.length; ++i) {
                    String mStoragePath = (String) mStorageVolume.getMethod("getPath", new Class[0])
                            .invoke(mStorageVolumeList[i], new Object[0]);
                    boolean isRemovableSD = ((Boolean) mIsRemovableMethod.invoke(mStorageVolumeList[i], new Object[0]))
                            .booleanValue();
                    if (sdPath.startsWith(mStoragePath) && isRemovableSD)
                        return false;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return true;
    }

    /**
     * 枚举所有挂载点(未挂载的也会获取)(外置sd卡被移除的时候，可能获取不到)
     */
    // 如果生成不了文件夹，可能是外置SD卡需要写入特定目录:外置sd卡路径/Android/data/包名/...
    // getExternalFilesDir(null);//(只有这种方法可以创建)创建sd卡的包名/files目录(之后要休眠一下才可创建其他文件?)，应用删除后，files/cache中的内容也会被删除
    // 可能会存在有外置sd卡，但外置sd卡的所有目录都无法创建/写入文件的情况
    // 外置sd卡创建目录之后立刻判断是否存在，可能会返回false
    public static String[] getVolumePaths(Context context) {
        String[] paths = null;
        StorageManager mStorageManager = null;
        Method mMethodGetPaths = null;
        try {
            mStorageManager = (StorageManager) context.getSystemService(Context.STORAGE_SERVICE);
            mMethodGetPaths = mStorageManager.getClass().getMethod("getVolumePaths");
            paths = (String[]) mMethodGetPaths.invoke(mStorageManager);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return paths;
    }

    /**
     * 检查是否挂载,只能是sd卡父目录
     */
    public static boolean checkMounted(Context context, String mountPoint) {
        if (mountPoint == null)
            return false;
        StorageManager storageManager = (StorageManager) context.getSystemService(Context.STORAGE_SERVICE);
        try {
            Method getVolumeState = storageManager.getClass().getMethod("getVolumeState", new Class[]{String.class});
            String state = (String) getVolumeState.invoke(storageManager, new Object[]{mountPoint});
            return Environment.MEDIA_MOUNTED.equals(state);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 获得SD卡总大小
     *
     * @return
     */
    @SuppressWarnings("deprecation")
    public static long getSDTotalSize(Context context, String path) {
        if (new File(path).exists()) {
            StatFs stat = new StatFs(path);
            long blockSize = 0;
            long totalBlocks = 0;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {// 获取版本等级号>=4.3时
                if (stat != null) {
                    blockSize = stat.getBlockSizeLong();// 4.3以上才支持(外存过大时)
                    totalBlocks = stat.getBlockCountLong();
                }
            } else {
                if (stat != null) {
                    blockSize = stat.getBlockSize();
                    totalBlocks = stat.getBlockCount();
                }
            }
            return blockSize * totalBlocks;
        }
        return 0;
    }

    /**
     * 获得sd卡剩余容量，即可用大小
     *
     * @return
     */
    @SuppressWarnings("deprecation")
    public static long getSDAvailableSize(Context context, String path) {
        if (new File(path).exists()) {
            StatFs stat = new StatFs(path);
            long blockSize = 0;
            long availableBlocks = 0;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {// 获取版本等级号>=4.3时
                if (stat != null) {
                    blockSize = stat.getBlockSizeLong();// 4.3以上才支持(外存过大时)
                    availableBlocks = stat.getAvailableBlocksLong();
                }
            } else {
                if (stat != null) {
                    blockSize = stat.getBlockSize();
                    availableBlocks = stat.getAvailableBlocks();
                }
            }
            return blockSize * availableBlocks;
        }
        return 0;
    }

    /**
     * 获得机身内存总大小
     *
     * @return
     */
    @SuppressWarnings("deprecation")
    public static long getRomTotalSize(Context context) {
        File path = Environment.getDataDirectory();
        StatFs stat = new StatFs(path.getPath());
        long blockSize;
        long totalBlocks;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {// 获取版本等级号>=4.3时
            blockSize = stat.getBlockSizeLong();// 4.3以上才支持(外存过大时)
            totalBlocks = stat.getBlockCountLong();
        } else {
            blockSize = stat.getBlockSize();
            totalBlocks = stat.getBlockCount();
        }
        return blockSize * totalBlocks;
    }

    /**
     * 获得机身可用内存
     *
     * @return
     */
    @SuppressWarnings("deprecation")
    public static long getRomAvailableSize(Context context) {
        File path = Environment.getDataDirectory();
        StatFs stat = new StatFs(path.getPath());
        long blockSize;
        long availableBlocks;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {// 获取版本等级号>=4.3时
            blockSize = stat.getBlockSizeLong();// 4.3以上才支持(外存过大时)
            availableBlocks = stat.getAvailableBlocksLong();
        } else {
            blockSize = stat.getBlockSize();
            availableBlocks = stat.getAvailableBlocks();
        }
        return blockSize * availableBlocks;
    }
}
