/*
 * Copyright (c) 2014 by EagleXad
 * Team: EagleXad
 * Create: 2014-08-29
 */

package com.eaglexad.lib.http.tool;

import android.os.Handler;
import android.os.Looper;

import com.eaglexad.lib.http.ExHttp;
import com.eaglexad.lib.http.entry.ExError;
import com.eaglexad.lib.http.entry.ExRequest;
import com.eaglexad.lib.http.entry.ExResponse;
import com.eaglexad.lib.http.exception.ExHttpException;
import com.eaglexad.lib.http.ible.IExNetwork;
import com.eaglexad.lib.http.ible.IExResponseDelivery;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author Aloneter
 * @ClassName: ExHttpQueue
 * @Description:
 */
public class ExHttpQueue {

    public static final int DEFAULT_NETWORK_THREAD_POOL_SIZE = 4;
    public static final int DEFAULT_NETWORK_THREAD_SINGLE_POOL_SIZE = 1;

    private AtomicInteger mSequenceGenerator = new AtomicInteger();

    private final Set<ExRequest> mCurrentRequests = new HashSet<>();
    private final Map<String, Queue<ExRequest>> mWaitingRequests = new HashMap<>();

    private final PriorityBlockingQueue<ExRequest> mCacheQueues = new PriorityBlockingQueue<>();
    private final PriorityBlockingQueue<ExRequest> mNetWorkQueues = new PriorityBlockingQueue<>();

    private final IExNetwork mNetWork;
    private final IExResponseDelivery mDelivery;

    private ExDispatcherNetwork[] mDispatcherNetWork;
    private ExDispatcherCache mDispatcherCache;

    public ExHttpQueue(IExNetwork netWork) {
        this(netWork, new ExResponseDelivery(new Handler(Looper.getMainLooper())));
    }

    public ExHttpQueue(IExNetwork netWork, int threadPoolSize) {
        this(netWork, new ExResponseDelivery(new Handler(Looper.getMainLooper())), threadPoolSize);
    }

    public ExHttpQueue(IExNetwork netWork, IExResponseDelivery delivery) {
        this(netWork, delivery, DEFAULT_NETWORK_THREAD_POOL_SIZE);
    }

    public ExHttpQueue(IExNetwork netWork, IExResponseDelivery delivery, int threadPoolSize) {
        mNetWork = netWork;
        mDelivery = delivery;
        mDispatcherNetWork = new ExDispatcherNetwork[threadPoolSize];
    }

    public void start() {

        stop();

        mDispatcherCache = new ExDispatcherCache(mCacheQueues, mNetWorkQueues, mDelivery);
        mDispatcherCache.start();

        int length = mDispatcherNetWork.length;

        for (int i = 0; i < length; i++) {
            ExDispatcherNetwork dispatcherNetwork = new ExDispatcherNetwork(mNetWorkQueues, mNetWork, mDelivery);
            mDispatcherNetWork[i] = dispatcherNetwork;
            dispatcherNetwork.start();
        }
    }

    public void stop() {

        if (mDispatcherCache != null) {
            mDispatcherCache.quit();
        }

        int length = mDispatcherNetWork.length;

        for (int i = 0; i < length; i++) {
            ExDispatcherNetwork dispatcherNetwork = mDispatcherNetWork[i];

            if (dispatcherNetwork != null) {
                dispatcherNetwork.quit();
            }
        }
    }

    public void add(ExRequest request) {

        if (ExHttpUtils.isEmpty(request) || request.isCancel) {
            return;
        }

        request.setQueue(this);

        synchronized (mCurrentRequests) {
            mCurrentRequests.add(request);
        }

        request.setSequence(getSequenceNumber());

        if (!request.shouldCache) {
            mNetWorkQueues.add(request);

            return;
        }

        synchronized (mWaitingRequests) {
            String keyCache = request.getCacheKey();

            if (mWaitingRequests.containsKey(keyCache)) {
                Queue<ExRequest> stagedRequests = mWaitingRequests.get(keyCache);

                if (stagedRequests == null) {
                    stagedRequests = new LinkedList<>();
                }

                stagedRequests.add(request);
                mWaitingRequests.put(keyCache, stagedRequests);

                return;
            }

            mWaitingRequests.put(keyCache, null);
            mCacheQueues.add(request);
        }
    }

    public void execute(ExRequest request) {

        if (ExHttpUtils.isEmpty(request) || request.isCancel) {
            return;
        }
        if (request.shouldCache) {
            mDispatcherCache.execute(request, true);
        }

        int length = mDispatcherNetWork.length;
        if (length > 0) {
            mDispatcherNetWork[0].execute(request, true);
        }
    }

    public void cancelAll(final String tag) {

        if (!ExHttpUtils.isEmpty(tag)) {
            cancelAll(new RequestFilter() {
                @Override
                public boolean apply(ExRequest request) {

                    return request.tag.equals(tag);
                }
            });
        }
    }

    public void cancelAll(RequestFilter filter) {

        synchronized (mCurrentRequests) {
            for (ExRequest request : mCurrentRequests) {
                if (filter.apply(request)) {
                    request.cancel();
                }
            }
        }
    }

    public void finish(ExRequest request) {

        synchronized (mCurrentRequests) {
            mCurrentRequests.remove(request);
        }

        if (request.shouldCache) {
            synchronized (mWaitingRequests) {
                String keyCache = request.getCacheKey();
                Queue<ExRequest> waitingRequests = mWaitingRequests.remove(keyCache);
                if (waitingRequests != null) {
                    mCacheQueues.addAll(waitingRequests);
                }
            }
        }
    }

    public int getSequenceNumber() {

        return mSequenceGenerator.incrementAndGet();
    }

    public interface RequestFilter {

        boolean apply(ExRequest request);

    }

}
