/*
 * Copyright (c) 2014 by EagleXad
 * Team: EagleXad
 * Create: 2014-08-29
 */

package com.eaglexad.lib.http.body;

import com.eaglexad.lib.http.entry.ExRequest;
import com.eaglexad.lib.http.ible.IExRequestBody;
import com.eaglexad.lib.http.tool.ExHttpUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author Aloneter
 * @ClassName: ExMultipartBody
 * @Description:
 */
public class ExMultipartBody implements IExRequestBody {

    private static byte[] BOUNDARY_PREFIX_BYTES = "--------7da3d81520810".getBytes();
    private static byte[] END_BYTES = "\r\n".getBytes();
    private static byte[] TWO_DASHES_BYTES = "--".getBytes();

    private byte[] mBoundaryPostfixBytes;
    private String mContentType; // multipart/subtype; boundary=xxx...
    private String mCharset = "UTF-8";

    private List<ExRequest.KeyValue> mMultipartParams;
    private long mTotal = 0;
    private long mCurrent = 0;

    public ExMultipartBody(List<ExRequest.KeyValue> multipartParams, String charset) {

        if (!ExHttpUtils.isEmpty(charset)) {
            mCharset = charset;
        }

        mMultipartParams = multipartParams;

        String boundaryPostfix = Double.toHexString(Math.random() * 0xFFFF);
        mBoundaryPostfixBytes = boundaryPostfix.getBytes();
        mContentType = "multipart/form-data; boundary=" + new String(BOUNDARY_PREFIX_BYTES) + boundaryPostfix;
    }

    @Override
    public long getContentLength() {

        return mTotal;
    }

    @Override
    public void setContentType(String contentType) {

        if (!ExHttpUtils.isEmpty(contentType)) {
            int index = contentType.indexOf(";");
            mContentType = "multipart/" + contentType + contentType.substring(index);
        }
    }

    @Override
    public String getContentType() {

        return mContentType;
    }

    @Override
    public void writeTo(OutputStream out) throws IOException {

        for (ExRequest.KeyValue kv : mMultipartParams) {
            String name = kv.key;
            Object value = kv.value;
            if (!ExHttpUtils.isEmpty(name) && value != null) {
                writeEntry(out, name, value);
            }
        }

        writeLine(out, TWO_DASHES_BYTES, BOUNDARY_PREFIX_BYTES, mBoundaryPostfixBytes, TWO_DASHES_BYTES);
        out.flush();
    }

    private void writeEntry(OutputStream out, String name, Object value) throws IOException {

        writeLine(out, TWO_DASHES_BYTES, BOUNDARY_PREFIX_BYTES, mBoundaryPostfixBytes);

        String fileName = "";
        String contentType = null;

        if (value instanceof File) {
            File file = (File) value;
            if (ExHttpUtils.isEmpty(fileName)) {
                fileName = file.getName();
            }

            contentType = ExHttpUtils.getFileContentType(file);

            writeLine(out, buildContentDisposition(name, fileName, mCharset));
            writeLine(out, buildContentType(value, contentType, mCharset));

            writeLine(out); // 内容前空一行

            writeFile(out, file);
            writeLine(out);
        } else {
            writeLine(out, buildContentDisposition(name, fileName, mCharset));
            writeLine(out, buildContentType(value, contentType, mCharset));

            writeLine(out); // 内容前空一行

            if (value instanceof InputStream) {
                writeStreamAndCloseIn(out, (InputStream) value);
                writeLine(out);
            } else {
                byte[] content;

                if (value instanceof byte[]) {
                    content = (byte[]) value;
                } else {
                    content = String.valueOf(value).getBytes(mCharset);
                }

                writeLine(out, content);
                mCurrent += content.length;
            }
        }
    }

    private void writeLine(OutputStream out, byte[]... bs) throws IOException {

        if (bs != null) {
            for (byte[] b : bs) {
                out.write(b);
            }
        }

        out.write(END_BYTES);
    }

    private void writeFile(OutputStream out, File file) throws IOException {

        if (out instanceof CounterOutputStream) {
            ((CounterOutputStream) out).addFile(file);
        } else {
            writeStreamAndCloseIn(out, new FileInputStream(file));
        }
    }

    private void writeStreamAndCloseIn(OutputStream out, InputStream in) throws IOException {

        if (out instanceof CounterOutputStream) {
            ((CounterOutputStream) out).addStream(in);
        } else {
            try {
                int len;
                byte[] buf = new byte[1024];

                while ((len = in.read(buf)) >= 0) {
                    out.write(buf, 0, len);
                    mCurrent += len;
                }
            } finally {
                ExHttpUtils.closeQuietly(in);
            }
        }
    }

    private static byte[] buildContentDisposition(String name, String fileName, String charset) throws UnsupportedEncodingException {

        StringBuilder result = new StringBuilder("Content-Disposition: form-data");

        result.append("; name=\"");
        result.append(name.replace("\"", "\\\""));
        result.append("\"");

        if (!ExHttpUtils.isEmpty(fileName)) {
            result.append("; filename=\"");
            result.append(fileName.replace("\"", "\\\""));
            result.append("\"");
        }

        return result.toString().getBytes(charset);
    }

    private static byte[] buildContentType(Object value, String contentType, String charset) throws UnsupportedEncodingException {

        StringBuilder result = new StringBuilder("Content-Type: ");

        if (ExHttpUtils.isEmpty(contentType)) {
            if (value instanceof String) {
                contentType = "text/plain; charset:" + charset;
            } else {
                contentType = "application/octet-stream";
            }
        } else {
            contentType = contentType.replaceFirst("\\/jpg$", "/jpeg");
        }
        result.append(contentType);
        return result.toString().getBytes(charset);
    }

    private class CounterOutputStream extends OutputStream {

        final AtomicLong total = new AtomicLong(0L);

        public CounterOutputStream() {
        }

        public void addFile(File file) {
            if (total.get() == -1L) return;
            total.addAndGet(file.length());
        }

        public void addStream(InputStream inputStream) {
            if (total.get() == -1L) return;
            long length = ExHttpUtils.getInputStreamLength(inputStream);
            if (length > 0) {
                total.addAndGet(length);
            } else {
                total.set(-1L);
            }
        }

        @Override
        public void write(int oneByte) throws IOException {
            if (total.get() == -1L) return;
            total.incrementAndGet();
        }

        @Override
        public void write(byte[] buffer) throws IOException {
            if (total.get() == -1L) return;
            total.addAndGet(buffer.length);
        }

        @Override
        public void write(byte[] buffer, int offset, int count) throws IOException {
            if (total.get() == -1L) return;
            total.addAndGet(count);
        }
    }

}
