package com.dynatrace.tools.android;

import static java.lang.Thread.currentThread;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;

import org.gradle.api.Action;
import org.gradle.api.NamedDomainObjectCollection;
import org.gradle.api.Project;

class DynatraceConfig {
	private final NamedDomainObjectCollection<ApplicationFlavor> productFlavors;
	private final ApplicationFlavor defaultConfig;
	private final Project project;
	private File apkitDir = null;

	public DynatraceConfig(Project project) {
		this.project = project;
		this.defaultConfig = new ApplicationFlavor("defaults");
		this.productFlavors = project.container(ApplicationFlavor.class, s -> {
			ApplicationFlavor flavor = new ApplicationFlavor(s);

			flavor.cluster(defaultConfig.getCluster());
			flavor.applicationId(defaultConfig.getApplicationId());
			flavor.environmentId(defaultConfig.getEnvironmentId());
			flavor.startupPath(defaultConfig.getStartupPath());
			flavor.agentProperties(defaultConfig.getAgentProperties());
			flavor.beaconURL(defaultConfig.getBeaconURL());

			return flavor;
		});
	}

	public NamedDomainObjectCollection<ApplicationFlavor> getProductFlavors() {
		return productFlavors;
	}

	public ApplicationFlavor getDefaultConfig() {
		return defaultConfig;
	}

	public void defaultConfig(Action<ApplicationFlavor> action) {
		action.execute(defaultConfig);
	}

	public void productFlavors(Action<NamedDomainObjectCollection<ApplicationFlavor>> action) {
		action.execute(productFlavors);
	}

	public synchronized File apkitDir() {
		String agentName = "dynatrace-mobile-agent-android-" + Version.getVersion();
		File tempDir = project.getTasks().getByName("prepareApkit").getTemporaryDir();
		if (apkitDir == null) {
			apkitDir = new File(tempDir, agentName);
		}
		if (!apkitDir.exists()) {
			String zipFileName = agentName + ".zip";
			File agentZip = new File(tempDir, zipFileName);
			if (agentZip.exists()) {
				agentZip.delete();
			}
			try (InputStream input = currentThread().getContextClassLoader().getResourceAsStream(zipFileName)) {
				if (input == null) {
					throw new FileNotFoundException("Could not find file '" + zipFileName + "'");
				}
				Files.copy(input, agentZip.toPath());
			} catch (IOException e) {
				throw new RuntimeException(e);
			}

			project.copy(copy -> {
				copy.from(project.zipTree(agentZip));
				copy.into(apkitDir);
			});
		}

		return apkitDir;
	}

	public String agent() {
		return "com.dynatrace.agent:agent-android:" + Version.getVersion() + "@aar";
	}

	ApplicationFlavor findProductFlavor(String flavorName) {
		ApplicationFlavor flavor = null;

		if (flavorName != null && flavorName.length() > 0) {
			flavor = productFlavors.findByName(flavorName.toLowerCase());
		}

		if (flavor == null) {
			flavor = defaultConfig;
		}

		return flavor;
	}
}
