/*
 * Modified by Vladyslav Lozytskyi on 11.04.18 1:29
 * Copyright (c) 2018. All rights reserved.
 */

package com.don11995.log;

import android.os.Build;
import android.support.annotation.IntDef;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * SimpleLog - an extended library for logging on devices running Android.
 * Unlike the standard log, the library can generate tags and output method names
 * depending on where the log was called, and also solves some additional tasks.
 */

@SuppressWarnings("unused")
public final class SimpleLog {

    /**
     * Priority constant for the println method;
     * Use SimpleLog.e, SimpleLog.fe, SimpleLog.te, SimpleLog.tfe.
     */
    public static final int LOG_LEVEL_ERROR = 0;

    /**
     * Priority constant for the println method;
     * Use SimpleLog.d, SimpleLog.fd, SimpleLog.td, SimpleLog.tfd.
     */
    public static final int LOG_LEVEL_DEBUG = 1;

    /**
     * Priority constant for the println method;
     * Use SimpleLog.w, SimpleLog.fw, SimpleLog.tw, SimpleLog.tfw.
     */
    public static final int LOG_LEVEL_WARNING = 2;

    /**
     * Priority constant for the println method;
     * Use SimpleLog.i, SimpleLog.fi, SimpleLog.ti, SimpleLog.tfi.
     */
    public static final int LOG_LEVEL_INFO = 3;

    /**
     * Priority constant for the println method;
     * Use SimpleLog.v, SimpleLog.fv, SimpleLog.tv, SimpleLog.tfv.
     */
    public static final int LOG_LEVEL_VERBOSE = 4;

    /**
     * Priority constant for the println method;
     * Use SimpleLog.wtf, SimpleLog.fwtf, SimpleLog.twtf, SimpleLog.tfwtf.
     */
    public static final int LOG_LEVEL_ASSERT = 5;

    /**
     * Max log size for one Log call
     */
    private static final int MAX_LOG_CHUNK_SIZE = 4000;

    /**
     * Max tag length for devices with target API < 24
     */
    private static final int MAX_TAG_LENGTH = 23;

    /**
     * Constant to get class name from stack
     */
    private static final int CALL_STACK_INDEX = 2;

    /**
     * Array that contains enabled log levels
     */
    private static final Set<Integer> sLogLevels;

    /**
     * Divider char to use with {@link Group}
     */
    private static char sDividerChar = '-';

    /**
     * Lenght of one block {@link #sDividerChar}
     */
    private static int sDividerBlockSize = 8;

    private static List<LogProcessor> sLogProcessorList;

    private static boolean sPrintReferences = false;

    static {
        sLogLevels = new HashSet<>();
        sLogProcessorList = new ArrayList<>();
        enableAllLogs();
    }

    private SimpleLog() {
    }

    /**
     * Disable all logs
     */
    public static void disableAllLogs() {
        sLogLevels.clear();
    }

    /**
     * Enable all logs
     */
    public static void enableAllLogs() {
        sLogLevels.add(LOG_LEVEL_ERROR);
        sLogLevels.add(LOG_LEVEL_DEBUG);
        sLogLevels.add(LOG_LEVEL_WARNING);
        sLogLevels.add(LOG_LEVEL_INFO);
        sLogLevels.add(LOG_LEVEL_VERBOSE);
        sLogLevels.add(LOG_LEVEL_ASSERT);
    }

    /**
     * Enable/disable log level
     *
     * @param logLevel log level to enable/disable. Can be one of {@link LogLevel}
     * @param enabled  enable/disable flag
     */
    public static void setLogLevelEnabled(@LogLevel int logLevel, boolean enabled) {
        if (enabled) {
            sLogLevels.add(logLevel);
        } else {
            sLogLevels.remove(logLevel);
        }
    }

    /**
     * Check if log level enabled/disabled
     *
     * @param logLevel log level to enable/disable. Can be one of {@link LogLevel}
     * @return true, if logLevel enabled
     */
    public static boolean isLogLevelEnabled(@LogLevel int logLevel) {
        return sLogLevels.contains(logLevel);
    }

    /**
     * Set divider char to use with {@link Group}
     *
     * @param divider char to set
     */
    public static void setDividerChar(char divider) {
        sDividerChar = divider;
    }

    /**
     * Set divider block size to use with {@link Group}
     *
     * @param dividerBlockSize length of one block size
     */
    public static void setDividerBlockSize(@IntRange(from = 1) int dividerBlockSize) {
        sDividerBlockSize = dividerBlockSize;
    }

    private static String formatText(String text, Object... args) {
        if (TextUtils.isEmpty(text)
                || args == null
                || args.length == 0)
            return text;
        return String.format(Locale.getDefault(), text, args);
    }

    private static String formatText(Group group) {
        String format = group.getText();
        String groupName = group.getGroupName();
        if (groupName == null) groupName = "";
        char[] buffer = new char[sDividerBlockSize];
        Arrays.fill(buffer,
                sDividerChar);
        String divider = new String(buffer);
        format = divider + groupName + divider + "\n\t" + format;
        buffer = new char[groupName.length()];
        Arrays.fill(buffer,
                sDividerChar);
        format += "\n\t" + divider + divider + new String(buffer);
        return format;
    }

    @Nullable
    private static String getTagFromObject(@Nullable Object object) {
        if (object instanceof Group) {
            return ((Group) object).getTag();
        } else {
            return null;
        }
    }

    @Nullable
    private static String getMessageFromObject(@Nullable Object object) {
        if (object instanceof Group) {
            return formatText((Group) object);
        } else if (object instanceof Throwable) {
            StringWriter errors = new StringWriter();
            ((Throwable) object).printStackTrace(new PrintWriter(errors));
            return errors.toString();
        } else {
            return object != null ? object.toString() : null;
        }
    }

    private static boolean isGroupObject(@Nullable Object object) {
        return object instanceof Group;
    }

    /**
     * Print current method name
     */
    public static void fd() {
        printLog(LOG_LEVEL_DEBUG, null, true, null, null, false);
    }

    /**
     * Print current method name
     */
    public static void fe() {
        printLog(LOG_LEVEL_ERROR, null, true, null, null, false);
    }

    /**
     * Print current method name
     */
    public static void fi() {
        printLog(LOG_LEVEL_INFO, null, true, null, null, false);
    }

    /**
     * Print current method name
     */
    public static void fv() {
        printLog(LOG_LEVEL_VERBOSE, null, true, null, null, false);
    }

    /**
     * Print current method name
     */
    public static void fw() {
        printLog(LOG_LEVEL_WARNING, null, true, null, null, false);
    }

    /**
     * Print current method name
     */
    public static void fwtf() {
        printLog(LOG_LEVEL_ASSERT, null, true, null, null, false);
    }

    /**
     * Print current method name with custom tag
     *
     * @param tag tag to use
     */
    public static void tfd(String tag) {
        printLog(LOG_LEVEL_DEBUG, null, true, tag, null, false);
    }

    /**
     * Print current method name with custom tag
     *
     * @param tag tag to use
     */
    public static void tfe(String tag) {
        printLog(LOG_LEVEL_ERROR, null, true, tag, null, false);
    }

    /**
     * Print current method name with custom tag
     *
     * @param tag tag to use
     */
    public static void tfi(String tag) {
        printLog(LOG_LEVEL_INFO, null, true, tag, null, false);
    }

    /**
     * Print current method name with custom tag
     *
     * @param tag tag to use
     */
    public static void tfv(String tag) {
        printLog(LOG_LEVEL_VERBOSE, null, true, tag, null, false);
    }

    /**
     * Print current method name with custom tag
     *
     * @param tag tag to use
     */
    public static void tfw(String tag) {
        printLog(LOG_LEVEL_WARNING, null, true, tag, null, false);
    }

    /**
     * Print current method name with custom tag
     *
     * @param tag tag to use
     */
    public static void tfwtf(String tag) {
        printLog(LOG_LEVEL_ASSERT, null, true, tag, null, false);
    }

    /**
     * Print Object into logcat. Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void d(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_DEBUG, getMessageFromObject(object), false, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat. Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void e(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_ERROR, getMessageFromObject(object), false, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat. Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void i(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_INFO, getMessageFromObject(object), false, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat. Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void v(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_VERBOSE, getMessageFromObject(object), false, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat. Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void w(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_WARNING, getMessageFromObject(object), false, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat. Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void wtf(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_ASSERT, getMessageFromObject(object), false, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void td(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_DEBUG, getMessageFromObject(object), false, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void te(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_ERROR, getMessageFromObject(object), false, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void ti(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_INFO, getMessageFromObject(object), false, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void tv(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_VERBOSE, getMessageFromObject(object), false, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void tw(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_WARNING, getMessageFromObject(object), false, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void twtf(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_ASSERT, getMessageFromObject(object), false, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void fd(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_DEBUG, getMessageFromObject(object), true, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void fe(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_ERROR, getMessageFromObject(object), true, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void fi(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_INFO, getMessageFromObject(object), true, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void fv(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_VERBOSE, getMessageFromObject(object), true, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void fw(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_WARNING, getMessageFromObject(object), true, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * Object can be any type or {@link Group}
     *
     * @param object object to print into logcat
     */
    public static void fwtf(Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_ASSERT, getMessageFromObject(object), true, getTagFromObject(object),
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void tfd(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_DEBUG, getMessageFromObject(object), true, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void tfe(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_ERROR, getMessageFromObject(object), true, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void tfi(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_INFO, getMessageFromObject(object), true, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void tfv(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_VERBOSE, getMessageFromObject(object), true, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void tfw(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_WARNING, getMessageFromObject(object), true, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag    tag to use
     * @param object object to print into logcat
     */
    public static void tfwtf(String tag, Object object) {
        boolean isGroup = isGroupObject(object);
        printLog(LOG_LEVEL_ASSERT, getMessageFromObject(object), true, tag,
                (object instanceof Throwable) ? (Throwable) object : null,
                isGroup);
    }

    /**
     * Print string with args using {@link String#format}
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void d(String format, Object... args) {
        printLog(LOG_LEVEL_DEBUG, formatText(format, args), false, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void e(String format, Object... args) {
        printLog(LOG_LEVEL_ERROR, formatText(format, args), false, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void i(String format, Object... args) {
        printLog(LOG_LEVEL_INFO, formatText(format, args), false, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void v(String format, Object... args) {
        printLog(LOG_LEVEL_VERBOSE, formatText(format, args), false, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void w(String format, Object... args) {
        printLog(LOG_LEVEL_WARNING, formatText(format, args), false, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void wtf(String format, Object... args) {
        printLog(LOG_LEVEL_ASSERT, formatText(format, args), false, null, null, false);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void td(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_DEBUG, formatText(format, objects), false, tag, null, false);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void te(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_ERROR, formatText(format, objects), false, tag, null, false);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void ti(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_INFO, formatText(format, objects), false, tag, null, false);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void tv(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_VERBOSE, formatText(format, objects), false, tag, null, false);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void tw(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_WARNING, formatText(format, objects), false, tag, null, false);
    }

    /**
     * Print Object into logcat with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void twtf(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_ASSERT, formatText(format, objects), false, tag, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     * with current method name at start
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void fd(String format, Object... args) {
        printLog(LOG_LEVEL_DEBUG, formatText(format, args), true, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     * with current method name at start
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void fe(String format, Object... args) {
        printLog(LOG_LEVEL_ERROR, formatText(format, args), true, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     * with current method name at start
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void fi(String format, Object... args) {
        printLog(LOG_LEVEL_INFO, formatText(format, args), true, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     * with current method name at start
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void fv(String format, Object... args) {
        printLog(LOG_LEVEL_VERBOSE, formatText(format, args), true, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     * with current method name at start
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void fw(String format, Object... args) {
        printLog(LOG_LEVEL_WARNING, formatText(format, args), true, null, null, false);
    }

    /**
     * Print string with args using {@link String#format}
     * with current method name at start
     *
     * @param format string to print
     * @param args   args to use with {@link String#format}
     */
    public static void fwtf(String format, Object... args) {
        printLog(LOG_LEVEL_ASSERT, formatText(format, args), true, null, null, false);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void tfd(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_DEBUG, formatText(format, objects), true, tag, null, false);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void tfe(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_ERROR, formatText(format, objects), true, tag, null, false);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void tfi(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_INFO, formatText(format, objects), true, tag, null, false);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void tfv(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_VERBOSE, formatText(format, objects), true, tag, null, false);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void tfw(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_WARNING, formatText(format, objects), true, tag, null, false);
    }

    /**
     * Print Object into logcat with current function name at start of log
     * and with custom tag
     * Object can be any type or {@link Group}
     *
     * @param tag     tag to use
     * @param format  string to print
     * @param objects args to use with {@link String#format}
     */
    public static void tfwtf(String tag, String format, Object... objects) {
        printLog(LOG_LEVEL_ASSERT, formatText(format, objects), true, tag, null, false);
    }

    /**
     * Add {@link LogProcessor} to log callbacks.
     * {@link LogProcessor} can add additional logic to every log call
     *
     * @param logProcessor {@link LogProcessor} to add
     */
    public static void addLogProcessor(@NonNull LogProcessor logProcessor) {
        sLogProcessorList.add(logProcessor);
    }

    /**
     * Remove {@link LogProcessor} from log callbacks
     *
     * @param logProcessor {@link LogProcessor} to delete
     */
    public static void removeLogProcessor(@NonNull LogProcessor logProcessor) {
        sLogProcessorList.remove(logProcessor);
    }

    /**
     * Enbale/disable printing line of code where log was called
     *
     * @param enabled enable/disable print log reference
     */
    public static void setPrintReferences(boolean enabled) {
        sPrintReferences = enabled;
    }

    private static String detectClassName(String origClassName) {
        if (TextUtils.isEmpty(origClassName)) return origClassName;
        String className = origClassName;
        int lastDot = className.lastIndexOf('.');
        if (lastDot >= 0) {
            className = className.substring(lastDot + 1);
        }
        int firstDollar = className.indexOf('$');
        if (firstDollar > 0) {
            className = className.substring(0, firstDollar);
        }
        return className;
    }

    private static void printLog(@LogLevel int logLevel,
                                 @Nullable String inMessage,
                                 boolean printMethodName,
                                 @Nullable String inTag,
                                 @Nullable Throwable e,
                                 boolean isGroup) {
        if (!sLogLevels.contains(logLevel)) return;
        String message = inMessage;
        String tag = inTag;
        if (message == null) message = "";
        StackTraceElement element = new Throwable().getStackTrace()[CALL_STACK_INDEX];
        String className = detectClassName(element.getClassName());
        if (TextUtils.isEmpty(tag)) {
            tag = className;
        }
        if (tag.length() > MAX_TAG_LENGTH && Build.VERSION.SDK_INT < Build.VERSION_CODES.N) {
            tag = tag.substring(0, MAX_TAG_LENGTH);
        }
        message = message.replaceAll("\r", "")
                .replaceAll("\n+", "\n")
                .trim();
        if (printMethodName) {
            String method = element.getMethodName();
            if (message.isEmpty()) {
                message = method + "()";
            } else {
                message = method + "() -> "
                        + (isGroup ? "\n\t" : "")
                        + message;
            }
        }
        if (sPrintReferences && e == null) {
            message = '(' + className + ".java:"
                    + element.getLineNumber() + ')'
                    + (isGroup ? "\n\t" : ' ')
                    + message;
            message = message.trim();
        }
        if (message.isEmpty()) return;
        ArrayList<String> logs = new ArrayList<>();
        while (message.length() > MAX_LOG_CHUNK_SIZE) {
            String temp = message.substring(0, MAX_LOG_CHUNK_SIZE);
            int nextStartIndex = temp.lastIndexOf('\n');
            if (nextStartIndex <= 0) {
                nextStartIndex = MAX_LOG_CHUNK_SIZE;
            }
            temp = temp.substring(0, nextStartIndex).trim();
            logs.add(temp);
            message = message.substring(nextStartIndex).trim();
        }
        logs.add(message);
        for (String s : logs) {
            switch (logLevel) {
                case LOG_LEVEL_DEBUG:
                    android.util.Log.d(tag, s);
                    break;
                case LOG_LEVEL_ERROR:
                    android.util.Log.e(tag, s);
                    break;
                case LOG_LEVEL_INFO:
                    android.util.Log.i(tag, s);
                    break;
                case LOG_LEVEL_VERBOSE:
                    android.util.Log.v(tag, s);
                    break;
                case LOG_LEVEL_WARNING:
                    android.util.Log.w(tag, s);
                    break;
                case LOG_LEVEL_ASSERT:
                    android.util.Log.wtf(tag, s);
                    break;
                default:
                    break;
            }
        }
        for (LogProcessor logProcessor : sLogProcessorList) {
            logProcessor.handleProcessLog(tag, message, logLevel, e);
        }
    }

    @SuppressWarnings("WeakerAccess")
    @Retention(RetentionPolicy.SOURCE)
    @IntDef({
            LOG_LEVEL_DEBUG,
            LOG_LEVEL_ERROR,
            LOG_LEVEL_INFO,
            LOG_LEVEL_VERBOSE,
            LOG_LEVEL_WARNING,
            LOG_LEVEL_ASSERT
    })
    public @interface LogLevel {
    }
}

