/*
 * digitalpetri OPC-UA SDK
 *
 * Copyright (C) 2015 Kevin Herron
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.digitalpetri.opcua.sdk.server;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;

import com.digitalpetri.opcua.sdk.server.services.AttributeServices;
import com.digitalpetri.opcua.sdk.server.services.MethodServices;
import com.digitalpetri.opcua.sdk.server.services.MonitoredItemServices;
import com.digitalpetri.opcua.sdk.server.services.NodeManagementServices;
import com.digitalpetri.opcua.sdk.server.services.QueryServices;
import com.digitalpetri.opcua.sdk.server.services.SubscriptionServices;
import com.digitalpetri.opcua.sdk.server.services.ViewServices;
import com.digitalpetri.opcua.sdk.server.subscriptions.SubscriptionManager;
import com.digitalpetri.opcua.stack.core.StatusCodes;
import com.digitalpetri.opcua.stack.core.UaException;
import com.digitalpetri.opcua.stack.core.application.services.NodeManagementServiceSet;
import com.digitalpetri.opcua.stack.core.application.services.ServiceRequest;
import com.digitalpetri.opcua.stack.core.application.services.SessionServiceSet;
import com.digitalpetri.opcua.stack.core.types.builtin.ByteString;
import com.digitalpetri.opcua.stack.core.types.builtin.NodeId;
import com.digitalpetri.opcua.stack.core.types.structured.ActivateSessionRequest;
import com.digitalpetri.opcua.stack.core.types.structured.ActivateSessionResponse;
import com.digitalpetri.opcua.stack.core.types.structured.CancelRequest;
import com.digitalpetri.opcua.stack.core.types.structured.CancelResponse;
import com.digitalpetri.opcua.stack.core.types.structured.CloseSessionRequest;
import com.digitalpetri.opcua.stack.core.types.structured.CloseSessionResponse;
import com.digitalpetri.opcua.stack.core.types.structured.CreateSessionRequest;
import com.digitalpetri.opcua.stack.core.types.structured.CreateSessionResponse;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.digitalpetri.opcua.stack.core.types.builtin.unsigned.Unsigned.uint;

public class Session implements SessionServiceSet {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final List<LifecycleListener> listeners = Lists.newCopyOnWriteArrayList();

    private final SubscriptionManager subscriptionManager;

    private volatile long secureChannelId;

    private volatile Object identityObject;
    private volatile ByteString clientCertificateBytes;

    private volatile ByteString lastNonce = ByteString.NULL_VALUE;

    private volatile long lastActivity = System.nanoTime();
    private volatile ScheduledFuture<?> checkTimeoutFuture;

    private final AttributeServices attributeServices;
    private final MethodServices methodServices;
    private final MonitoredItemServices monitoredItemServices;
    private final NodeManagementServiceSet nodeManagementServices;
    private final QueryServices queryServices;
    private final SubscriptionServices subscriptionServices;
    private final ViewServices viewServices;

    private final OpcUaServer server;
    private final NodeId sessionId;
    private final String sessionName;
    private final Duration sessionTimeout;

    public Session(OpcUaServer server,
                   NodeId sessionId,
                   String sessionName,
                   Duration sessionTimeout,
                   long secureChannelId) {

        this.server = server;
        this.sessionId = sessionId;
        this.sessionName = sessionName;
        this.sessionTimeout = sessionTimeout;
        this.secureChannelId = secureChannelId;

        subscriptionManager = new SubscriptionManager(this, server);

        attributeServices = new AttributeServices();
        methodServices = new MethodServices();
        monitoredItemServices = new MonitoredItemServices(subscriptionManager);
        nodeManagementServices = new NodeManagementServices();
        queryServices = new QueryServices();
        subscriptionServices = new SubscriptionServices(subscriptionManager);
        viewServices = new ViewServices();

        checkTimeoutFuture = server.getScheduledExecutorService().schedule(
                this::checkTimeout, sessionTimeout.toNanos(), TimeUnit.NANOSECONDS);
    }

    public long getSecureChannelId() {
        return secureChannelId;
    }

    @Nullable
    public Object getIdentityObject() {
        return identityObject;
    }

    @Nullable
    public ByteString getClientCertificateBytes() {
        return clientCertificateBytes;
    }

    public void setSecureChannelId(long secureChannelId) {
        this.secureChannelId = secureChannelId;
    }

    public void setIdentityObject(Object identityObject) {
        this.identityObject = identityObject;
    }

    public void setClientCertificateBytes(ByteString clientCertificateBytes) {
        this.clientCertificateBytes = clientCertificateBytes;
    }

    void addLifecycleListener(LifecycleListener listener) {
        listeners.add(listener);
    }

    void updateLastActivity() {
        lastActivity = System.nanoTime();
    }

    void setLastNonce(ByteString lastNonce) {
        this.lastNonce = lastNonce;
    }

    public ByteString getLastNonce() {
        return lastNonce;
    }

    private void checkTimeout() {
        long elapsed = Math.abs(System.nanoTime() - lastActivity);

        if (elapsed > sessionTimeout.toNanos()) {
            logger.debug("Session id={} lifetime expired ({}ms).", sessionId, sessionTimeout.toMillis());

            subscriptionManager.sessionClosed(true);

            listeners.forEach(listener -> listener.onSessionClosed(this, true));
        } else {
            long remaining = sessionTimeout.toNanos() - elapsed;
            logger.trace("Session id={} timeout scheduled for +{}s.",
                    sessionId, Duration.ofNanos(remaining).getSeconds());

            checkTimeoutFuture = server.getScheduledExecutorService()
                    .schedule(this::checkTimeout, remaining, TimeUnit.NANOSECONDS);
        }
    }

    public NodeId getSessionId() {
        return sessionId;
    }

    public String getSessionName() {
        return sessionName;
    }

    public AttributeServices getAttributeServices() {
        return attributeServices;
    }

    public MethodServices getMethodServices() {
        return methodServices;
    }

    public MonitoredItemServices getMonitoredItemServices() {
        return monitoredItemServices;
    }

    public NodeManagementServiceSet getNodeManagementServices() {
        return nodeManagementServices;
    }

    public QueryServices getQueryServices() {
        return queryServices;
    }

    public SubscriptionServices getSubscriptionServices() {
        return subscriptionServices;
    }

    public ViewServices getViewServices() {
        return viewServices;
    }

    public SubscriptionManager getSubscriptionManager() {
        return subscriptionManager;
    }

    //region Session Services
    @Override
    public void onCreateSession(ServiceRequest<CreateSessionRequest, CreateSessionResponse> req) throws UaException {
        throw new UaException(StatusCodes.Bad_InternalError);
    }

    @Override
    public void onActivateSession(ServiceRequest<ActivateSessionRequest, ActivateSessionResponse> req) throws UaException {
        throw new UaException(StatusCodes.Bad_InternalError);
    }

    @Override
    public void onCloseSession(ServiceRequest<CloseSessionRequest, CloseSessionResponse> serviceRequest) throws UaException {
        close(serviceRequest.getRequest().getDeleteSubscriptions());

        serviceRequest.setResponse(new CloseSessionResponse(serviceRequest.createResponseHeader()));
    }

    void close(boolean deleteSubscriptions) {
        if (checkTimeoutFuture != null) {
            checkTimeoutFuture.cancel(false);
        }

        subscriptionManager.sessionClosed(deleteSubscriptions);

        listeners.forEach(listener -> listener.onSessionClosed(this, deleteSubscriptions));
    }

    @Override
    public void onCancel(ServiceRequest<CancelRequest, CancelResponse> serviceRequest) throws UaException {
        serviceRequest.setResponse(new CancelResponse(serviceRequest.createResponseHeader(), uint(0)));
    }
    //endregion

    public static interface LifecycleListener {
        void onSessionClosed(Session session, boolean subscriptionsDeleted);
    }
}
