/*
 * digitalpetri OPC-UA SDK
 *
 * Copyright (C) 2015 Kevin Herron
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.digitalpetri.opcua.sdk.core.events;

import java.util.Optional;

import com.digitalpetri.opcua.sdk.core.nodes.Node;
import com.digitalpetri.opcua.stack.core.types.builtin.ByteString;
import com.digitalpetri.opcua.stack.core.types.builtin.DataValue;
import com.digitalpetri.opcua.stack.core.types.builtin.DateTime;
import com.digitalpetri.opcua.stack.core.types.builtin.LocalizedText;
import com.digitalpetri.opcua.stack.core.types.builtin.NodeId;
import com.digitalpetri.opcua.stack.core.types.builtin.QualifiedName;
import com.digitalpetri.opcua.stack.core.types.builtin.unsigned.UShort;
import com.digitalpetri.opcua.stack.core.types.structured.TimeZoneDataType;

public interface BaseEventType {

    /**
     * An EventId is generated by the Server to uniquely identify a particular Event. The Server is responsible to
     * ensure that each Event has its unique EventId.
     *
     * @return an {@link ByteString} that uniquely identifies this Event.
     */
    ByteString getEventId();

    /**
     * @return the {@link NodeId} of this type of Event.
     */
    NodeId getEventType();

    /**
     * The Node that the Event originated from. If the Event is not specific to a Node the NodeId is set to null.
     *
     * @return the {@link Node} that the Event originated from.
     */
    NodeId getSourceNode();

    /**
     * A description of the source of the Event. This could be the DisplayName of the Event source, if the Event is
     * specific to a Node, or some server-specific notation.
     *
     * @return a description of the source of the Event.
     */
    String getSourceName();

    /**
     * The time the Event occurred. This value is set as close to the event generator as possible. It often comes from
     * the underlying system or device.
     *
     * @return the time the Event occurred.
     */
    DateTime getTime();

    /**
     * Provides the time the OPC UA Server received the Event from the underlying device of another Server.
     *
     * @return the time the OPC UA Server received the Event from the underlying device of another Server.
     */
    DateTime getReceiveTime();

    /**
     * {@link TimeZoneDataType} is a structure containing the Offset and the DaylightSavingInOffset flag. The Offset
     * specifies the time difference (in minutes) between the Time Property and the time at the location in which the
     * event was issued.
     * <p>
     * If DaylightSavingInOffset is TRUE, then Standard/Daylight savings time (DST) at the originating location is in
     * effect and Offset includes the DST correction.
     * <p>
     * If FALSE then the Offset does not include DST correction and DST may or may not have been in effect.
     *
     * @return {@link TimeZoneDataType}, a structure containing the Offset and the DaylightSavingInOffset flag.
     */
    Optional<TimeZoneDataType> getLocalTime();

    /**
     * A human-readable and localizable text description of the Event. The Server may return any appropriate text to
     * describe the Event. A null string is not a valid value; if the Server does not have a description, it shall
     * return the string part of the BrowseName of the Node associated with the Event.
     *
     * @return a human-readable and localizable text description of the Event.
     */
    LocalizedText getMessage();

    /**
     * An indication of the urgency of the Event. This is also commonly called “priority”. Values will range from 1 to
     * 1000, with 1 being the lowest severity and 1000 being the highest. Typically, a severity of 1 would indicate an
     * Event which is informational in nature, while a value of 1000 would indicate an Event of catastrophic nature,
     * which could potentially result in severe financial loss or loss of life.
     *
     * @return an indication of the urgency of the Event.
     */
    UShort getSeverity();

    /**
     * Get the value of a property by name.
     *
     * @param propertyName the {@link QualifiedName} of the property.
     * @return the {@link DataValue} of the property indicated by {@code propertyName}.
     */
    DataValue getProperty(QualifiedName propertyName);

}
