/*
 * Decompiled with CFR 0.152.
 */
package com.diffplug.spotless.maven.incremental;

import com.diffplug.common.annotations.VisibleForTesting;
import com.diffplug.spotless.maven.incremental.FileIndexConfig;
import com.diffplug.spotless.maven.incremental.PluginFingerprint;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.util.Map;
import java.util.TreeMap;
import javax.annotation.Nullable;
import org.apache.maven.plugin.logging.Log;

class FileIndex {
    private static final String SEPARATOR = " ";
    private final Path indexFile;
    private final PluginFingerprint pluginFingerprint;
    private final Map<Path, Instant> fileToLastModifiedTime;
    private final Path projectDir;
    private boolean modified;

    private FileIndex(Path indexFile, PluginFingerprint pluginFingerprint, Map<Path, Instant> fileToLastModifiedTime, Path projectDir, boolean needsRewrite) {
        this.indexFile = indexFile;
        this.pluginFingerprint = pluginFingerprint;
        this.fileToLastModifiedTime = fileToLastModifiedTime;
        this.projectDir = projectDir;
        this.modified = needsRewrite;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static FileIndex read(FileIndexConfig config, Log log) {
        Path indexFile = config.getIndexFile();
        if (Files.notExists(indexFile, new LinkOption[0])) {
            log.info((CharSequence)"Index file does not exist. Fallback to an empty index");
            return FileIndex.emptyIndexFallback(config);
        }
        try (BufferedReader reader = Files.newBufferedReader(indexFile, StandardCharsets.UTF_8);){
            PluginFingerprint storedFingerprint;
            String firstLine = reader.readLine();
            if (firstLine == null) {
                log.info((CharSequence)"Index file is empty. Fallback to an empty index");
                FileIndex fileIndex = FileIndex.emptyIndexFallback(config);
                return fileIndex;
            }
            PluginFingerprint computedFingerprint = config.getPluginFingerprint();
            if (!computedFingerprint.equals(storedFingerprint = PluginFingerprint.from(firstLine))) {
                log.info((CharSequence)"Index file corresponds to a different configuration of the plugin. Either the plugin version or its configuration has changed. Fallback to an empty index");
                FileIndex fileIndex = FileIndex.emptyIndexFallback(config);
                return fileIndex;
            }
            Content content = FileIndex.readIndexContent(reader, config.getProjectDir(), log);
            FileIndex fileIndex = new FileIndex(indexFile, computedFingerprint, content.fileToLastModifiedTime, config.getProjectDir(), content.needsRewrite);
            return fileIndex;
        }
        catch (IOException e) {
            log.warn((CharSequence)"Error reading the index file. Fallback to an empty index", (Throwable)e);
            return FileIndex.emptyIndexFallback(config);
        }
    }

    static void delete(FileIndexConfig config, Log log) {
        Path indexFile = config.getIndexFile();
        boolean deleted = false;
        try {
            deleted = Files.deleteIfExists(indexFile);
        }
        catch (IOException e) {
            log.warn((CharSequence)("Unable to delete the index file: " + String.valueOf(indexFile)), (Throwable)e);
        }
        if (deleted) {
            log.info((CharSequence)("Deleted the index file: " + String.valueOf(indexFile)));
        }
    }

    @Nullable
    Instant getLastModifiedTime(Path file) {
        if (!file.startsWith(this.projectDir)) {
            return null;
        }
        Path relativeFile = this.projectDir.relativize(file);
        return this.fileToLastModifiedTime.get(relativeFile);
    }

    void setLastModifiedTime(Path file, Instant time) {
        Path relativeFile = this.projectDir.relativize(file);
        this.fileToLastModifiedTime.put(relativeFile, time);
        this.modified = true;
    }

    @VisibleForTesting
    int size() {
        return this.fileToLastModifiedTime.size();
    }

    void write() {
        if (!this.modified) {
            return;
        }
        this.ensureParentDirExists();
        try (PrintWriter writer = new PrintWriter(Files.newBufferedWriter(this.indexFile, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING));){
            writer.println(this.pluginFingerprint.value());
            for (Map.Entry<Path, Instant> entry : this.fileToLastModifiedTime.entrySet()) {
                writer.println(String.valueOf(entry.getKey()) + SEPARATOR + String.valueOf(entry.getValue()));
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to write the index", e);
        }
    }

    private void ensureParentDirExists() {
        Path parentDir = this.indexFile.getParent();
        if (parentDir == null) {
            throw new IllegalStateException("Index file does not have a parent dir: " + String.valueOf(this.indexFile));
        }
        try {
            if (Files.exists(parentDir, LinkOption.NOFOLLOW_LINKS)) {
                Path realPath = parentDir.toRealPath(new LinkOption[0]);
                if (!Files.exists(realPath, new LinkOption[0])) {
                    Files.createDirectories(realPath, new FileAttribute[0]);
                }
            } else {
                Files.createDirectories(parentDir, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to create parent directory for the index file: " + String.valueOf(this.indexFile), e);
        }
    }

    private static Content readIndexContent(BufferedReader reader, Path projectDir, Log log) throws IOException {
        String line;
        TreeMap<Path, Instant> fileToLastModifiedTime = new TreeMap<Path, Instant>();
        boolean needsRewrite = false;
        while ((line = reader.readLine()) != null) {
            int separatorIndex = line.lastIndexOf(SEPARATOR);
            if (separatorIndex == -1) {
                throw new IOException("Incorrect index file. No separator found in '" + line + "'");
            }
            Path relativeFile = Paths.get(line.substring(0, separatorIndex), new String[0]);
            Path absoluteFile = projectDir.resolve(relativeFile);
            if (Files.notExists(absoluteFile, new LinkOption[0])) {
                log.info((CharSequence)("File stored in the index does not exist: " + String.valueOf(relativeFile)));
                needsRewrite = true;
                continue;
            }
            Instant lastModifiedTime = FileIndex.parseLastModifiedTime(line, separatorIndex);
            fileToLastModifiedTime.put(relativeFile, lastModifiedTime);
        }
        return new Content(fileToLastModifiedTime, needsRewrite);
    }

    private static Instant parseLastModifiedTime(String line, int separatorIndex) throws IOException {
        try {
            return Instant.parse(line.substring(separatorIndex + 1));
        }
        catch (DateTimeParseException e) {
            throw new IOException("Incorrect index file. Unable to parse last modified time from '" + line + "'", e);
        }
    }

    private static FileIndex emptyIndexFallback(FileIndexConfig config) {
        return new FileIndex(config.getIndexFile(), config.getPluginFingerprint(), new TreeMap<Path, Instant>(), config.getProjectDir(), true);
    }

    private static class Content {
        final Map<Path, Instant> fileToLastModifiedTime;
        final boolean needsRewrite;

        Content(Map<Path, Instant> fileToLastModifiedTime, boolean needsRewrite) {
            this.fileToLastModifiedTime = fileToLastModifiedTime;
            this.needsRewrite = needsRewrite;
        }
    }
}

