package com.devicenative.dna.utils;

import android.content.ComponentName;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.LauncherApps;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Process;
import android.os.UserHandle;

import org.json.JSONArray;
import org.json.JSONException;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DNAUtils {

    public static List<String> convertJSONArrayToStrings(JSONArray jsonArray) throws JSONException {
        List<String> strings = new ArrayList<>();
        for (int i = 0; i < jsonArray.length(); i++) {
            strings.add(jsonArray.getString(i));
        }
        return strings;
    }

    private static final ConcurrentHashMap<String, Integer> userHandleCache = new ConcurrentHashMap<>();
    private static final Pattern USER_HANDLE_PATTERN = Pattern.compile("\\{(\\d+)\\}");

    public static int readPidFromUserHandleString(String userHandle) {
        if (userHandle == null || userHandle.isEmpty()) {
            return -1;
        }

        // Check cache first
        Integer cached = userHandleCache.get(userHandle);
        if (cached != null) {
            return cached;
        }

        // Parse and cache result
        Matcher m = USER_HANDLE_PATTERN.matcher(userHandle);
        if (m.find()) {
            int userId = Integer.parseInt(m.group(1));
            userHandleCache.put(userHandle, userId);
            return userId;
        }

        return -1;
    }

    public static UserHandle getUserHandleFromFormattedString(String formattedUserIdString) {
        try {
            // Extract user ID from "UserHandle{0}" format
            if (formattedUserIdString.startsWith("UserHandle{") && formattedUserIdString.endsWith("}")) {
                String userIdNumStr = formattedUserIdString.substring(11, formattedUserIdString.length() - 1);
                int userId = Integer.parseInt(userIdNumStr);
                // Use reflection to create UserHandle since UserHandle.of() might not be available on all API levels
                Class<?> userHandleClass = Class.forName("android.os.UserHandle");
                java.lang.reflect.Method ofMethod = userHandleClass.getMethod("of", int.class);
                return (UserHandle) ofMethod.invoke(null, userId);
            }
        } catch (Exception e) {
            DNALogger.e("DNAPackageChangedReceiver: Error creating UserHandle: " + e.getMessage());
        }
        return Process.myUserHandle();
    }

    // Helper method to format userId string to database format (UserHandle{0})
    public static String formatUserIdString(String userIdString) {
        if (userIdString == null || userIdString.isEmpty()) {
            // Return current user handle format
            UserHandle currentUserHandle = Process.myUserHandle();
            return currentUserHandle.toString();
        }

        try {
            // Parse user ID from string format like "UserHandle{0}" or "{0}"
            String userIdNumStr = null;

            if (userIdString.startsWith("UserHandle{") && userIdString.endsWith("}")) {
                // Already in correct format: "UserHandle{0}"
                return userIdString;
            } else if (userIdString.startsWith("{") && userIdString.endsWith("}")) {
                // Format: "{0}" - extract number and format as "UserHandle{0}"
                userIdNumStr = userIdString.substring(1, userIdString.length() - 1);
                return "UserHandle{" + userIdNumStr + "}";
            } else {
                // Try to parse as plain number
                int userId = Integer.parseInt(userIdString);
                return "UserHandle{" + userId + "}";
            }
        } catch (Exception e) {
            DNALogger.e("DNAPackageChangedReceiver: Error formatting user ID: " + e.getMessage());
            // Fallback to current user
            UserHandle currentUserHandle = Process.myUserHandle();
            return currentUserHandle.toString();
        }
    }

    public static boolean isArchived(ApplicationInfo info) {
        if (Build.VERSION.SDK_INT < 34) return false;
        try {
            Field f = ApplicationInfo.class.getField("isArchived");
            return f.getBoolean(info);
        } catch (NoSuchFieldException | IllegalAccessException ignored) {
            // Fallback for future SDKs where the field is promoted/open.
            int flag;
            try {
                Field flagF = ApplicationInfo.class.getField("FLAG_ARCHIVED");
                flag = flagF.getInt(null);
                return (info.flags & flag) != 0;
            } catch (Exception e) {
                return false;
            }
        }
    }

    public static boolean isSuspended(ApplicationInfo info) {
        // Fast path when we have the flag at compile-time
        try {
            return (info.flags & ApplicationInfo.FLAG_SUSPENDED) != 0;
        } catch (NoSuchFieldError e) {
        }

        // Reflection fallback keeps the code compiling even if FLAG_SUSPENDED
        // isn’t present in compile-time stubs.
        try {
            Field f = ApplicationInfo.class.getField("FLAG_SUSPENDED");
            int FLAG_SUSPENDED = f.getInt(null);
            return (info.flags & FLAG_SUSPENDED) != 0;
        } catch (ReflectiveOperationException ignored) {
            return false;   // Unknown flag on this build setup
        }
    }

    /**
     * Helper method to safely extract activity name from component name string.
     * Handles formats: 
     * 1. "packageName/activityName#id" - extracts activityName
     * 2. "ComponentInfo{packageName/activityName}" - extracts activityName
     * If not in these formats, returns the original componentName.
     * @param componentName The component name to parse
     * @return The extracted activity name or original componentName if not in expected format
     */
    public static String extractActivityName(String componentName) {
        if (componentName == null || componentName.isEmpty()) {
            return componentName;
        }
        
        String workingString = componentName;
        
        // Handle ComponentInfo{...} format
        if (componentName.startsWith("ComponentInfo{") && componentName.endsWith("}")) {
            workingString = componentName.substring(14, componentName.length() - 1);
        }
        
        // Check if it's in the format "packageName/activityName#id" or "packageName/activityName"
        int slashIndex = workingString.indexOf('/');
        if (slashIndex == -1) {
            // No slash found, return as is
            return componentName;
        }
        
        // Extract the part after the slash
        String afterSlash = workingString.substring(slashIndex + 1);
        
        // Check if there's a # and remove everything after it
        int hashIndex = afterSlash.indexOf('#');
        if (hashIndex != -1) {
            afterSlash = afterSlash.substring(0, hashIndex);
        }
        
        return afterSlash;
    }

}
