package com.devicenative.dna.network;

import android.os.NetworkOnMainThreadException;
import android.util.Log;

import com.devicenative.dna.utils.DNAError;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.OutputStreamWriter;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.util.zip.GZIPInputStream;

import javax.net.ssl.HttpsURLConnection;

public class DNANetworkInterfaceHTTPConnection extends DNANetworkInterface {

    @Override
    public DNANetworkRawResponse doGet(String url) throws DNANetworkException {
        return doGet(url, 0);
    }

    @Override
    public DNANetworkRawResponse doPost(String url, JSONObject payload) throws DNANetworkException {
        return doPost(url, payload, 0);
    }


    ///-------------- private methods to implement GET / POST using HttpURLConnection ---------------//
    private DNANetworkRawResponse doGet(String url, int retryNumber) throws DNANetworkException {
        HttpsURLConnection connection = null;
        DNAPreferences prefHelper = DNAPreferences.getInstance();
        try {
            String appendKey = url.contains("?") ? "&" : "?";
            String modifiedUrl = url + appendKey + RETRY_NUMBER + "=" + retryNumber;
            URL urlObject = new URL(modifiedUrl);
            connection = (HttpsURLConnection) urlObject.openConnection();
            connection.setConnectTimeout(30000);
            connection.setReadTimeout(30000);
            connection.setRequestProperty("Accept-Encoding", "gzip");

            int responseCode = connection.getResponseCode();
            if (responseCode >= 500 && retryNumber < prefHelper.getRetryCount()) {
                try {
                    Thread.sleep(prefHelper.getRetryInterval());
                } catch (InterruptedException e) {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(e));
                }
                retryNumber++;
                return doGet(url, retryNumber);
            } else {
                DNANetworkRawResponse result;
                try {
                    String encoding = connection.getHeaderField("Content-Encoding");
                    if (responseCode != HttpsURLConnection.HTTP_OK && connection.getErrorStream() != null) {
                        result = new DNANetworkRawResponse(getResponseString(encoding, connection.getErrorStream()), responseCode);
                    } else {
                        result = new DNANetworkRawResponse(getResponseString(encoding, connection.getInputStream()), responseCode);
                    }
                } catch (FileNotFoundException ex) {
                    // In case of Resource conflict getInputStream will throw FileNotFoundException. Handle it here in order to send the right status code
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: A resource conflict occurred with this request " + url);
                    result = new DNANetworkRawResponse(null, responseCode);
                }
                return result;
            }
        } catch (SocketException ex) {
            DNALogger.e("DNANetworkInterfaceHTTPConnection: Http connect exception: " + ex.getMessage());
            throw new DNANetworkException(DNAError.ERR_DNA_NO_CONNECTIVITY, ex.getMessage());

        } catch (SocketTimeoutException ex) {
            // On socket  time out retry the request for retryNumber of times
            if (retryNumber < prefHelper.getRetryCount()) {
                try {
                    Thread.sleep(prefHelper.getRetryInterval());
                } catch (InterruptedException e) {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(e));
                }
                retryNumber++;
                return doGet(url, retryNumber);
            } else {
                throw new DNANetworkException(DNAError.ERR_DNA_REQ_TIMED_OUT, ex.getMessage());
            }
        } catch(InterruptedIOException ex){
            // When the thread times out before or while sending the request
            if (retryNumber < prefHelper.getRetryCount()) {
                try {
                    Thread.sleep(prefHelper.getRetryInterval());
                } catch (InterruptedException e) {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(e));
                }
                retryNumber++;
                return doGet(url, retryNumber);
            } else {
                throw new DNANetworkException(DNAError.ERR_DNA_TASK_TIMEOUT, ex.getMessage());
            }
        } catch (IOException ex) {
            DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(ex));
            if (retryNumber < prefHelper.getRetryCount()) {
                try {
                    Thread.sleep(prefHelper.getRetryInterval());
                } catch (InterruptedException e) {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(e));
                }
                retryNumber++;
                return doGet(url, retryNumber);
            } else {
                throw new DNANetworkException(DNAError.ERR_DNA_NO_CONNECTIVITY, ex.getMessage());
            }
        } catch (Exception ex) {
            DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(ex));
            if (ex instanceof NetworkOnMainThreadException) {
                DNALogger.i("DNANetworkInterfaceHTTPConnection: Cannot make network request on main thread: " + Log.getStackTraceString(ex));
                throw new DNANetworkException((DNAError.ERR_NETWORK_ON_MAIN), ex.getMessage());
            }
            throw new DNANetworkException(DNAError.ERR_OTHER, ex.getMessage());
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }


    private DNANetworkRawResponse doPost(String url, JSONObject payload, int retryNumber) throws DNANetworkException {
        HttpsURLConnection connection = null;
        DNAPreferences prefHelper = DNAPreferences.getInstance();

        try {
            payload.put(RETRY_NUMBER, retryNumber);
        } catch (JSONException e) {
            DNALogger.e("DNANetworkInterfaceHTTPConnection: " + e.getMessage());
        }
        try {
            URL urlObject = new URL(url);
            connection = (HttpsURLConnection) urlObject.openConnection();
            connection.setConnectTimeout(30000);
            connection.setReadTimeout(30000);
            connection.setDoInput(true);
            connection.setDoOutput(true);
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setRequestProperty("Accept", "application/json");
            connection.setRequestProperty("Accept-Encoding", "gzip");
            connection.setRequestProperty("Content-Length", String.valueOf(payload.toString().getBytes().length));

            connection.setRequestMethod("POST");

            OutputStreamWriter outputStreamWriter = new OutputStreamWriter(connection.getOutputStream());

            outputStreamWriter.write(payload.toString());
            outputStreamWriter.flush();
            outputStreamWriter.close();

            int responseCode = connection.getResponseCode();
            
            // Handle redirects manually for 3XX status codes
            if (responseCode >= 300 && responseCode < 400) {
                String redirectUrl = connection.getHeaderField("Location");
                if (redirectUrl != null && !redirectUrl.trim().isEmpty()) {
                    // Handle relative URLs by resolving them against the original URL
                    if (redirectUrl.startsWith("/")) {
                        try {
                            URL originalUrl = new URL(url);
                            redirectUrl = originalUrl.getProtocol() + "://" + originalUrl.getHost() + 
                                         (originalUrl.getPort() != -1 ? ":" + originalUrl.getPort() : "") + redirectUrl;
                        } catch (Exception e) {
                            DNALogger.e("DNANetworkInterfaceHTTPConnection: Error resolving relative redirect URL: " + e.getMessage());
                            return new DNANetworkRawResponse("Error resolving redirect URL", responseCode);
                        }
                    }
                    
                    DNALogger.i("DNANetworkInterfaceHTTPConnection: POST received " + responseCode + " redirect, following to: " + redirectUrl);
                    connection.disconnect();
                    
                    // Follow redirect with GET request (since redirect target is likely serving static JSON)
                    // Don't increment retry count for redirects since this is expected behavior
                    return doGet(redirectUrl, 0);
                } else {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Redirect response " + responseCode + " without valid Location header");
                    return new DNANetworkRawResponse("Redirect without Location header", responseCode);
                }
            }
            
            if (responseCode >= HttpsURLConnection.HTTP_INTERNAL_ERROR
                    && retryNumber < prefHelper.getRetryCount()) {
                try {
                    Thread.sleep(prefHelper.getRetryInterval());
                } catch (InterruptedException e) {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(e));
                }
                retryNumber++;
                return doPost(url, payload, retryNumber);
            } else {
                DNANetworkRawResponse result;
                try {
                    String encoding = connection.getHeaderField("Content-Encoding");
                    if (responseCode != HttpsURLConnection.HTTP_OK && connection.getErrorStream() != null) {
                        result = new DNANetworkRawResponse(getResponseString(encoding, connection.getErrorStream()), responseCode);
                    } else {
                        result = new DNANetworkRawResponse(getResponseString(encoding, connection.getInputStream()), responseCode);
                    }

                } catch (FileNotFoundException ex) {
                    // In case of Resource conflict getInputStream will throw FileNotFoundException. Handle it here in order to send the right status code
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: A resource conflict occurred with this request " + url);
                    result = new DNANetworkRawResponse(null, responseCode);
                }

                return result;
            }


        } catch (SocketTimeoutException ex) {
            DNALogger.i("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(ex));
            // On socket  time out retry the request for retryNumber of times
            if (retryNumber < prefHelper.getRetryCount()) {
                try {
                    Thread.sleep(prefHelper.getRetryInterval());
                } catch (InterruptedException e) {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(e));
                }
                retryNumber++;
                return doPost(url, payload, retryNumber);
            } else {
                throw new DNANetworkException(DNAError.ERR_DNA_REQ_TIMED_OUT, Log.getStackTraceString(ex));
            }
        } catch(InterruptedIOException ex){
            DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + ex);
            // When the thread times out before or while sending the request
            if (retryNumber < prefHelper.getRetryCount()) {
                try {
                    Thread.sleep(prefHelper.getRetryInterval());
                } catch (InterruptedException e) {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(e));
                }
                retryNumber++;
                return doPost(url, payload, retryNumber);
            } else {
                throw new DNANetworkException(DNAError.ERR_DNA_TASK_TIMEOUT, ex.getMessage());
            }
        }
        // Unable to resolve host/Unknown host exception
        catch (IOException ex) {
            DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(ex));
            if (retryNumber < prefHelper.getRetryCount()) {
                try {
                    Thread.sleep(prefHelper.getRetryInterval());
                }
                catch (InterruptedException e) {
                    DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(e));
                }
                retryNumber++;
                return doPost(url, payload, retryNumber);
            }
            else {
                throw new DNANetworkException(DNAError.ERR_DNA_NO_CONNECTIVITY, ex.getMessage());
            }
        } catch (Exception ex) {
            DNALogger.e("DNANetworkInterfaceHTTPConnection: Encountered exception while attempting network request: " + Log.getStackTraceString(ex));
            if (ex instanceof NetworkOnMainThreadException) {
                DNALogger.i("DNANetworkInterfaceHTTPConnection: Cannot make network request on main thread: " + Log.getStackTraceString(ex));
                throw new DNANetworkException((DNAError.ERR_NETWORK_ON_MAIN), ex.getMessage());
            }
            throw new DNANetworkException(DNAError.ERR_OTHER, ex.getMessage());
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    private String getResponseString(String encoding, InputStream inputStream) {
        String responseString = null;
        if (inputStream != null) {
            try {
                // Check for GZIP encoding and wrap stream if needed
                InputStream decodedStream = "gzip".equalsIgnoreCase(encoding)
                        ? new GZIPInputStream(inputStream)
                        : inputStream;

                BufferedReader rd = new BufferedReader(new InputStreamReader(decodedStream));
                StringBuilder sb = new StringBuilder();
                String line;
                while ((line = rd.readLine()) != null) {
                    sb.append(line);
                }
                rd.close();
                responseString = sb.toString();
            } catch (IOException e) {
                DNALogger.e("DNANetworkInterfaceHTTPConnection: " + e.getMessage());
            }
        }
        return responseString;
    }

}
