package com.devicenative.dna.datasource;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.LauncherApps;

import android.os.Process;
import android.os.UserManager;

import com.devicenative.dna.broadcast.DNAPackageChangedReceiver;
import com.devicenative.dna.datasource.loader.DNAAppLoader;
import com.devicenative.dna.db.DNADatabaseQueue;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNALauncherAppsCallback;

public class DNAAppDataSource extends DNADataSource {

    public DNAAppDataSource(Service service) {
        super(service);

        Context context = service.getApplicationContext();
        // Package installation/uninstallation events for the main
        // profile are still handled using PackageAddedRemovedHandler itself
        final UserManager manager = (UserManager) context.getSystemService(Context.USER_SERVICE);
        if (manager == null) {
            DNALogger.e("DNAAppDataSource: Unable to get user manager");
            return;
        }

        final LauncherApps launcher = (LauncherApps) context.getSystemService(Context.LAUNCHER_APPS_SERVICE);
        if (launcher == null) {
            DNALogger.e("DNAAppDataSource: Unable to get launcher service");
            return;
        }

        launcher.registerCallback(new DNALauncherAppsCallback() {
            @Override
            public void onPackageAdded(String packageName, android.os.UserHandle user) {
                handleEvent(Intent.ACTION_PACKAGE_ADDED, packageName, null, user, false);
            }

            @Override
            public void onPackageChanged(String packageName, android.os.UserHandle user) {
                handleEvent(Intent.ACTION_PACKAGE_CHANGED, packageName, null, user, false);
            }

            @Override
            public void onPackageRemoved(String packageName, android.os.UserHandle user) {
                handleEvent(Intent.ACTION_PACKAGE_REMOVED, packageName, null, user, false);
            }

            @Override
            public void onPackagesAvailable(String[] packageNames, android.os.UserHandle user, boolean replacing) {
                handleEvent(Intent.ACTION_MEDIA_MOUNTED, null, packageNames, user, replacing);
            }

            @Override
            public void onPackagesSuspended(String[] packageNames, android.os.UserHandle user) {
                handleEvent(Intent.ACTION_PACKAGES_SUSPENDED, null, packageNames, user, false);
            }

            @Override
            public void onPackagesUnsuspended(String[] packageNames, android.os.UserHandle user) {
                handleEvent(Intent.ACTION_PACKAGES_UNSUSPENDED, null, packageNames, user, false);
            }

            @Override
            public void onPackagesUnavailable(String[] packageNames, android.os.UserHandle user, boolean replacing) {
                handleEvent(Intent.ACTION_MEDIA_UNMOUNTED, null, packageNames, user, replacing);
            }

            private void handleEvent(String action, String packageName, String[] packageList, android.os.UserHandle user, boolean replacing) {
                DNADatabaseQueue.enqueue(DNAAppDataSource.this.context.get(), () -> {
                    DNAPackageChangedReceiver.handleEvent(DNAAppDataSource.this.context.get(),
                    action,
                    packageName,
                    packageList,
                    user.toString(),
                    replacing
                    );
                });
            }
        });

        IntentFilter appChangedFilter = new IntentFilter();
        appChangedFilter.addAction(Intent.ACTION_PACKAGE_ADDED);
        appChangedFilter.addAction(Intent.ACTION_PACKAGE_CHANGED);
        appChangedFilter.addAction(Intent.ACTION_PACKAGE_REMOVED);
        appChangedFilter.addDataScheme("package");
        appChangedFilter.addDataScheme("file");
        context.registerReceiver(new DNAPackageChangedReceiver(), appChangedFilter);
    }

    @Override
    public void loadData() {
        super.loadData();
        this.initialize(new DNAAppLoader(this.context.get()));
    }
}


