package com.devicenative.dna.broadcast;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.LauncherActivityInfo;
import android.content.pm.LauncherApps;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.UserHandle;

import com.devicenative.dna.DeviceNativeAds;
import com.devicenative.dna.ads.DNAAppSearchFallback;
import com.devicenative.dna.ads.DNAResultServer;
import com.devicenative.dna.db.DNAAppRecord;
import com.devicenative.dna.db.DNADatabaseInterface;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class DNAPackageChangedReceiver extends BroadcastReceiver {
    public static void handleEvent(Context ctx, String action, String packageName, String[] packageList, String userId, boolean replacing) {
        if (Intent.ACTION_PACKAGE_ADDED.equals(action)) {
            if (!replacing) {
                DNALogger.i("DNAPackageChangedReceiver added: " + packageName);
                handleCaseOfAppInstalled(ctx, packageName, userId);
            }
        } else if (Intent.ACTION_PACKAGE_REMOVED.equals(action)) {
            if (!replacing) {
                DNALogger.i("DNAPackageChangedReceiver removed: " + packageName);
                handleCaseOfAppUninstalled(ctx, packageName, userId);
            }
        } else if (Intent.ACTION_PACKAGE_CHANGED.equals(action) && packageName != null) {
            DNALogger.i("DNAPackageChangedReceiver changed: " + packageName + ", userId: " + userId);
            LauncherApps la = ctx.getSystemService(LauncherApps.class);
            if (la == null) {
                return;
            }

            UserHandle profile = DNAUtils.getUserHandleFromFormattedString(userId);

            if (!la.isPackageEnabled(packageName, profile)) {
                DNALogger.i("DNAPackageChangedReceiver: app is disabled");
                handleCaseOfAppUninstalled(ctx, packageName, userId);
                return;
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                DNAPreferences prefs = DNAPreferences.getInstance(ctx);
                boolean isPackageHidden = prefs.isPackageHidden(packageName, userId.toString());
                if (la.shouldHideFromSuggestions(packageName, profile)) {
                    prefs.setPackageHidden(packageName, userId.toString(), true);
                    DNALogger.i("DNAPackageChangedReceiver: app is hidden from suggestions");
                    DNAResultServer resultServer = DNAResultServer.getInstance(ctx);
                    resultServer.clearCache();
                    return;
                } else if (isPackageHidden) {
                    prefs.setPackageHidden(packageName, userId.toString(), false);
                    DNALogger.i("DNAPackageChangedReceiver: app is no longer hidden from suggestions");
                    DNAResultServer resultServer = DNAResultServer.getInstance(ctx);
                    resultServer.clearCache();
                }
            }

            boolean disabled = false;

            List<LauncherActivityInfo> launcherActivities = null;
            try {
                launcherActivities = la.getActivityList(packageName, profile);
            } catch (Exception e) {
                DNALogger.e("DNAPackageChangedReceiver: getActivityList failed for " + packageName + ": " + e.getMessage());
                disabled = true;
            }
            if (launcherActivities == null || launcherActivities.isEmpty()) {
                DNALogger.i("DNAPackageChangedReceiver added: no launcher activities found");
                disabled = true;
            } else {
                for (LauncherActivityInfo activityInfo : launcherActivities) {
                    ApplicationInfo appInfo = activityInfo.getApplicationInfo();
                    if (DNAUtils.isSuspended(appInfo) || DNAUtils.isArchived(appInfo)) {
                        DNALogger.i("DNAPackageChangedReceiver: app is suspended or archived");
                        disabled = true;
                        break;
                    }
                }
            }

            if (disabled) {
                DNALogger.i("DNAPackageChangedReceiver: app is disabled, so about to updated uninstalledAt state");
                handleCaseOfAppUninstalled(ctx, packageName, userId);
            } else {
                if (DNADatabaseInterface.isAppRecordUninstalled(ctx, packageName, userId)) {
                    DNALogger.i("DNAPackageChangedReceiver: app record is already uninstalled in DB, so going to set it as installed now");
                    handleCaseOfAppInstalled(ctx, packageName, userId);
                }
            }
        } else if (Intent.ACTION_PACKAGES_SUSPENDED.equals(action)) {
            // Play has archived / paused one or more apps
            if (packageList == null || packageList.length == 0) {
                return;
            }
            DNAPreferences prefs = DNAPreferences.getInstance(ctx);
            for (String p : packageList) {
                DNALogger.i("DNAPackageChangedReceiver suspended so going to set it hidden: " + p + ", userId: " + userId);
                prefs.setPackageHidden(p, userId.toString(), true);
            }
            DNAResultServer resultServer = DNAResultServer.getInstance(ctx);
            resultServer.clearCache();
        } else if (Intent.ACTION_PACKAGES_UNSUSPENDED.equals(action)) {
            if (packageList == null || packageList.length == 0) {
                return;
            }
            DNAPreferences prefs = DNAPreferences.getInstance(ctx);
            for (String p : packageList) {
                prefs.setPackageHidden(p, userId.toString(), false);
                DNALogger.i("DNAPackageChangedReceiver unsuspended so going to set it as hidden: " + p + ", userId: " + userId);
            }
            DNAResultServer resultServer = DNAResultServer.getInstance(ctx);
            resultServer.clearCache();
        }
    }

    private static void handleCaseOfAppInstalled(Context ctx, String packageName, String userId) {
        // Ensure userId is in correct format for database operations
        String dbUserId = DNAUtils.formatUserIdString(userId);
        UserHandle profile = DNAUtils.getUserHandleFromFormattedString(dbUserId);

        PackageManager pManager = ctx.getPackageManager();
        LauncherApps launcher = (LauncherApps) ctx.getSystemService(Context.LAUNCHER_APPS_SERVICE);

        List<LauncherActivityInfo> launcherActivities = null;
        try {
            launcherActivities = launcher.getActivityList(packageName, profile);
        } catch (Exception e) {
            DNALogger.e("handleCaseOfAppInstalled: getActivityList failed for " + packageName + ": " + e.getMessage());
            return;
        }

        if (launcherActivities == null || launcherActivities.isEmpty()) {
            DNALogger.i("handleCaseOfAppInstalled added: no launcher activities found");
            return;
        }

        List<DNAAppRecord> apps = new ArrayList<>();
        for (LauncherActivityInfo activityInfo : launcherActivities) {
            if (!launcher.isActivityEnabled(activityInfo.getComponentName(), profile)) {
                DNALogger.i("handleCaseOfAppInstalled: activity is disabled");
                continue;
            }

            ApplicationInfo appInfo = activityInfo.getApplicationInfo();

            DNAAppRecord app = new DNAAppRecord();
            app.packageName = appInfo.packageName;
            app.userId = profile.toString();
            if (activityInfo.getLabel() != null) {
                app.name = String.valueOf(activityInfo.getLabel());
            } else {
                app.name = appInfo.loadLabel(pManager).toString();
            }
            app.component = activityInfo.getName();
            if (activityInfo.getFirstInstallTime() > 0) {
                app.installedAt = activityInfo.getFirstInstallTime();
            } else {
                app.installedAt = System.currentTimeMillis();
            }

            app.updatedAt = System.currentTimeMillis();

            apps.add(app);
        }

        List<DNAAppRecord> existingApps = DNADatabaseInterface.getAppRecords(ctx, packageName, dbUserId);
        List<DNAAppRecord> appsToInsertOrUpdate = new ArrayList<>();

        for (DNAAppRecord app : apps) {
            boolean found = false;
            for (DNAAppRecord existingApp : existingApps) {
                if (existingApp.packageName.equals(app.packageName) &&
                        existingApp.userId.equals(app.userId) &&
                        existingApp.uninstalledAt == 0 &&
                        (existingApp.component == null || Objects.equals(existingApp.component, app.component)) &&
                        Objects.equals(existingApp.name, app.name)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                appsToInsertOrUpdate.add(app);
            }
        }

        for (DNAAppRecord existingApp : existingApps) {
            boolean found = false;
            for (DNAAppRecord app : apps) {
                if (existingApp.packageName.equals(app.packageName) && Objects.equals(existingApp.userId, app.userId)) {
                    found = true;
                    break;
                }
            }
            if (!found & existingApp.uninstalledAt == 0) {
                existingApp.uninstalledAt = System.currentTimeMillis();
                appsToInsertOrUpdate.add(existingApp);
            }
        }

        DNADatabaseInterface.insertOrUpdateAppRecords(ctx, appsToInsertOrUpdate);

        // new app installed so let's see if there's a good ad for it
        DNAResultServer resultServer = DNAResultServer.getInstance(ctx);
        resultServer.prefetchSuggestions();

        DNAAppSearchFallback.clearCache();
    }

    private static void handleCaseOfAppUninstalled(Context ctx, String packageName, String userId) {
        // Ensure userId is in correct format for database operations
        String dbUserId = DNAUtils.formatUserIdString(userId);

        List<DNAAppRecord> appRecords = DNADatabaseInterface.getAppRecords(ctx, packageName, dbUserId);
        if (appRecords == null || appRecords.isEmpty()) {
            DNALogger.i("handleCaseOfAppUninstalled removed: no app record found");
            return;
        }

        for (DNAAppRecord appRecord : appRecords) {
            DNALogger.i("handleCaseOfAppUninstalled removed: " + appRecord.packageName + ", " + appRecord.userId + ", " + appRecord.component);
            appRecord.uninstalledAt = System.currentTimeMillis();
            DNADatabaseInterface.insertOrUpdateAppRecord(ctx, appRecord);
        }
        DNADatabaseInterface.removeShortcuts(ctx, packageName);

        DNAResultServer resultServer = DNAResultServer.getInstance(ctx);
        resultServer.prefetchSuggestions();

        DNAAppSearchFallback.clearCache();
    }

    /**
     * Helper method to trigger immediate data reload by firing the wake broadcast intent.
     * This allows for immediate response to app changes instead of waiting for the next scheduled wake.
     */
    public static void triggerImmediateReload(Context ctx) {
        DeviceNativeAds.getInstance(ctx).triggerDataRefresh();
    }

    @Override
    public void onReceive(Context ctx, Intent intent) {
        DNALogger.i("DNAPackageChangedReceiver: received broadcast to wake up callbacks: " + intent.getAction());
    }

}
