package com.devicenative.dna.ads;

import com.devicenative.dna.DNAResultItem;

import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

public class DNAdCacheManager {
    private final long blackoutTimeLimitMs;
    private final long adCacheTimeLimitMs;
    private final ConcurrentHashMap<String, Long> packageTimestamps = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, CacheItem> displayCache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, CacheItem> searchCache = new ConcurrentHashMap<>();

    public DNAdCacheManager(long blackoutTimeLimitMs, long adCacheTimeLimitMs) {
        this.blackoutTimeLimitMs = blackoutTimeLimitMs;
        this.adCacheTimeLimitMs = adCacheTimeLimitMs;
    }

    public synchronized void setPackageImpressionTimestamp(String packageName) {
        cleanupExpiredPackageImpressionTimestamps();
        packageTimestamps.put(packageName, System.currentTimeMillis());
    }

    public boolean getPackageImpressionTimestampExpired(String packageName) {
        Long timestamp = packageTimestamps.get(packageName);
        return timestamp == null || (System.currentTimeMillis() - timestamp) > blackoutTimeLimitMs;
    }

    public synchronized void setDisplayCache(List<DNAResultItem> results) {
        setDispCache("org-display", results);
    }

    public synchronized void setDisplayLinkCache(List<DNAResultItem> results) {
        setDispCache("org-display-links", results);
    }

    public synchronized void setHotAppsCache(List<DNAResultItem> results) {
        setDispCache("hot-apps", results);
    }

    public synchronized void setAdsDisplayCache(List<DNAResultItem> results) {
        setDispCache("ads-display", results);
    }

    private void setDispCache(String key, List<DNAResultItem> results) {
        cleanupExpiredItems(displayCache);
        CacheItem item = new CacheItem(results, adCacheTimeLimitMs);
        displayCache.put(key, item);
    }

    public List<DNAResultItem> getDisplayCache() {
        return getDispCache("org-display");
    }

    public List<DNAResultItem> getDisplayLinksCache() {
        return getDispCache("org-display-links");
    }

    public List<DNAResultItem> getHotAppsCache() {
        return getDispCache("hot-apps");
    }

    public List<DNAResultItem> getAdsDisplayCache() {
        return getDispCache("ads-display");
    }

    private List<DNAResultItem> getDispCache(String key) {
        CacheItem item = displayCache.get(key);
        if (item != null && item.isExpired()) {
            displayCache.remove(key);
            return null;
        }
        if (item != null && !item.isExpired()) {
            return item.getResults();
        }
        return null; // Return null if expired or not found
    }

    public synchronized void clearSearchCache() {
        searchCache.clear();
    }

    public synchronized void setSearchCache(String query, List<DNAResultItem> results) {
        cleanupExpiredItems(searchCache);
        CacheItem item = new CacheItem(results, adCacheTimeLimitMs);
        searchCache.put(query, item);
    }

    public List<DNAResultItem> getSearchCache(String query) {
        CacheItem item = searchCache.get(query.toLowerCase());
        if (item != null && item.isExpired()) {
            searchCache.remove(query);
            return null;
        }
        if (item != null && !item.isExpired()) {
            return item.getResults();
        }
        return null;
    }

    private void cleanupExpiredPackageImpressionTimestamps() {
        long now = System.currentTimeMillis();
        packageTimestamps.entrySet().removeIf(entry -> now - entry.getValue() > blackoutTimeLimitMs);
    }

    private void cleanupExpiredItems(ConcurrentHashMap<String, CacheItem> cache) {
        cache.entrySet().removeIf(entry -> entry.getValue().isExpired());
    }

    // Inner class to hold cache items
    public static class CacheItem {
        private final long cacheTimeLimitMs;
        private final long lastLoadTime;
        private final List<DNAResultItem> results;

        public CacheItem(List<DNAResultItem> results, long cacheTimeLimitMs) {
            this.cacheTimeLimitMs = cacheTimeLimitMs;
            this.lastLoadTime = System.currentTimeMillis();
            this.results = results;
        }

        public boolean isExpired() {
            return System.currentTimeMillis() - lastLoadTime > this.cacheTimeLimitMs;
        }

        public List<DNAResultItem> getResults() {
            return results;
        }
    }
}
