package com.devicenative.dna.ads;

import android.content.Context;
import android.content.Intent;
import android.os.UserHandle;
import android.os.Process;

import com.devicenative.dna.DNAResultItem;
import com.devicenative.dna.DeviceNativeClickHandler;
import com.devicenative.dna.db.DNADatabaseInterface;
import com.devicenative.dna.db.DNADatabaseQueue;
import com.devicenative.dna.db.DNAResultEventRecord;
import com.devicenative.dna.network.DNAStatSyncExecutor;
import com.devicenative.dna.utils.DNAConstants;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;
import com.devicenative.dna.ads.DNAAppSearchFallback;
import com.devicenative.dna.db.DNAAppRecord;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class DNAResultServer {
    private static final String[] ALPHABET_QUERIES = {
            "a", "b", "c", "d", "e", "f", "g", "h", "i", "j",
            "k", "l", "m", "n", "o", "p", "q", "r", "s", "t",
            "u", "v", "w", "x", "y", "z"
    };

    private static DNAResultServer instance_;

    private Context context_;

    private boolean isShutdown_ = true;

    private DNAPreferences dnaPreferences_;

    private DNAResultEngine resultEngine_;

    private DNAdCacheManager adCacheManager_;

    private DNAdLinkHandler adLinkHandler_;

    private DNAStatSyncExecutor statSyncExecutor_;

    private ExecutorService executorService_;

    private ExecutorService clickExecutorService_;

    private ScheduledExecutorService scheduledExecutorService_;

    private Future<?> futurePrefetchTask_;
    private Future<?> searchDebounceTask_;
    private Future<?> statsSyncTask_;
    private Future<?> usageSyncTask_;

    private DNAResultServer(Context context) {
        init(context);
    }

    public void init(Context context) {
        this.context_ = context;
        this.dnaPreferences_ = DNAPreferences.getInstance(context);

        this.statSyncExecutor_ = new DNAStatSyncExecutor(context);
        this.adLinkHandler_ = new DNAdLinkHandler(dnaPreferences_.getAdCacheTimeout());
        this.adCacheManager_ = new DNAdCacheManager(dnaPreferences_.getImpressionBlackoutWindow(), dnaPreferences_.getAdCacheTimeout());
        this.executorService_ = Executors.newFixedThreadPool(1);
        this.resultEngine_ = new DNAResultEngine(context, dnaPreferences_, executorService_);
        this.clickExecutorService_ = new ThreadPoolExecutor(
                1,
                1,
                0L,
                TimeUnit.MILLISECONDS,
                new SynchronousQueue<Runnable>(),
                new ThreadFactory() {
                    @Override
                    public Thread newThread(Runnable r) {
                        Thread thread = new Thread(r);
                        thread.setPriority(Thread.MAX_PRIORITY); // Set to maximum priority
                        return thread;
                    }
                }
        );
        this.scheduledExecutorService_ = Executors.newScheduledThreadPool(1);
        this.isShutdown_ = false;
    }

    public static DNAResultServer getInstance(Context context) {
        if (instance_ == null || instance_.isShutdown_) {
            instance_ = new DNAResultServer(context);
        }
        return instance_;
    }

    synchronized private void validateIsActiveAndReinitialize() {
        if (isShutdown_) {
            this.dnaPreferences_ = DNAPreferences.getInstance(context_);
            this.adLinkHandler_ = new DNAdLinkHandler(dnaPreferences_.getAdCacheTimeout());
            this.adCacheManager_ = new DNAdCacheManager(dnaPreferences_.getImpressionBlackoutWindow(), dnaPreferences_.getAdCacheTimeout());
            this.executorService_ = Executors.newFixedThreadPool(1);
            this.resultEngine_ = new DNAResultEngine(context_, dnaPreferences_, executorService_);
            this.clickExecutorService_ = new ThreadPoolExecutor(
                    1,
                    1,
                    0L,
                    TimeUnit.MILLISECONDS,
                    new SynchronousQueue<Runnable>(),
                    new ThreadFactory() {
                        @Override
                        public Thread newThread(Runnable r) {
                            Thread thread = new Thread(r);
                            thread.setPriority(Thread.MAX_PRIORITY); // Set to maximum priority
                            return thread;
                        }
                    }
            );
            this.scheduledExecutorService_ = Executors.newScheduledThreadPool(1);
            this.isShutdown_ = false;
        }
    }

    public static void shutDown() {
        DNAAppSearchFallback.clearCache();
        if (instance_ != null) {
            instance_.dnaPreferences_ = null;
            instance_.adCacheManager_ = null;
            instance_.adLinkHandler_ = null;
            instance_.executorService_.shutdown();
            instance_.executorService_ = null;
            instance_.clickExecutorService_.shutdown();
            instance_.clickExecutorService_ = null;
            instance_.scheduledExecutorService_.shutdown();
            instance_.scheduledExecutorService_ = null;
            instance_.resultEngine_ = null;
            instance_.futurePrefetchTask_ = null;
            instance_.searchDebounceTask_ = null;
            instance_.statsSyncTask_ = null;
            instance_.isShutdown_ = true;
        }
        // Reset all of the statics.
        instance_ = null;
    }

    private List<DNAResultItem> retrieveCachedDisplayAds(int count) {
        DNALogger.i("retrieveCachedDisplayAds: retrieving cached display ads");

        List<DNAResultItem> ads = adCacheManager_.getAdsDisplayCache();
        if (ads != null && !ads.isEmpty() && ads.size() >= count) {
            DNALogger.i("retrieveCachedDisplayAds: retrieved " + ads.size() + " cached display ads");
            // return only the count of ads
            return ads.subList(0, count);
        }
        return null;
    }

    private List<DNAResultItem> retrieveCachedSuggestions(int count) {
        DNALogger.i("retrieveCachedSuggestions: retrieving cached recommendations");
        List<DNAResultItem> results = adCacheManager_.getDisplayCache();
        if (results != null && !results.isEmpty() && results.size() >= count) {
            DNALogger.i("retrieveCachedSuggestions: retrieved " + results.size() + " cached recommendations");
            // return only the count of ads
            return results.subList(0, count);
        }
        return null;
    }

    private List<DNAResultItem> retrieveCachedHotApps(int count) {
        DNALogger.i("retrieveCachedHotApps: retrieving cached hot apps");
        List<DNAResultItem> results = adCacheManager_.getHotAppsCache();
        if (results != null && !results.isEmpty() && results.size() >= count) {
            DNALogger.i("retrieveCachedHotApps: retrieved " + results.size() + " cached hot apps");
            // return only the count of ads
            return results.subList(0, count);
        }
        return null;
    }

    private List<DNAResultItem> retrieveCachedLinkSuggestions(int count) {
        DNALogger.i("retrieveCachedSuggestions: retrieving cached link recommendations");
        List<DNAResultItem> results = adCacheManager_.getDisplayLinksCache();
        if (results != null && !results.isEmpty() && results.size() >= count) {
            DNALogger.i("retrieveCachedSuggestions: retrieved " + results.size() + " cached recommendations");
            // return only the count of ads
            return results.subList(0, count);
        }
        return null;
    }

    public void clearCache() {
        if (adCacheManager_ == null) {
            return;
        }
        adCacheManager_.setDisplayCache(null);
        adCacheManager_.clearSearchCache();
    }

    public void prefetchSuggestions() {
        Runnable prefetchRunnable = () -> {
            DNALogger.i("prefetchSuggestions: prefetching suggestions");
            validateIsActiveAndReinitialize();

            adCacheManager_.setDisplayCache(null);
            DNAResultEngine resultEngine = new DNAResultEngine(context_, dnaPreferences_, executorService_);
            ArrayList<DNAResultItem> results = resultEngine.fetchResultsGeneric(DNAConstants.SOURCE_SUGGESTED_APPS, null, true, false, false,null, 4);
            adCacheManager_.setDisplayCache(results);

            adCacheManager_.clearSearchCache();
            for (String query : ALPHABET_QUERIES) {
                results = resultEngine_.fetchResultsGeneric(DNAConstants.SOURCE_SEARCH, null, true, false, false, query, -1);
                adCacheManager_.setSearchCache(query, results);
            }
        };
        if (futurePrefetchTask_ != null && !futurePrefetchTask_.isDone()) {
            futurePrefetchTask_.cancel(true);
        }
        futurePrefetchTask_ = scheduledExecutorService_.schedule(prefetchRunnable, 1000, TimeUnit.MILLISECONDS);
    }

    public List<DNAResultItem> fetchOrganicAppSuggestions(int count, String placementTag) {
        DNALogger.i("fetchOrganicAppSuggestions: fetching suggestions with count: " + count);
        long currentTime = System.currentTimeMillis();

        ArrayList<DNAStatsLogger.StatItem> statsItems = new ArrayList<>();
        ArrayList<DNAResultItem> resultItemsToImpress = new ArrayList<>();

        List<DNAResultItem> resultsToReturn = retrieveCachedSuggestions(count);
        boolean isCache = true;
        boolean isFallback = false;
        String source = DNAConstants.SOURCE_SUGGESTED_APPS;

        if (resultsToReturn == null || resultsToReturn.isEmpty()) {
            resultsToReturn = resultEngine_.fetchResultsGeneric(source,  placementTag,true,false, false, null, count);
            isCache = false;
            adCacheManager_.setDisplayCache(resultsToReturn);
        }

        if (resultsToReturn == null || resultsToReturn.isEmpty()) {
            DNALogger.i("fetchOrganicResultsForSearch: no search results found, triggering fallback");
            resultsToReturn = DNAAppSearchFallback.getAppResultsForDisplayFallback(context_, count);
            isCache = false;
            isFallback = true;
        }

        DNALogger.i("fetchOrganicAppSuggestions: retrieved " + resultsToReturn.size() + " suggestions");
        long endTime = System.currentTimeMillis();

        JSONObject statsMetadata = new JSONObject();
        try {
            statsMetadata.put("sId", dnaPreferences_.getSessionID());
            statsMetadata.put("cache", isCache);
            statsMetadata.put("fallback", isFallback);
            statsMetadata.put("source", source);
            if (placementTag != null && !placementTag.isEmpty()) {
                statsMetadata.put("placementTag", placementTag);
            } else {
                statsMetadata.put("placementTag", source);
            }
        } catch (Exception je) {
            DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
        }
        statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Organic Display", (int) (endTime - currentTime), statsMetadata));

        for (DNAResultItem resultItem : resultsToReturn) {
            resultItem.placementTag = placementTag;
            resultItem.source = source;
            if (isCache) {
                resultItem.updateAppInstalledState(context_);
            }

            if (resultItem.resultType.equals(DNAResultItem.TYPE_AD)) {
                statsMetadata = new JSONObject();
                try {
                    statsMetadata.put("sId", dnaPreferences_.getSessionID());
                    statsMetadata.put("adId", resultItem.id);
                    statsMetadata.put("adPackageName", resultItem.packageName);
                    statsMetadata.put("adAppName", resultItem.appName);
                    statsMetadata.put("here", resultItem.isInstalled);
                    statsMetadata.put("source", source);
                    statsMetadata.put("cache", isCache);
                    if (placementTag != null && !placementTag.isEmpty()) {
                        statsMetadata.put("placementTag", placementTag);
                    } else {
                        resultItem.placementTag = source;
                        statsMetadata.put("placementTag", source);
                    }
                    if (resultItem.statMetadata != null) {
                        Iterator<String> keys = resultItem.statMetadata.keys();
                        while (keys.hasNext()) {
                            String key = keys.next();
                            statsMetadata.put(key, resultItem.statMetadata.get(key));
                        }
                    }
                } catch (Exception je) {
                    DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                }
                statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Organic Display", (int) (endTime - currentTime), statsMetadata));
            }

            resultItemsToImpress.add(resultItem);
        }

        if (!resultItemsToImpress.isEmpty()) {
            registerBatchImpression(resultItemsToImpress, null, true, null);
        }

        attemptToExecuteRunnableAndReinit(() -> DNAStatsLogger.logBatchInternalSync(context_, statsItems));

        return resultsToReturn;
    }

    public List<DNAResultItem> fetchOrganicLinkSuggestions(int count, String placementTag) {
        DNALogger.i("fetchOrganicAppSuggestions: fetching suggestions with count: " + count);
        long currentTime = System.currentTimeMillis();

        ArrayList<DNAStatsLogger.StatItem> statsItems = new ArrayList<>();
        ArrayList<DNAResultItem> resultItemsToImpress = new ArrayList<>();

        List<DNAResultItem> resultsToReturn = retrieveCachedLinkSuggestions(count);
        String source = DNAConstants.SOURCE_SUGGESTED_LINKS;
        boolean isCache = true;

        if (resultsToReturn == null || resultsToReturn.isEmpty()) {
            resultsToReturn = resultEngine_.fetchResultsGeneric(source, placementTag, true, true, false, null, count);
            isCache = false;
            adCacheManager_.setDisplayLinkCache(resultsToReturn);
        }

        DNALogger.i("fetchOrganicLinkSuggestions: retrieved " + resultsToReturn.size() + " suggestions");
        long endTime = System.currentTimeMillis();

        for (DNAResultItem resultItem : resultsToReturn) {
            resultItem.placementTag = placementTag;
            resultItem.source = source;
            if (isCache) {
                resultItem.updateAppInstalledState(context_);
            }

            if (resultItem.resultType.equals(DNAResultItem.TYPE_AD)) {
                JSONObject statsMetadata = new JSONObject();
                try {
                    statsMetadata.put("sId", dnaPreferences_.getSessionID());
                    statsMetadata.put("adId", resultItem.id);
                    statsMetadata.put("adPackageName", resultItem.packageName);
                    statsMetadata.put("adAppName", resultItem.appName);
                    statsMetadata.put("here", resultItem.isInstalled);
                    statsMetadata.put("source", source);
                    statsMetadata.put("cache", isCache);
                    if (placementTag != null && !placementTag.isEmpty()) {
                        statsMetadata.put("placementTag", placementTag);
                    } else {
                        resultItem.placementTag = source;
                        statsMetadata.put("placementTag", source);
                    }
                    if (resultItem.statMetadata != null) {
                        Iterator<String> keys = resultItem.statMetadata.keys();
                        while (keys.hasNext()) {
                            String key = keys.next();
                            statsMetadata.put(key, resultItem.statMetadata.get(key));
                        }
                    }
                } catch (Exception je) {
                    DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                }
                statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Organic Display Links", (int) (endTime - currentTime), statsMetadata));
            }

            resultItemsToImpress.add(resultItem);
        }

        if (!resultItemsToImpress.isEmpty()) {
            registerBatchImpression(resultItemsToImpress, null);
        }

        attemptToExecuteRunnableAndReinit(() -> DNAStatsLogger.logBatchInternalSync(context_, statsItems));

        return resultsToReturn;
    }

    public List<DNAResultItem> fetchOrganicResultsForSearch(String query, String placementTag, boolean forDisplay, int count) {
        DNALogger.i("fetchOrganicResultsForSearch: fetching results for query: " + query);
        long currentTime = System.currentTimeMillis();

        ArrayList<DNAStatsLogger.StatItem> statsItems = new ArrayList<>();
        ArrayList<DNAResultItem> resultsToImpress = new ArrayList<>();

        List<DNAResultItem> resultItems = adCacheManager_.getSearchCache(query);
        String source = DNAConstants.SOURCE_SEARCH;
        boolean isCache = true;
        boolean isFallback = false;

        if (resultItems == null || resultItems.isEmpty()) {
            DNALogger.i("fetchOrganicResultsForSearch: no cached search results found, fetching new results");
            resultItems = resultEngine_.fetchResultsGeneric(source, placementTag, true, false, false, query, -1);
            isCache = false;
            adCacheManager_.setSearchCache(query, resultItems);
        }

        if (resultItems == null || resultItems.isEmpty()) {
            DNALogger.i("fetchOrganicResultsForSearch: no search results found, triggering fallback");
            resultItems = DNAAppSearchFallback.getAppResultsForSearchFallback(context_, query);
            isCache = false;
            isFallback = true;
        }

        DNALogger.i("fetchOrganicResultsForSearch: fetched " + resultItems.size() + " search results");

        long endTime = System.currentTimeMillis();

        JSONObject statsMetadata = new JSONObject();
        try {
            statsMetadata.put("sId", dnaPreferences_.getSessionID());
            statsMetadata.put("cache", isCache);
            statsMetadata.put("fallback", isFallback);
            statsMetadata.put("queryLength", query.length());
            statsMetadata.put("source", source);
            if (placementTag != null && !placementTag.isEmpty()) {
                statsMetadata.put("placementTag", placementTag);
            } else {
                statsMetadata.put("placementTag", source);
            }
        } catch (Exception je) {
            DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
        }
        statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Organic Search", (int) (endTime - currentTime), statsMetadata));

        int index = 0;
        for (DNAResultItem resultItem : resultItems) {
            resultItem.placementTag = placementTag;
            resultItem.source = source;
            if (isCache) {
                resultItem.updateAppInstalledState(context_);
            }

            if (resultItem.resultType.equals(DNAResultItem.TYPE_AD)) {
                statsMetadata = new JSONObject();
                try {
                    statsMetadata.put("sId", dnaPreferences_.getSessionID());
                    statsMetadata.put("adId", resultItem.id);
                    statsMetadata.put("adPackageName", resultItem.packageName);
                    statsMetadata.put("adAppName", resultItem.appName);
                    statsMetadata.put("here", resultItem.isInstalled);
                    statsMetadata.put("cache", isCache);
                    statsMetadata.put("queryLength", query.length());
                    statsMetadata.put("source", source);
                    if (placementTag != null && !placementTag.isEmpty()) {
                        statsMetadata.put("placementTag", placementTag);
                    } else {
                        resultItem.placementTag = source;
                        statsMetadata.put("placementTag", source);
                    }
                    if (resultItem.statMetadata != null) {
                        Iterator<String> keys = resultItem.statMetadata.keys();
                        while (keys.hasNext()) {
                            String key = keys.next();
                            statsMetadata.put(key, resultItem.statMetadata.get(key));
                        }
                    }
                } catch (Exception je) {
                    DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                }
                statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Organic Search", (int) (endTime - currentTime), statsMetadata));
            }
            DNALogger.i("fetchOrganicResultsForSearch: evaluating result " + resultItem.packageName);

            if ((resultItem.resultType.equals(DNAResultItem.TYPE_AD) ||
                    resultItem.resultType.equals(DNAResultItem.TYPE_APP)) &&
                    ((count > 0 && index < count) || count == -1)) {
                resultsToImpress.add(resultItem);
            }
            index++;
        }

        if (!resultsToImpress.isEmpty() && forDisplay) {
            Runnable impressionRunnable = () -> {
                DNALogger.i("fetchAdsForSearch: registering impressions for " + resultsToImpress.size() + " results");
                registerBatchImpression(resultsToImpress, null);
            };
            if (searchDebounceTask_ != null && !searchDebounceTask_.isDone()) {
                searchDebounceTask_.cancel(true);
            }
            searchDebounceTask_ = scheduledExecutorService_.schedule(impressionRunnable, 1, TimeUnit.SECONDS);
        }

        attemptToExecuteRunnableAndReinit(() -> DNAStatsLogger.logBatchInternalSync(context_, statsItems));

        return resultItems;
    }

    public List<DNAResultItem> fetchAdsForCache(int count, String placementTag) {
        return fetchAdsForDisplayInternal(false, count, placementTag);
    }

    public List<DNAResultItem> fetchAdsForDisplay(int count, String placementTag) {
        return fetchAdsForDisplayInternal(true, count, placementTag);
    }

    private List<DNAResultItem> fetchAdsForDisplayInternal(boolean forDisplay, int count, String placementTag) {
        DNALogger.i("fetchAdsForDisplayInternal: fetching display ads for display: " + forDisplay + " with count: " + count);

        long currentTime = System.currentTimeMillis();
        ArrayList<DNAStatsLogger.StatItem> statsItems = new ArrayList<>();
        ArrayList<DNAResultItem> adUnitsToImpress = new ArrayList<>();

        List<DNAResultItem> adsToReturn = retrieveCachedDisplayAds(count);
        String source = DNAConstants.SOURCE_SUGGESTED_APPS;
        boolean isCache = true;

        if (adsToReturn == null  || adsToReturn.isEmpty()) {
            adsToReturn = resultEngine_.fetchResultsGeneric(source, placementTag, false, false, false, null, count);
            isCache = false;
            adCacheManager_.setAdsDisplayCache(adsToReturn);
        }

        if (adsToReturn == null) {
            adsToReturn = new ArrayList<>();
        }

        DNALogger.i("fetchAdsForDisplayInternal: retrieved display ads from source " + source + " with count: " + count);
        long endTime = System.currentTimeMillis();

        JSONObject statsMetadata = new JSONObject();
        try {
            statsMetadata.put("sId", dnaPreferences_.getSessionID());
            statsMetadata.put("source", source);
            statsMetadata.put("cache", isCache);
            if (placementTag != null && !placementTag.isEmpty()) {
                statsMetadata.put("placementTag", placementTag);
            } else {
                statsMetadata.put("placementTag", source);
            }
        } catch (Exception je) {
            DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
        }
        if (forDisplay) {
            statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Display", (int) (endTime - currentTime), statsMetadata));
        } else {
            statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Cache", (int) (endTime - currentTime), statsMetadata));
        }

        for (DNAResultItem adUnit : adsToReturn) {
            statsMetadata = new JSONObject();
            adUnit.placementTag = placementTag;
            adUnit.source = source;
            if (isCache) {
                adUnit.updateAppInstalledState(context_);
            }

            try {
                statsMetadata.put("sId", dnaPreferences_.getSessionID());
                statsMetadata.put("adId", adUnit.id);
                statsMetadata.put("adPackageName", adUnit.packageName);
                statsMetadata.put("adAppName", adUnit.appName);
                statsMetadata.put("source", source);
                statsMetadata.put("cache", isCache);
                statsMetadata.put("here", adUnit.isInstalled);
                if (placementTag != null && !placementTag.isEmpty()) {
                    statsMetadata.put("placementTag", placementTag);
                } else {
                    adUnit.placementTag = source;
                    statsMetadata.put("placementTag", source);
                }
                if (adUnit.statMetadata != null) {
                    Iterator<String> keys = adUnit.statMetadata.keys();
                    while (keys.hasNext()) {
                        String key = keys.next();
                        statsMetadata.put(key, adUnit.statMetadata.get(key));
                    }
                }
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            if (forDisplay) {
                statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Display", (int) (endTime - currentTime), statsMetadata));
            } else {
                statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Cache", (int) (endTime - currentTime), statsMetadata));
            }
//            DNALogger.i("fetchAdsForDisplayInternal: evaluating " + adUnit.packageName);

            adUnitsToImpress.add(adUnit);
        }

        if (!adUnitsToImpress.isEmpty() && forDisplay) {
            DNALogger.i("fetchAdsForDisplayInternal: registering impressions for " + adUnitsToImpress.size() + " ads");
            registerBatchImpression(adUnitsToImpress, null, true, null);
        }

        attemptToExecuteRunnableAndReinit(() -> DNAStatsLogger.logBatchInternalSync(context_, statsItems));

        return adsToReturn;
    }

    public List<DNAResultItem> fetchHotAppsList(int count, String placementTag) {
        DNALogger.i("fetchHotAppsList: fetching hot apps with count: " + count);
        long currentTime = System.currentTimeMillis();

        ArrayList<DNAStatsLogger.StatItem> statsItems = new ArrayList<>();
        ArrayList<DNAResultItem> resultItemsToImpress = new ArrayList<>();

        List<DNAResultItem> resultsToReturn = retrieveCachedHotApps(count);
        String source = DNAConstants.SOURCE_HOT_APPS;
        boolean isCache = true;

        if (resultsToReturn == null || resultsToReturn.isEmpty()) {
            resultsToReturn = resultEngine_.fetchResultsGeneric(source, placementTag,false, false, true, null, count);
            isCache = false;
            adCacheManager_.setHotAppsCache(resultsToReturn);
        }

        DNALogger.i("fetchHotAppsList: retrieved " + resultsToReturn.size() + " hot apps");
        long endTime = System.currentTimeMillis();

        for (DNAResultItem resultItem : resultsToReturn) {
            resultItem.placementTag = placementTag;
            resultItem.source = source;
            if (isCache) {
                resultItem.updateAppInstalledState(context_);
            }

            if (resultItem.resultType.equals(DNAResultItem.TYPE_AD)) {
                JSONObject statsMetadata = new JSONObject();
                try {
                    statsMetadata.put("sId", dnaPreferences_.getSessionID());
                    statsMetadata.put("adId", resultItem.id);
                    statsMetadata.put("adPackageName", resultItem.packageName);
                    statsMetadata.put("adAppName", resultItem.appName);
                    statsMetadata.put("here", resultItem.isInstalled);
                    statsMetadata.put("source", source);
                    statsMetadata.put("cache", isCache);
                    if (placementTag != null && !placementTag.isEmpty()) {
                        statsMetadata.put("placementTag", placementTag);
                    } else {
                        resultItem.placementTag = source;
                        statsMetadata.put("placementTag", source);
                    }
                    if (resultItem.statMetadata != null) {
                        Iterator<String> keys = resultItem.statMetadata.keys();
                        while (keys.hasNext()) {
                            String key = keys.next();
                            statsMetadata.put(key, resultItem.statMetadata.get(key));
                        }
                    }
                } catch (Exception je) {
                    DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                }
                statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Hot Apps", (int) (endTime - currentTime), statsMetadata));
            }

            resultItemsToImpress.add(resultItem);
        }

        if (!resultItemsToImpress.isEmpty()) {
            registerBatchImpression(resultItemsToImpress, null);
        }

        attemptToExecuteRunnableAndReinit(() -> DNAStatsLogger.logBatchInternalSync(context_, statsItems));

        return resultsToReturn;
    }

    public List<DNAResultItem> fetchAdsForSearch(String query, String placementTag, boolean forDisplay) {
        DNALogger.i("fetchAdsForSearch: fetching search ads for query: " + query);
        long currentTime = System.currentTimeMillis();
        ArrayList<DNAStatsLogger.StatItem> statsItems = new ArrayList<>();
        ArrayList<DNAResultItem> adUnitsToImpress = new ArrayList<>();
        List<DNAResultItem> adUnits = adCacheManager_.getSearchCache(query);

        String source = DNAConstants.SOURCE_SEARCH;
        boolean isCache = true;
        if (adUnits == null || adUnits.isEmpty()) {
            DNALogger.i("fetchAdsForSearch: no cached search ads found, fetching new ads");
            adUnits = resultEngine_.fetchResultsGeneric(source, placementTag, false, false, false, query, -1);
            isCache = false;
            adCacheManager_.setSearchCache(query, adUnits);
        }

        DNALogger.i("fetchAdsForSearch: fetched " + adUnits.size() + " search ads");

        long endTime = System.currentTimeMillis();

        JSONObject statsMetadata = new JSONObject();
        try {
            statsMetadata.put("sId", dnaPreferences_.getSessionID());
            statsMetadata.put("cache", isCache);
            statsMetadata.put("queryLength", query.length());
            statsMetadata.put("source", source);
            if (placementTag != null && !placementTag.isEmpty()) {
                statsMetadata.put("placementTag", placementTag);
            } else {
                statsMetadata.put("placementTag", source);
            }
        } catch (Exception je) {
            DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
        }
        statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Search", (int) (endTime - currentTime), statsMetadata));

        for (DNAResultItem adUnit : adUnits) {
            statsMetadata = new JSONObject();
            adUnit.placementTag = placementTag;
            adUnit.source = source;
            if (isCache) {
                adUnit.updateAppInstalledState(context_);
            }

            try {
                statsMetadata.put("sId", dnaPreferences_.getSessionID());
                statsMetadata.put("adId", adUnit.id);
                statsMetadata.put("adPackageName", adUnit.packageName);
                statsMetadata.put("adAppName", adUnit.appName);
                statsMetadata.put("here", adUnit.isInstalled);
                statsMetadata.put("cache", isCache);
                statsMetadata.put("queryLength", query.length());
                statsMetadata.put("source", source);
                if (placementTag != null && !placementTag.isEmpty()) {
                    statsMetadata.put("placementTag", placementTag);
                } else {
                    adUnit.placementTag = source;
                    statsMetadata.put("placementTag", source);
                }
                if (adUnit.statMetadata != null) {
                    Iterator<String> keys = adUnit.statMetadata.keys();
                    while (keys.hasNext()) {
                        String key = keys.next();
                        statsMetadata.put(key, adUnit.statMetadata.get(key));
                    }
                }
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            statsItems.add(new DNAStatsLogger.StatItem("Ad Request: Search", (int) (endTime - currentTime), statsMetadata));

            DNALogger.i("fetchAdsForSearch: evaluating adUnit " + adUnit.packageName);

            adUnitsToImpress.add(adUnit);
        }

        if (!adUnitsToImpress.isEmpty() && forDisplay) {
            Runnable impressionRunnable = () -> {
                DNALogger.i("fetchAdsForSearch: registering impressions for " + adUnitsToImpress.size() + " adUnits");
                registerBatchImpression(adUnitsToImpress, null);
            };
            if (searchDebounceTask_ != null && !searchDebounceTask_.isDone()) {
                searchDebounceTask_.cancel(true);
            }
            searchDebounceTask_ = scheduledExecutorService_.schedule(impressionRunnable, 1, TimeUnit.SECONDS);
        }

        attemptToExecuteRunnableAndReinit(() -> DNAStatsLogger.logBatchInternalSync(context_, statsItems));

        return adUnits;
    }

    public void registerBatchImpression(List<DNAResultItem> resultItems, DeviceNativeClickHandler clickHandler) {
        registerBatchImpression(resultItems, null, false, clickHandler);
    }

    public void registerBatchImpression(List<DNAResultItem> resultItems, String updatedPlacementTag, boolean refreshUsage, DeviceNativeClickHandler clickHandler) {
        executorService_.submit(() -> {
            validateIsActiveAndReinitialize();
            ArrayList<DNAStatsLogger.StatItem> statsItems = new ArrayList<>();
            try {
                for (DNAResultItem resultItem : resultItems) {
                    if (!adCacheManager_.getPackageImpressionTimestampExpired(resultItem.packageName)) {
                        DNALogger.i("registerBatchImpression: ad " + resultItem.packageName + " in imp blackout window");
                        continue;
                    }
                    adCacheManager_.setPackageImpressionTimestamp(resultItem.packageName);

                    String clickId = java.util.UUID.randomUUID().toString();

                    if (updatedPlacementTag != null && !updatedPlacementTag.isEmpty()) {
                        resultItem.placementTag = updatedPlacementTag;
                    }

                    JSONObject statsMetadata = new JSONObject();
                    if (resultItem.resultType.equals(DNAResultItem.TYPE_AD)) {
                        DNALogger.i("registerBatchImpression: registering batch impression for ad: " + resultItem.packageName);
                        // Log the view event

                        try {
                            statsMetadata.put("sId", dnaPreferences_.getSessionID());
                            statsMetadata.put("clickId", clickId);
                            statsMetadata.put("adId", resultItem.id);
                            statsMetadata.put("adPackageName", resultItem.packageName);
                            statsMetadata.put("adAppName", resultItem.appName);
                            statsMetadata.put("here", resultItem.isInstalled);
                            if (resultItem.placementTag != null && !resultItem.placementTag.isEmpty()) {
                                statsMetadata.put("placementTag", resultItem.placementTag);
                            }
                            if (resultItem.source != null && !resultItem.source.isEmpty()) {
                                statsMetadata.put("source", resultItem.source);
                            }
                        } catch (Exception je) {
                            DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                        }
                        statsItems.add(new DNAStatsLogger.StatItem("Ad Imp", null, statsMetadata));

                    }

                    // Create the local event record
                    DNAResultEventRecord eventRecord = new DNAResultEventRecord();
                    eventRecord.resId = resultItem.id;
                    eventRecord.event = "view";
                    eventRecord.type = resultItem.resultType;
                    if (resultItem.placementTag != null && !resultItem.placementTag.isEmpty()) {
                        eventRecord.placement = resultItem.placementTag;
                    } else {
                        eventRecord.placement = "organic";
                    }
                    eventRecord.packageName = resultItem.packageName;
                    DNADatabaseInterface.insertResEventRecord(context_, eventRecord);

                    // Fire the impression URL if present
                    if (resultItem.impressionUrl != null && !resultItem.impressionUrl.isEmpty()) {
                        DNALogger.i("registerBatchImpression: firing impression URL for ad: " + resultItem.packageName);
                        adLinkHandler_.asyncLinkProcessor(context_, resultItem.impressionUrl, resultItem.id, resultItem.appName, resultItem.packageName, clickId, resultItem.source, clickHandler);
                    }

                    // Check if VTA enabled
                    if (resultItem.viewThrough > 0 &&
                            resultItem.clickUrl != null &&
                            !resultItem.clickUrl.isEmpty() &&
                            dnaPreferences_.getVTAEnabled()) {

                        // Check if VTA has been fired for this ad before
                        if (!DNADatabaseInterface.hasVTABeenFired(context_, resultItem.id)) {
                            DNALogger.i("registerBatchImpression: firing VTA for ad: " + resultItem.packageName);
                            // Register the local event
                            statsMetadata = new JSONObject();
                            try {
                                statsMetadata.put("sId", dnaPreferences_.getSessionID());
                                statsMetadata.put("clickId", clickId);
                                statsMetadata.put("adId", resultItem.id);
                                statsMetadata.put("adPackageName", resultItem.packageName);
                                statsMetadata.put("adAppName", resultItem.appName);
                                statsMetadata.put("here", resultItem.isInstalled);
                                if (resultItem.placementTag != null && !resultItem.placementTag.isEmpty()) {
                                    statsMetadata.put("placementTag", resultItem.placementTag);
                                }
                                if (resultItem.source != null && !resultItem.source.isEmpty()) {
                                    statsMetadata.put("source", resultItem.source);
                                }
                            } catch (Exception je) {
                                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                            }
                            statsItems.add(new DNAStatsLogger.StatItem("Ad VTA", null, statsMetadata));

                            // Actually fire the VTA link
                            adLinkHandler_.linkRouter(context_, resultItem.clickUrl, resultItem.destinationUrl, resultItem.id, resultItem.appName, resultItem.packageName, resultItem.className, resultItem.userHandle, clickId, resultItem.source, false, false, null, null);

                            // Save a new ad record
                            eventRecord = new DNAResultEventRecord();
                            eventRecord.resId = resultItem.id;
                            eventRecord.type = resultItem.resultType;
                            eventRecord.placement = resultItem.placementTag;
                            eventRecord.event = "vta";
                            eventRecord.packageName = resultItem.packageName;
                            DNADatabaseInterface.insertResEventRecord(context_, eventRecord);
                        }
                    }
                }

                fireOffTheStatsUploadRunnable();
                if (refreshUsage) {
                    fireOffTheUsageRefreshRunnable();
                }
            } catch (Exception e) {
                DNALogger.e("DeviceNativeAds: error processing batch impression ad: " + e.getMessage());
                JSONObject statsMetadata = new JSONObject();
                try {
                    statsMetadata.put("sId", dnaPreferences_.getSessionID());
                    statsMetadata.put("loc", "registerBatchImpression");
                    statsMetadata.put("m", e.getMessage());
                } catch (Exception je) {
                    DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                }
                statsItems.add(new DNAStatsLogger.StatItem("Error", null, statsMetadata));
            }
            DNAStatsLogger.logBatchInternalSync(context_, statsItems);
        });
    }

    public void registerClick(DNAResultItem resultItem, DeviceNativeClickHandler clickHandler) {
        final String clickId = java.util.UUID.randomUUID().toString();
        if (resultItem.resultType.equals(DNAResultItem.TYPE_AD) || (resultItem.parentResultItem != null && resultItem.parentResultItem.resultType.equals(DNAResultItem.TYPE_AD))) {
            DNAResultItem logResultItem = resultItem;
            if (resultItem.parentResultItem != null) {
                logResultItem = resultItem.parentResultItem;
            }
            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", dnaPreferences_.getSessionID());
                statsMetadata.put("resType", logResultItem.resultType);
                statsMetadata.put("clickId", clickId);
                statsMetadata.put("adId", logResultItem.id);
                statsMetadata.put("adPackageName", logResultItem.packageName);
                statsMetadata.put("adAppName", logResultItem.appName);
                statsMetadata.put("here", logResultItem.isInstalled);
                if (resultItem.placementTag != null && !resultItem.placementTag.isEmpty()) {
                    statsMetadata.put("placementTag", resultItem.placementTag);
                }
                if (resultItem.source != null && !resultItem.source.isEmpty()) {
                    statsMetadata.put("source", resultItem.source);
                }
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            DNAStatsLogger.logInternal(context_, "Click", null, statsMetadata);
        }

        fireOffTheStatsUploadRunnable();

        attemptToExecuteRunnableAndReinit(() -> {
            DNAResultEventRecord eventRecord = new DNAResultEventRecord();
            eventRecord.resId = resultItem.id;
            eventRecord.event = "click";
            eventRecord.type = resultItem.resultType;
            if (resultItem.placementTag != null && !resultItem.placementTag.isEmpty()) {
                eventRecord.placement = resultItem.placementTag;
            } else {
                eventRecord.placement = "organic";
            }
            eventRecord.packageName = resultItem.packageName;
            DNADatabaseInterface.insertResEventRecord(context_, eventRecord);

            fireOffTheUsageRefreshRunnable();

            DNAResultItem logResultItem = resultItem;
            if (resultItem.parentResultItem != null) {
                logResultItem = resultItem.parentResultItem;
            }
            if (logResultItem.clickUrl != null && !logResultItem.clickUrl.isEmpty()) {
                adLinkHandler_.asyncLinkProcessor(context_, logResultItem.clickUrl, logResultItem.id, logResultItem.appName, logResultItem.packageName, clickId, resultItem.source, clickHandler);
            }
        });
    }

    /**
     * Method to register a click for non-DNAResultItem objects. This retrieves the app record
     * from the database and creates a click event record.
     * @param packageName The package name of the app being clicked.
     * @param componentName The component name of the app being clicked.
     * @param userHandle The user handle for multi-user support. Can be null for current user.
     */
    public void registerClick(String packageName, String componentName, UserHandle userHandle) {
        attemptToExecuteRunnableAndReinit(() -> {
            // Convert UserHandle to string, use current user if null
            String userId = null;
            if (userHandle != null) {
                userId = userHandle.toString();
            } else {
                try {
                    UserHandle currentUserHandle = Process.myUserHandle();
                    userId = currentUserHandle.toString();
                } catch (Exception e) {
                    DNALogger.e("registerClick: Unable to get current user handle, using null");
                }
            }

            // Retrieve app records to find the matching record
            List<DNAAppRecord> appRecords = DNADatabaseInterface.getAppRecords(context_, packageName, userId);
            
            // Find the matching app record by component
            DNAAppRecord matchingRecord = null;
            if (appRecords != null && !appRecords.isEmpty()) {
                for (DNAAppRecord record : appRecords) {
                    if (componentName != null && componentName.equals(record.component)) {
                        matchingRecord = record;
                        break;
                    } else if (componentName == null && record.component == null) {
                        // If no component specified, use the first record for this package/user
                        matchingRecord = record;
                        break;
                    }
                }
                
                // If no exact component match found and componentName was specified, 
                // fall back to first record for this package/user
                if (matchingRecord == null && !appRecords.isEmpty()) {
                    matchingRecord = appRecords.get(0);
                }
            }
            
            if (matchingRecord != null) {
                // Create the click event record using the app record's database ID
                DNAResultEventRecord eventRecord = new DNAResultEventRecord();
                eventRecord.resId = String.valueOf(matchingRecord.dbId);
                eventRecord.event = "click";
                eventRecord.type = "app";
                eventRecord.placement = "organic";
                eventRecord.packageName = packageName;
                
                DNADatabaseInterface.insertResEventRecord(context_, eventRecord);
                
                DNALogger.i("registerClick: Registered click for package: " + packageName + ", component: " + componentName + ", user: " + userId + ", resId: " + matchingRecord.dbId);

                fireOffTheUsageRefreshRunnable();
            } else {
                DNALogger.e("registerClick: No matching app record found for package: " + packageName + ", component: " + componentName + ", user: " + userId);
            }
        });
    }


    public void clickAndRoute(DNAResultItem resultItem, String destinationUrlOverride, DeviceNativeClickHandler clickHandler) {
        final String clickId = java.util.UUID.randomUUID().toString();
        if (resultItem.resultType.equals(DNAResultItem.TYPE_SHORTCUT) && !resultItem.clickUrl.contains("://")) {
            if (resultItem.packageName.equals("com.android.settings")) {
                DNALogger.i("DeviceNativeAds: setting shortcut router for " + resultItem.packageName + " with click url: " + resultItem.clickUrl);
                adLinkHandler_.settingShortcutRouter(context_, resultItem.clickUrl, resultItem.packageName, clickHandler);
            } else {
                adLinkHandler_.shortcutRouter(context_, resultItem.clickUrl, resultItem.appName, resultItem.packageName, clickHandler);
            }
        } else if (resultItem.clickUrl != null && !resultItem.clickUrl.isEmpty()) {
            try {
                boolean earlyOpenPlayStore = !resultItem.isInstalled && !resultItem.notPlayStore;
                clickExecutorService_.execute(() -> adLinkHandler_.linkRouter(context_, resultItem.clickUrl, resultItem.destinationUrl, resultItem.id, resultItem.appName, resultItem.packageName, resultItem.className, resultItem.userHandle, clickId, resultItem.source, true, earlyOpenPlayStore, destinationUrlOverride,  clickHandler));
            } catch(Exception e) {
                DNALogger.i("DeviceNativeAds: Thread pool full, already processing link redirect!");
            }
        } else {
            adLinkHandler_.fallbackLinkRouter(context_, resultItem.packageName, resultItem.className, resultItem.id, resultItem.appName, resultItem.userHandle, clickHandler);
        }

        attemptToExecuteRunnableAndReinit(() -> {
            if (resultItem.resultType.equals(DNAResultItem.TYPE_AD) || (resultItem.parentResultItem != null && resultItem.parentResultItem.resultType.equals(DNAResultItem.TYPE_AD))) {
                DNAResultItem logResultItem = resultItem;
                if (resultItem.parentResultItem != null) {
                    logResultItem = resultItem.parentResultItem;
                }
                JSONObject statsMetadata = new JSONObject();
                try {
                    statsMetadata.put("sId", dnaPreferences_.getSessionID());
                    statsMetadata.put("resType", logResultItem.resultType);
                    statsMetadata.put("clickId", clickId);
                    statsMetadata.put("adId", logResultItem.id);
                    statsMetadata.put("adPackageName", logResultItem.packageName);
                    statsMetadata.put("adAppName", logResultItem.appName);
                    statsMetadata.put("here", logResultItem.isInstalled);
                    if (resultItem.placementTag != null && !resultItem.placementTag.isEmpty()) {
                        statsMetadata.put("placementTag", resultItem.placementTag);
                    }
                    if (resultItem.source != null && !resultItem.source.isEmpty()) {
                        statsMetadata.put("source", resultItem.source);
                    }
                } catch (Exception je) {
                    DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                }
                DNAStatsLogger.logInternalSync(context_, "Ad Click & Route", null, statsMetadata);

                if (resultItem.parentResultItem != null) {
                    registerClick(resultItem.parentResultItem, null);
                }
            }

            fireOffTheStatsUploadRunnable();

            DNAResultEventRecord eventRecord = new DNAResultEventRecord();
            eventRecord.resId = resultItem.id;
            eventRecord.event = "click";
            eventRecord.type = resultItem.resultType;
            if (resultItem.placementTag != null && !resultItem.placementTag.isEmpty()) {
                eventRecord.placement = resultItem.placementTag;
            } else {
                eventRecord.placement = "organic";
            }
            eventRecord.packageName = resultItem.packageName;
            DNADatabaseInterface.insertResEventRecord(context_, eventRecord);

            fireOffTheUsageRefreshRunnable();
        });
    }

    private void fireOffTheStatsUploadRunnable() {
        Runnable statsUploadRunnable = () -> {
            if (statSyncExecutor_ != null) {
                statSyncExecutor_.execute();
            }
        };
        if (statsSyncTask_ != null && !statsSyncTask_.isDone()) {
            statsSyncTask_.cancel(true);
        }
        statsSyncTask_ = scheduledExecutorService_.schedule(statsUploadRunnable, dnaPreferences_.getImpressionBlackoutWindow(), TimeUnit.MILLISECONDS);
    }

    private void fireOffTheUsageRefreshRunnable() {
        Runnable usageRefreshRunnable = () -> {
            Intent wakeIntent = new Intent("com.devicenative.dna.WAKE_AND_SYNC");
            if (context_ != null) {
                context_.sendBroadcast(wakeIntent);
            }
        };
        if (usageSyncTask_ != null && !usageSyncTask_.isDone()) {
            usageSyncTask_.cancel(true);
        }
        usageSyncTask_ = scheduledExecutorService_.schedule(usageRefreshRunnable, DNAPreferences.USAGE_SYNC_INTERVAL, TimeUnit.MILLISECONDS);
    }

    private void attemptToExecuteRunnableAndReinit(Runnable runner) {
        validateIsActiveAndReinitialize();
        executorService_.submit(runner);
    }
}
