package com.devicenative.dna.ads;

import android.content.Context;
import android.os.Process;
import android.os.UserHandle;
import android.text.TextUtils;

import com.devicenative.dna.DNAResultItem;
import com.devicenative.dna.db.DNAAdRecord;
import com.devicenative.dna.db.DNAAppRecord;
import com.devicenative.dna.db.DNADatabaseInterface;
import com.devicenative.dna.db.DNANotifRecord;
import com.devicenative.dna.db.DNAResultEventRecord;
import com.devicenative.dna.db.DNAShortcutRecord;
import com.devicenative.dna.utils.DNAConstants;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;
import com.devicenative.dna.utils.DNAUtils;
import com.shiqi.quickjs.JSArray;
import com.shiqi.quickjs.JSContext;
import com.shiqi.quickjs.JSNumber;
import com.shiqi.quickjs.JSObject;
import com.shiqi.quickjs.JSRuntime;
import com.shiqi.quickjs.JSString;
import com.shiqi.quickjs.JSUndefined;
import com.shiqi.quickjs.JSValue;
import com.shiqi.quickjs.QuickJS;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;
import java.io.FileWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.text.Normalizer;
import java.util.Locale;

public class DNAResultEngine {
    private final Context context_;
    private final DNAPreferences dnaPreferences_;
    private final ExecutorService executorService_;

    public DNAResultEngine(Context context, DNAPreferences preferences, ExecutorService executorService) {
        context_ = context;
        dnaPreferences_ = preferences;
        executorService_ = executorService;

        // Warm up JSRuntime
        QuickJS quickJS = new QuickJS.Builder().build();
        try (JSRuntime runtime = quickJS.createJSRuntime()) {
            try (JSContext jsContext = runtime.createJSContext()) {
                populateJSContext(jsContext, null);
                jsContext.evaluate("'make'+'$'", "warmup.js", String.class);
            }
        }
    }

    private static int mapCodePoint(int cp) {
        switch (cp) {
            /* ===== Devanagari ===== */
            case 0x0945: // DEVANAGARI VOWEL SIGN CANDRA E
            case 0x0946: return 0x0947;         // → VOWEL SIGN E
            case 0x0949: // CANDRA O
            case 0x094A: return 0x094B;         // → VOWEL SIGN O
            /* ===== Gujarati ===== */
            case 0x0AC6: return 0x0AC7;         // SHORT E → E
            case 0x0ACA: return 0x0ACB;         // SHORT O → O
            /* ===== Thai ===== */
            case 0x0E33: return 0x0E32;         // SARA AM → SARA AA
            case 0x0E4D:                        // NIKHAHIT (handled above)
            case 0x0E48: case 0x0E49:           // tone marks & thantakhat
            case 0x0E4A: case 0x0E4B:
            case 0x0E4C: case 0x0E4E: return -1;
            /* ===== Lao ===== */
            case 0x0EB3: return 0x0EB2;         // VOWEL AM → AA
            case 0x0ECD:                        // NIGGAHITA
            case 0x0EC8: case 0x0EC9:
            case 0x0ECA: case 0x0ECB:
            case 0x0ECC: return -1;             // tone marks
            /* ===== Myanmar ===== */
            case 0x103A: return 0x1039;         // ASAT ⇢ VIRAMA
            /* fall-through */
            default:   return cp;               // keep as-is
        }
    }

    public static String normalizeText(String text) {
        if (text == null || text.isEmpty()) return "";
        StringBuilder sb = new StringBuilder(text.length());

        for (int i = 0; i < text.length(); ) {
            int cp = text.codePointAt(i);
            int mapped = mapCodePoint(cp);
            if (mapped >= 0) sb.appendCodePoint(mapped);
            i += Character.charCount(cp);
        }
        // 1) NFC to stabilise decompositions, 2) case-fold for Latin/Greek etc.
        return Normalizer.normalize(sb, Normalizer.Form.NFC);
    }
    public ArrayList<DNAResultItem> fetchResultsGeneric(String source, String placement, Boolean fetchOrganic, Boolean linksOnly, Boolean installOnly, String query, int count) {
        ArrayList<DNAResultItem> resultItems = new ArrayList<>();
        ArrayList<DNAStatsLogger.StatItem> statsItems = new ArrayList<>();
        String jsData = "[]";
        long startTime = System.currentTimeMillis();
        long endTime = System.currentTimeMillis();
        QuickJS quickJS = new QuickJS.Builder().build();
        try (JSRuntime runtime = quickJS.createJSRuntime()) {
            try (JSContext context = runtime.createJSContext()) {
                populateJSContext(context, source);
                if (query != null && !query.isEmpty()) {
                    // Normalize the query for better matching - replace entirely since only used for matching
                    String normalizedQuery = normalizeText(query);
                    context.getGlobalObject().setProperty("query", context.createJSString(normalizedQuery));
                } else {
                    context.getGlobalObject().setProperty("query", context.createJSString(""));
                }
                context.getGlobalObject().setProperty("organicRequest", context.createJSBoolean(fetchOrganic));
                context.getGlobalObject().setProperty("linksOnly", context.createJSBoolean(linksOnly));
                context.getGlobalObject().setProperty("installOnly", context.createJSBoolean(installOnly));
                context.getGlobalObject().setProperty("count", context.createJSNumber(count));
                if (placement != null && !placement.isEmpty()) {
                    context.getGlobalObject().setProperty("placement", context.createJSString(placement));
                } else {
                    context.getGlobalObject().setProperty("placement", context.createJSString(""));
                }
                context.getGlobalObject().setProperty("source", context.createJSString(source));
                context.getGlobalObject().setProperty("device", context.createJSString(dnaPreferences_.getDeviceModel()));

                String adSelectionScript = dnaPreferences_.getAdSelectionCode();
                jsData = context.evaluate(adSelectionScript, "ads.js", String.class);
                DNALogger.i("DeviceNativeAds: ad data report: " + jsData);

                if (jsData == null || jsData.isEmpty()) {
                    return new ArrayList<>();
                }

                endTime = System.currentTimeMillis();
                JSONArray dataArray = new JSONArray(jsData);
                for (int i = 0; i < dataArray.length(); i++) {
                    JSONObject resDataObject = dataArray.getJSONObject(i);
                    DNAResultItem parentResultItem;
                    if (resDataObject.has("ad")) {
                        JSONObject ad = resDataObject.getJSONObject("ad");
                        parentResultItem = new DNAResultItem(ad, DNAResultItem.TYPE_AD);
                        if (resDataObject.has("app")) {
                            parentResultItem.userHandle = resDataObject.getJSONObject("app").getString("userId");
                            parentResultItem.uId = DNAUtils.readPidFromUserHandleString(parentResultItem.userHandle);
                            parentResultItem.className = resDataObject.getJSONObject("app").getString("component");
                        }
                        parentResultItem.updateAppInstalledState(context_);
                        if (ad.has("log")) {
                            JSONObject log = ad.getJSONObject("log");
                            parentResultItem.statMetadata = new JSONObject();
                            Iterator<String> keys = log.keys();
                            while (keys.hasNext()) {
                                String key = keys.next();
                                parentResultItem.statMetadata.put(key, log.get(key));
                            }
                            parentResultItem.statMetadata.put("duration", endTime - startTime);
                        }
                        resultItems.add(parentResultItem);
                    } else if (resDataObject.has("shortcut") || resDataObject.has("notif")) {
                        if (resDataObject.has("shortcut")) {
                            parentResultItem = new DNAResultItem(resDataObject.getJSONObject("shortcut"), DNAResultItem.TYPE_SHORTCUT);
                        } else {
                            parentResultItem = new DNAResultItem(resDataObject.getJSONObject("notif"), DNAResultItem.TYPE_NOTIFICATION);
                        }
                        if (resDataObject.has("app")) {
                            parentResultItem.userHandle = resDataObject.getJSONObject("app").getString("userId");
                            parentResultItem.uId = DNAUtils.readPidFromUserHandleString(parentResultItem.userHandle);
                            parentResultItem.className = resDataObject.getJSONObject("app").getString("component");
                        }
                        parentResultItem.updateAppInstalledState(context_);
                        resultItems.add(parentResultItem);
                    } else {
                        parentResultItem = new DNAResultItem(resDataObject.getJSONObject("app"), DNAResultItem.TYPE_APP);
                        parentResultItem.updateAppInstalledState(context_);
                        resultItems.add(parentResultItem);
                    }

                    if (resDataObject.has("sublinks")) {
                        JSONArray sublinks = resDataObject.getJSONArray("sublinks");
                        for (int j = 0; j < sublinks.length(); j++) {
                            JSONObject sublink = sublinks.getJSONObject(j);
                            if (sublink.optString("type").equals("notif")) {
                                DNAResultItem notifResult = new DNAResultItem(sublink, DNAResultItem.TYPE_NOTIFICATION);
                                notifResult.updateAppInstalledState(context_);
                                notifResult.setParentResultItem(parentResultItem);
                                resultItems.add(notifResult);
                            } else if (sublink.optString("type").equals("shortcut")) {
                                DNAResultItem shortcutResult = new DNAResultItem(sublink, DNAResultItem.TYPE_SHORTCUT);
                                shortcutResult.updateAppInstalledState(context_);
                                shortcutResult.setParentResultItem(parentResultItem);
                                resultItems.add(shortcutResult);
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            DNALogger.e("DeviceNativeAds: error fetching ad: " + e.getMessage());
            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", dnaPreferences_.getSessionID());
                statsMetadata.put("loc", "fetchResultsGeneric");
                statsMetadata.put("source", source);
                statsMetadata.put("organicRequest", fetchOrganic);
                statsMetadata.put("m", e.getMessage());
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            statsItems.add(new DNAStatsLogger.StatItem("Error", null, statsMetadata));
        }

        DNALogger.i("DeviceNativeAds: ad request time: " + (endTime - startTime) + " milliseconds");

        if (statsItems.size() > 0) {
            executorService_.execute(() -> DNAStatsLogger.logBatchInternalSync(context_, statsItems));
        }

        return resultItems;
    }

    private void populateJSContext(JSContext jsContext, String source) {
        List<DNAAppRecord> apps = DNADatabaseInterface.getAllAppRecords(context_);
        List<DNAAdRecord> ads = DNADatabaseInterface.getAllAdRecords(context_);
        List<DNAShortcutRecord> shortcuts = DNADatabaseInterface.getAllShortcuts(context_);
        List<DNANotifRecord> notifs = DNADatabaseInterface.getAllNotifs(context_);
        List<DNAResultEventRecord> adEvents = DNADatabaseInterface.getRecentResEventRecords(context_);
        boolean usageStatsEnabled = dnaPreferences_.getUsageStatsEnabled();
        boolean adsEnabled = !dnaPreferences_.getConfigAdsDisabled();
        boolean installAdsEnabled = dnaPreferences_.getInstallAdsEnabled();
        boolean webAdsEnabled = dnaPreferences_.getWebAdsEnabled();
        boolean organicAdsEnabled = dnaPreferences_.getOrganicAdsEnabled();
        boolean organicAdsBoostEnabled = dnaPreferences_.getPaidOrganicBoostEnabled();
        boolean pureOrganicAdsEnabled = dnaPreferences_.getOrganicAdsOnly();
        String currPackageName = context_.getPackageName();

         //printJSONObjectToLocalFile(usageStatsEnabled, apps, ads, shortcuts, notifs, adEvents);

        if (source != null) {
            if (source.equals(DNAConstants.SOURCE_SEARCH)) {
                adsEnabled = adsEnabled && !dnaPreferences_.getSearchAdsDisabled();
                installAdsEnabled = installAdsEnabled && !dnaPreferences_.getSearchInstallAdsDisabled();
            } else if (source.equals(DNAConstants.SOURCE_SUGGESTED_APPS)) {
                adsEnabled = adsEnabled && !dnaPreferences_.getRecomAdsDisabled();
                installAdsEnabled = installAdsEnabled && !dnaPreferences_.getRecomInstallAdsDisabled();
            }
        }

        jsContext.getGlobalObject().setProperty("webAdsEnabled", jsContext.createJSBoolean(webAdsEnabled));
        jsContext.getGlobalObject().setProperty("usageStatsEnabled", jsContext.createJSBoolean(usageStatsEnabled));
        jsContext.getGlobalObject().setProperty("installAdsEnabled", jsContext.createJSBoolean(installAdsEnabled));
        jsContext.getGlobalObject().setProperty("organicAdsEnabled", jsContext.createJSBoolean(organicAdsEnabled));
        jsContext.getGlobalObject().setProperty("organicAdsBoostEnabled", jsContext.createJSBoolean(organicAdsBoostEnabled));
        jsContext.getGlobalObject().setProperty("pureOrganicAdsEnabled", jsContext.createJSBoolean(pureOrganicAdsEnabled));
        jsContext.getGlobalObject().setProperty("currentTime", jsContext.createJSNumber(System.currentTimeMillis()));
        jsContext.getGlobalObject().setProperty("currentPackage", jsContext.createJSString(currPackageName));
        jsContext.getGlobalObject().setProperty("userId", jsContext.createJSString(Process.myUserHandle().toString()));

        JSArray jsAppBlacklist = jsContext.createJSArray();
        List<String> appBlacklist = dnaPreferences_.getBlacklistPackages();
        if (appBlacklist != null) {
            for (int i = 0; i < appBlacklist.size(); i++) {
                jsAppBlacklist.setProperty(i, jsContext.createJSString(appBlacklist.get(i)));
            }
        }
        jsContext.getGlobalObject().setProperty("blacklist", jsAppBlacklist);

        JSObject jsMultiActivityApps = jsContext.createJSObject();
        JSObject jsDualApps = jsContext.createJSObject();
        JSObject jsApps = jsContext.createJSObject();
//        DNALogger.i("DeviceNativeAds: apps size: " + apps.size());
        for (DNAAppRecord app : apps) {
            // Filter out hidden apps for suggested apps source
            if (DNAConstants.SOURCE_SUGGESTED_APPS.equals(source)) {
                boolean isHidden = false;
                
                // Check if component is hidden (more specific check first)
                if (app.component != null && !app.component.isEmpty()) {
                    if (dnaPreferences_.isComponentHidden(app.component, app.userId)) {
                        isHidden = true;
                    }
                }
                
                // Check if package is hidden (fallback or when component is null)
                if (!isHidden && app.packageName != null && !app.packageName.isEmpty()) {
                    if (dnaPreferences_.isPackageHidden(app.packageName, app.userId)) {
                        isHidden = true;
                    }
                }
                
                if (isHidden) {
                    continue; // Skip this app
                }
            }
            
            JSObject jsApp = jsContext.createJSObject();
//            DNALogger.i("DeviceNativeAds: app name: " + app.name + " package: " + app.packageName + " component: " + app.component + " userId: " + app.userId);
            jsApp.setProperty("id", jsContext.createJSString(String.valueOf(app.dbId)));
            jsApp.setProperty("packageName", jsContext.createJSString(app.packageName));
            jsApp.setProperty("appName", jsContext.createJSString(app.name));
            jsApp.setProperty("normAppName", jsContext.createJSString(normalizeText(app.name)));
            if (app.aliases != null && !app.aliases.isEmpty()) {
                jsApp.setProperty("aliases", jsContext.createJSString(app.aliases));
            }
            jsApp.setProperty("userId", jsContext.createJSString(app.userId));
            if (app.component != null) {
                jsApp.setProperty("component", jsContext.createJSString(app.component));
            }
            jsApp.setProperty("timeUsedInLast7D", jsContext.createJSNumber(app.timeUsedInLast7D));
            jsApp.setProperty("timeUsedInLast3D", jsContext.createJSNumber(app.timeUsedInLast3D));
            jsApp.setProperty("timeUsedInLast1D", jsContext.createJSNumber(app.timeUsedInLast1D));
            jsApp.setProperty("timeUsedInLast3Y", jsContext.createJSNumber(app.timeUsedInLast3Y));
            jsApp.setProperty("installedAt", jsContext.createJSNumber(app.installedAt));
            jsApp.setProperty("updatedAt", jsContext.createJSNumber(app.updatedAt));
            jsApp.setProperty("uninstalledAt", jsContext.createJSNumber(app.uninstalledAt));
            jsApp.setProperty("lastTimeInForeground", jsContext.createJSNumber(app.lastTimeInForeground));
            jsApp.setProperty("lastUsedAt", jsContext.createJSNumber(app.lastUsedAt));

            if (jsApps.getProperty(app.packageName).getClass().equals(JSObject.class)) {
                UserHandle currUser = Process.myUserHandle();

                JSArray jsDualApp;
                int length = 1;
                if (jsDualApps.getProperty(app.packageName).getClass().equals(JSArray.class)) {
                    jsDualApp = (JSArray) jsDualApps.getProperty(app.packageName);
                    length = jsDualApp.getLength();
                } else {
                    jsDualApp = jsContext.createJSArray();
                    jsDualApp.setProperty(0, jsApps.getProperty(app.packageName));
                }
                jsDualApp.setProperty(length, jsApp);
                jsDualApps.setProperty(app.packageName, jsDualApp);

                if (app.userId.equals(currUser.toString())) {
                    length = 1;
                    JSArray jsMultiActivityApp;
                    if (jsMultiActivityApps.getProperty(app.packageName).getClass().equals(JSArray.class)) {
                        jsMultiActivityApp = (JSArray) jsMultiActivityApps.getProperty(app.packageName);
                        length = jsMultiActivityApp.getLength();
                    } else {
                        jsMultiActivityApp = jsContext.createJSArray();
                        jsMultiActivityApp.setProperty(0, jsApps.getProperty(app.packageName));
                    }
                    jsMultiActivityApp.setProperty(length, jsApp);
                    jsApps.setProperty(app.packageName, jsApp);
//                    DNALogger.i("DeviceNativeAds: setting multi actibity app name: " + app.name + " package: " + app.packageName + " component: " + app.component + " userId: " + app.userId + " length: " + length);
                    jsMultiActivityApps.setProperty(app.packageName, jsMultiActivityApp);
                }
            } else {
                jsApps.setProperty(app.packageName, jsApp);
            }
        }
        jsContext.getGlobalObject().setProperty("apps", jsApps);
        jsContext.getGlobalObject().setProperty("dualApps", jsDualApps);
        jsContext.getGlobalObject().setProperty("multiActivityApps", jsMultiActivityApps);

        JSArray jsAds = jsContext.createJSArray();
//        DNALogger.i("DeviceNativeAds: ads size: " + ads.size());
        int i = 0;
        if (adsEnabled) {
            for (DNAAdRecord ad : ads) {
                JSObject jsAd = jsContext.createJSObject();
                jsAd.setProperty("id", jsContext.createJSString(String.valueOf(ad.adId)));
                jsAd.setProperty("packageName", jsContext.createJSString(ad.packageName));
                jsAd.setProperty("appName", jsContext.createJSString(ad.name));
                jsAd.setProperty("normAppName", jsContext.createJSString(normalizeText(ad.name)));
                jsAd.setProperty("type", jsContext.createJSString(!TextUtils.isEmpty(ad.type) ? ad.type : ""));
                jsAd.setProperty("creativeSource", jsContext.createJSString(!TextUtils.isEmpty(ad.creativeSource) ? ad.creativeSource : ""));
                jsAd.setProperty("destinationUrl", jsContext.createJSString(!TextUtils.isEmpty(ad.destinationUrl) ? ad.destinationUrl : ""));
                jsAd.setProperty("viewThrough", jsContext.createJSNumber(ad.viewThrough));
                jsAd.setProperty("notPlayStore", jsContext.createJSBoolean(ad.notPlayStore));
                jsAd.setProperty("title", jsContext.createJSString(!TextUtils.isEmpty(ad.title) ? ad.title : ad.name));
                jsAd.setProperty("description", jsContext.createJSString(!TextUtils.isEmpty(ad.description) ? ad.description : ""));
                jsAd.setProperty("iconUrl", jsContext.createJSString(!TextUtils.isEmpty(ad.iconUrl) ? ad.iconUrl : ""));
                jsAd.setProperty("clickUrl", jsContext.createJSString(!TextUtils.isEmpty(ad.clickUrl) ? ad.clickUrl : ""));
                jsAd.setProperty("destinationUrl", jsContext.createJSString(!TextUtils.isEmpty(ad.destinationUrl) ? ad.destinationUrl : ""));
                jsAd.setProperty("impressionUrl", jsContext.createJSString(!TextUtils.isEmpty(ad.impressionUrl) ? ad.impressionUrl : ""));
                jsAd.setProperty("viewThrough", jsContext.createJSNumber(ad.viewThrough));
                jsAd.setProperty("targetingMetadata", jsContext.createJSString(!TextUtils.isEmpty(ad.targetingMetadata) ? ad.targetingMetadata : "{}"));
                jsAds.setProperty(i, jsAd);
                i++;
            }
        }
        jsContext.getGlobalObject().setProperty("ads", jsAds);

        JSObject jsEvents = jsContext.createJSObject();
//        DNALogger.i("DeviceNativeAds: adEvents size: " + adEvents.size());
        for (DNAResultEventRecord adRecord : adEvents) {
            JSObject jsAdRecord = jsContext.createJSObject();
            jsAdRecord.setProperty("resId", jsContext.createJSString(String.valueOf(adRecord.resId)));
            jsAdRecord.setProperty("packageName", jsContext.createJSString(adRecord.packageName));
            jsAdRecord.setProperty("event", jsContext.createJSString(adRecord.event));
            jsAdRecord.setProperty("type", jsContext.createJSString(adRecord.type));
            jsAdRecord.setProperty("placement", jsContext.createJSString(adRecord.placement));
            jsAdRecord.setProperty("completedAt", jsContext.createJSNumber(adRecord.completedAt));

            int j = 0;
            JSValue currPackageList = jsEvents.getProperty(adRecord.packageName);
            if (currPackageList.getClass().equals(JSUndefined.class)) {
                currPackageList = jsContext.createJSArray();
                jsEvents.setProperty(adRecord.packageName, currPackageList);
            }
            j = ((JSArray)currPackageList).getLength();
            ((JSArray)currPackageList).setProperty(j, jsAdRecord);
        }
        jsContext.getGlobalObject().setProperty("events", jsEvents);

        JSObject jsShortcuts = jsContext.createJSObject();
//        DNALogger.i("DeviceNativeAds: shortcuts size: " + shortcuts.size());
//        for (DNAShortcutRecord shortcut : shortcuts) {
//            JSObject jsShortcut = jsContext.createJSObject();
//            jsShortcut.setProperty("id", jsContext.createJSString(String.valueOf(shortcut.dbId)));
//            jsShortcut.setProperty("packageName", jsContext.createJSString(shortcut.packageName));
//            jsShortcut.setProperty("name", jsContext.createJSString(shortcut.name));
//            jsShortcut.setProperty("description", jsContext.createJSString(!TextUtils.isEmpty(shortcut.description) ? shortcut.description : ""));
//            jsShortcut.setProperty("intentUri", jsContext.createJSString(shortcut.intentUri));
//            jsShortcut.setProperty("source", jsContext.createJSNumber(shortcut.source));
//            jsShortcut.setProperty("rank", jsContext.createJSNumber(shortcut.rank));
//            jsShortcut.setProperty("updatedAt", jsContext.createJSNumber(shortcut.updatedAt));
//
//            int j = 0;
//            JSValue currPackageList = jsShortcuts.getProperty(shortcut.packageName);
//            if (currPackageList.getClass().equals(JSUndefined.class)) {
//                currPackageList = jsContext.createJSArray();
//                jsShortcuts.setProperty(shortcut.packageName, currPackageList);
//            }
//            j = ((JSArray)currPackageList).getLength();
//            ((JSArray)currPackageList).setProperty(j, jsShortcut);
//        }
        jsContext.getGlobalObject().setProperty("shortcuts", jsShortcuts);

        JSObject jsNotifs = jsContext.createJSObject();
//        DNALogger.i("DeviceNativeAds: notifs size: " + notifs.size());
//        for (DNANotifRecord notif : notifs) {
//            JSObject jsNotif = jsContext.createJSObject();
//            jsNotif.setProperty("id", jsContext.createJSString(String.valueOf(notif.dbId)));
//            jsNotif.setProperty("packageName", jsContext.createJSString(notif.packageName));
//            jsNotif.setProperty("name", jsContext.createJSString(notif.name));
//            jsNotif.setProperty("description", jsContext.createJSString(!TextUtils.isEmpty(notif.description) ? notif.description : ""));
//            jsNotif.setProperty("sentAt", jsContext.createJSNumber(notif.sentAt));
//
//            int j = 0;
//            JSValue currPackageList = jsNotifs.getProperty(notif.packageName);
//            if (currPackageList.getClass().equals(JSUndefined.class)) {
//                currPackageList = jsContext.createJSArray();
//                jsNotifs.setProperty(notif.packageName, currPackageList);
//            }
//            j = ((JSArray)currPackageList).getLength();
//            ((JSArray)currPackageList).setProperty(j, jsNotif);
//
//        }
        jsContext.getGlobalObject().setProperty("notifs", jsNotifs);

        apps.clear();
        ads.clear();
        adEvents.clear();
    }

}
