package com.devdigital.smartlock;

import android.app.Activity;
import android.content.Intent;
import android.content.IntentSender;
import android.support.annotation.NonNull;
import android.util.Log;
import android.widget.Toast;

import com.google.android.gms.auth.api.credentials.Credential;
import com.google.android.gms.auth.api.credentials.CredentialRequest;
import com.google.android.gms.auth.api.credentials.CredentialRequestResponse;
import com.google.android.gms.auth.api.credentials.Credentials;
import com.google.android.gms.auth.api.credentials.CredentialsClient;
import com.google.android.gms.common.api.ApiException;
import com.google.android.gms.common.api.ResolvableApiException;
import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.Task;

/**
 * Auto Save/Fill password having SDK lower than oreo.
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 20 March 2018
 */
public class SmartLockManager {

    private static final String TAG = SmartLockManager.class.getSimpleName();
    private static final int RC_READ = 456;
    private static final int RC_SAVE = 458;

    /**
     * Activity Instance
     */
    private final Activity mActivity;
    private final CredentialsClient mCredentialsClient;
    private CredentialListener mCredentialListener;

    public SmartLockManager(Activity activity) {
        this.mActivity = activity;
        this.mCredentialsClient = Credentials.getClient(activity);
    }

    /**
     * Listen to Credential info
     *
     * @param credentialListener CredentialListener object
     */
    public void setCredentialListener(CredentialListener credentialListener) {
        this.mCredentialListener = credentialListener;
    }

    /**
     * Initiate get Credentials Request
     */
    public void getCredentialsAsync() {
        CredentialRequest mCredentialRequest = new CredentialRequest.Builder()
                .setPasswordLoginSupported(true)
             //   .setAccountTypes(IdentityProviders.TWITTER)
             //   .setAccountTypes("https://accounts.google.com")
                .build();

        mCredentialsClient.request(mCredentialRequest).addOnCompleteListener(
                new OnCompleteListener<CredentialRequestResponse>() {
                    @Override
                    public void onComplete(@NonNull Task<CredentialRequestResponse> task) {
                        if (task.isSuccessful()) {
                            // See "Handle successful credential requests"
                            onCredentialRetrieved(task.getResult().getCredential());
                            return;
                        }

                        Exception e = task.getException();
                        if (e instanceof ResolvableApiException) {
                            // This is most likely the case where the user has multiple saved
                            // credentials and needs to pick one. This requires showing UI to
                            // resolve the read request.
                            try {
                                ResolvableApiException rae = (ResolvableApiException) e;
                                rae.startResolutionForResult(mActivity, RC_READ);
                            } catch (IntentSender.SendIntentException ex) {
                                Log.e(TAG, "Failed to send resolution.", ex);
                            }
                        } else if (e instanceof ApiException) {
                            // The user must create an account or sign in manually.
                            Log.e(TAG, "Unsuccessful credential request.", e);

                            ApiException ae = (ApiException) e;
                            int code = ae.getStatusCode();

                        }
                    }
                });
    }


    /**
     * Handle Activity Result for Get/Save Credential
     *
     * @param requestCode int value which is used to start the activity
     * @param resultCode Determine whether request is failure or success
     * @param data Activity Result data
     */
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode == RC_READ) {
            if (resultCode == Activity.RESULT_OK) {
                Credential credential = data.getParcelableExtra(Credential.EXTRA_KEY);
                onCredentialRetrieved(credential);
            } else {
                Log.e(TAG, "Credential Read: NOT OK");
            }
        } else if (requestCode == RC_SAVE) {
            if (resultCode != Activity.RESULT_OK) {
                Log.e(TAG, "Save Credentials Failed");
            }
        }
    }

    /**
     * Retrieve Credential Info
     *
     * @param credential Credential object
     */
    private void onCredentialRetrieved(Credential credential) {
        if (mCredentialListener != null)
            mCredentialListener.onCredential(credential);

        Log.e("credential", credential.getId()+"=>"+credential.getPassword());

        /*String accountType = credential.getAccountType();
        if (accountType == null) {
            // Sign the user in with information from the Credential.
            Log.e("credential", credential.getId()+"=>"+credential.getPassword());
        } else if (accountType.equals(IdentityProviders.GOOGLE)) {
            // The user has previously signed in with Google Sign-In. Silently
            // sign in the user with the same ID.
            // See https://developers.google.com/identity/sign-in/android/
            GoogleSignInOptions gso =
                    new GoogleSignInOptions.Builder(GoogleSignInOptions.DEFAULT_SIGN_IN)
                            .requestEmail()
                            .build();

            GoogleSignInClient signInClient = GoogleSignIn.getClient(mActivity, gso);
            Task<GoogleSignInAccount> task = signInClient.silentSignIn();
            task.addOnCompleteListener(mActivity, new OnCompleteListener<GoogleSignInAccount>(){
                @Override
                public void onComplete(@NonNull Task task) {
                    if(task.isSuccessful()){

                    }
                }
            });
        }*/
    }

    /**
     * Save new credentials
     *
     * @param name name of Credentials
     * @param email username of Credentials
     * @param password password of Credentials
     */
    public void saveCredentials(String name, String email, String password) {
        Credential credential = new Credential.Builder(email)
                .setPassword(password)
                .setName(name)
                .build();

        mCredentialsClient.save(credential).addOnCompleteListener(new OnCompleteListener<Void>() {
            @Override
            public void onComplete(@NonNull Task<Void> task) {
                if (task.isSuccessful()) {
                    Log.d(TAG, "SAVE: OK");
                    return;
                }

                Exception e = task.getException();
                if (e instanceof ResolvableApiException) {
                    // Try to resolve the save request. This will prompt the user if
                    // the credential is new.
                    ResolvableApiException rae = (ResolvableApiException) e;
                    try {
                        rae.startResolutionForResult(mActivity, RC_SAVE);
                    } catch (IntentSender.SendIntentException ex) {
                        // Could not resolve the request
                        Log.e(TAG, "Failed to send resolution.", ex);
                        //Toast.makeText(mActivity, "Failed to save credentials", Toast.LENGTH_SHORT).show();
                    }
                } else {
                    // Request has no resolution
                    //Toast.makeText(mActivity, "Failed to save credentials", Toast.LENGTH_SHORT).show();
                }
            }
        });
    }

    /**
     * Listen to Credential
     */
    public interface CredentialListener {
        /**
         * @param credential Credential object
         */
        void onCredential(Credential credential);
    }

}
