package com.devdigital.networklib;

import android.content.Context;
import android.os.Bundle;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.gsonparserfactory.GsonParserFactory;
import com.androidnetworking.interceptors.HttpLoggingInterceptor;
import com.devdigital.networklib.handler.BaseWebResponseHandler;
import com.devdigital.networklib.handler.RefreshTokenManager;
import com.devdigital.networklib.ssl.Tls12SocketFactory;
import com.devdigital.networklib.stack.AndroidNetworkingStack;
import com.devdigital.networklib.stack.INetworkStack;
import com.facebook.stetho.Stetho;
import com.facebook.stetho.okhttp3.StethoInterceptor;

import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;

/**
 * Network Lib module dependency injector
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 22 November 2017
 */
public class NetworkController {

    /**
     * Default Network Stack
     */
    public static final Class<? extends INetworkStack> NETWORK_STACK_CLASS = AndroidNetworkingStack.class;

    private Class<? extends BaseWebResponseHandler> mWebResponseHandlerClass;

    private static NetworkController mNetworkController;
    private boolean mIsRefreshTokenSupported;
    private RefreshTokenManager mRefreshTokenManager;
    private boolean isDebuggable;

    private NetworkController(){

    }

    public static NetworkController getInstance() {
        if(mNetworkController==null){
            synchronized (NetworkController.class){
                if(mNetworkController==null){
                    mNetworkController = new NetworkController();
                }
            }
        }

        return mNetworkController;
    }

    /**
     * Call this method once from main {@link android.app.Application} call once to
     * initialize Stetho and AndroidNetworking library
     * <p>
     * To Use Stetho open  chrome://inspect/#devices in Chrome Browser
     *
     * @param context      {@link Context} object
     * @param isDebuggable boolean, if true then Stetho is enable else Stetho is disabled.
     */
    public NetworkController initAndroidNetworking(Context context, boolean isDebuggable) {
        this.isDebuggable = isDebuggable;
        if (isDebuggable) {
            //Initialize Stetho
            Stetho.initializeWithDefaults(context);
        }

        OkHttpClient.Builder mClientBuilder = new OkHttpClient.Builder()
                .followRedirects(true)
                .followSslRedirects(true)
                .retryOnConnectionFailure(true)
                .cache(null)
                .connectTimeout(5, TimeUnit.SECONDS)
                .writeTimeout(5, TimeUnit.SECONDS)
                .readTimeout(5, TimeUnit.SECONDS);

        OkHttpClient.Builder builder = Tls12SocketFactory.enableTls12OnPreLollipop(mClientBuilder);

        if (isDebuggable) {
            //Intercept Network Request with Stetho
            builder.addNetworkInterceptor(new StethoInterceptor());
        }
        //Set Read Timeout as 1 Minute
        builder.readTimeout(1, TimeUnit.MINUTES);
        //Set Connection Timeout as 10 Second
        builder.connectTimeout(10, TimeUnit.SECONDS);
        OkHttpClient okHttpClient = builder.build();

        //Set OkHttpClient as HTTP stack
        AndroidNetworking.initialize(context, okHttpClient);

        //Add GSON as JSON Parser
        AndroidNetworking.setParserFactory(new GsonParserFactory());

        if (isDebuggable) {
            AndroidNetworking.enableLogging(HttpLoggingInterceptor.Level.BODY);
        }
        return this;
    }


    /**
     * @param cls BaseWebResponseHandler implementation
     */
    public void setWebResponseHandler(Class<? extends BaseWebResponseHandler> cls) {
        mWebResponseHandlerClass = cls;
    }

    public Class<? extends BaseWebResponseHandler> getWebResponseHandler() {
        return mWebResponseHandlerClass;
    }


    public NetworkController setRefreshTokenSupported(){
        mIsRefreshTokenSupported = true;
        return this;
    }

    public boolean isRefreshTokenSupported() {
        return mIsRefreshTokenSupported;
    }

    public void setRefreshTokenManager(RefreshTokenManager manager) {
        this.mRefreshTokenManager = manager;
    }

    public RefreshTokenManager getRefreshTokenManager() {
        return mRefreshTokenManager;
    }

    private HttpLoggingInterceptor.Level mLogLevel;
    public void setLogLevel(HttpLoggingInterceptor.Level level){
        if(mLogLevel==null || !mLogLevel.equals(level)) {
            AndroidNetworking.enableLogging(level);
        }
    }

}
