package com.devdigital.networklib.handler;

import android.os.Handler;

import com.devdigital.networklib.NetworkController;
import com.devdigital.networklib.constants.NetworkError;
import com.devdigital.networklib.constants.ResponseStatus;
import com.devdigital.networklib.entity.BaseEntity;
import com.devdigital.networklib.json.JSONFactory;
import com.devdigital.networklib.listener.IWebResponseListener;
import com.devdigital.networklib.listener.RefreshTokenResponseListener;
import com.devdigital.networklib.model.NetworkStackResponse;
import com.devdigital.networklib.stack.INetworkStack;
import com.devdigital.networklib.task.JSONArrayTask;
import com.devdigital.networklib.task.JSONObjectTask;
import com.devdigital.networklib.task.ObjectTask;
import com.devdigital.networklib.task.StringTask;
import com.devdigital.networklib.task.Task;
import com.devdigital.networklib.utils.NetworkUtils;
import com.google.gson.JsonParseException;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Map;

/**
 * Implementation for {@link IWebRequestExecutor}
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 21 November 2017
 */
public class WebRequestExecutor implements IWebRequestExecutor, IWebResponseListener {

    private WebRequestBuilder mWebRequestBuilder;
    private INetworkStack mNetworkStack;
    private BaseWebResponseHandler mWebResponseHandler;
    private NetworkController mNetworkController;

    private Class mResponseClass;
    private Task mTask;

    public WebRequestExecutor(WebRequestBuilder builder) {
        this.mWebRequestBuilder = builder;

        mNetworkController = NetworkController.getInstance();

        newNetworkStackInstance();

        try {
            mWebResponseHandler = mNetworkController.getWebResponseHandler().newInstance();
            mWebResponseHandler.setWebRequestBuilder(builder);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void newNetworkStackInstance(){
        try {
            //Using reflection get Network Instance class.
            Constructor<?> ctor = mWebRequestBuilder.getNetworkStack().getConstructor(WebRequestBuilder.class);
            mNetworkStack = (INetworkStack) ctor.newInstance(mWebRequestBuilder);
            mNetworkStack.setResponseListener(this);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public Task<JSONObject> getAsJsonObjectAsync() {
        mTask = new JSONObjectTask<JSONObject>();
        getStringAsync();
        return mTask;
    }

    @Override
    public Task<JSONArray> getAsJsonArrayAsync() {
        mTask = new JSONArrayTask<JSONArray>();
        getStringAsync();
        return mTask;
    }

    @Override
    public Task<BaseEntity> getAsObjectAsync() {
        return getAsObjectAsync(BaseEntity.class);
    }

    @Override
    public <T>Task<T> getAsObjectAsync(Class<T> cls) {
        mTask = new ObjectTask<Object>();
        mResponseClass = cls;
        getStringAsync();
        return mTask;
    }

    @Override
    public NetworkStackResponse<JSONObject> getAsJsonObject()  {
        return getResponse(getAsString(), JSONObject.class);
    }

    @Override
    public NetworkStackResponse<JSONArray> getAsJsonArray() {
        return getResponse(getAsString(), JSONArray.class);
    }

    @Override
    public NetworkStackResponse<BaseEntity> getAsObject() {
        return getAsObject(BaseEntity.class);
    }

    @Override
    public <T> NetworkStackResponse<T> getAsObject(Class<T> cls) {
        return getResponse(getAsString(), cls);
    }

    @Override
    public Task<String> getAsStringAsync() {
        mTask = new StringTask<String>();
        getStringAsync();
        return mTask;
    }

    private void getStringAsync() {
        if (!NetworkUtils.isNetworkAvailable(mWebRequestBuilder.getContext())) {
            new Handler().postDelayed(() -> {
                NetworkStackResponse response = getNetworkErrorResponse();
                onResponse(ResponseStatus.SERVER_ERROR, response);
            }, 250);
        }else{
            mWebResponseHandler.showProgressDialog();
            mNetworkStack.getAsync();
        }
    }

    @Override
    public NetworkStackResponse<String> getAsString()  {
        if (!NetworkUtils.isNetworkAvailable(mWebRequestBuilder.getContext())) {
            NetworkStackResponse<String> response = new NetworkStackResponse<>();
            response.setSuccess(false);
            response.setError(NetworkError.NO_NETWORK_ERROR);
            return response;
        }

        NetworkStackResponse<String> response = mNetworkStack.getSync();
        return getResponse(response, String.class);
    }

    /**
     * Handle response of getAsync Method
     *
     * @param responseStatus ResponseStatus object
     * @param result         NetworkStackResponse object
     */
    @Override
    public void onResponse(ResponseStatus responseStatus, Object result) {
        mWebResponseHandler.hideProgressDialog();
        NetworkStackResponse<String> response = (NetworkStackResponse<String>) result;

        if(mNetworkController.isRefreshTokenSupported() && mNetworkController.getRefreshTokenManager().isTokenRefreshRequired(response)){
            handleTokenExpiry(response);
            return;
        }

        publishTaskResponse(response);
    }

    private void publishTaskResponse(NetworkStackResponse<String> response) {
        if(mTask == null || mTask instanceof StringTask){
            publishResponse(response, String.class);
        }else if(mTask instanceof ObjectTask){
            publishResponse(response, mResponseClass);
        }else if(mTask instanceof JSONObjectTask){
            publishResponse(response, JSONObject.class);
        }else if(mTask instanceof JSONArrayTask){
            publishResponse(response, JSONArray.class);
        }
    }

    private void handleTokenExpiry(NetworkStackResponse<String> response) {
        mNetworkController.getRefreshTokenManager()
                .setRefreshTokenResponseListener(new RefreshTokenResponseListener() {
                    @Override
                    public void onResponse(boolean status, Map<String, Object> headers) {
                        if(status){
                            Map<String, Object> header = mWebRequestBuilder.getHeader();
                            if(header==null) header = new HashMap<>();
                            header.putAll(headers);
                            mWebRequestBuilder.setHeader(headers);
                            newNetworkStackInstance();
                            getStringAsync();
                        } else {
                            publishTaskResponse(response);
                        }
                    }
                })
                .refreshAsync(mWebRequestBuilder.getContext());
    }

    private void publishResponse(NetworkStackResponse<String> response, Class cls) {
        NetworkStackResponse finalResponse = response;
        if(cls != String.class) finalResponse = getResponse(response, cls);
        if(finalResponse.isSuccess()) {
            mTask.publishResult(finalResponse.getResult());
        } else {
            mTask.publishError(finalResponse);
        }
    }

    private NetworkStackResponse getNetworkErrorResponse() {
        //Show error message if network not available
        NetworkStackResponse response = new NetworkStackResponse();
        response.setError(NetworkError.NO_NETWORK_ERROR);
        response.setSuccess(false);
        return response;
    }

    private <T> NetworkStackResponse<T> getResponse(NetworkStackResponse<String> response, Class<T> t) {
        NetworkStackResponse<T> stackResponse = new NetworkStackResponse<>();
        stackResponse.setError(response.getError());
        stackResponse.setSuccess(response.isSuccess());
        stackResponse.setStatus(response.getStatus());
        stackResponse.setHeaders(response.getHeaders());

        try {
            if(response.isSuccess()) {
                if (t == JSONObject.class) {
                    stackResponse.setResult((T) new JSONObject(response.getResult()));
                } else if (t == JSONArray.class) {
                    stackResponse.setResult((T) new JSONArray(response.getResult()));
                } else if(t!=String.class){
                    stackResponse.setResult((T) JSONFactory.getInstance().fromJson(response.getResult(), t));
                }
            }
        } catch (JSONException | JsonParseException ex) {
            ex.printStackTrace();
            stackResponse.setSuccess(false);
            stackResponse.setError(NetworkError.INVALID_JSON_ERROR);
        }

        if(mWebRequestBuilder.isHandleError()){
            if(stackResponse.isSuccess()){
                boolean status;
                if(t == String.class){
                    status = mWebResponseHandler.handleStringResponse((String) stackResponse.getResult());
                } else if (t == JSONObject.class) {
                    status = mWebResponseHandler.handleJSONObjectResponse((JSONObject) stackResponse.getResult());
                } else if (t == JSONArray.class) {
                    status = mWebResponseHandler.handleJSONArrayResponse((JSONArray) stackResponse.getResult());
                } else {
                    status = mWebResponseHandler.handleObjectResponse(stackResponse.getResult());
                }
                if(!status){
                    stackResponse.setSuccess(false);
                    stackResponse.setError(NetworkError.RESPONSE_HANDLING_ERROR);
                    return stackResponse;
                }
            }else{
                mWebResponseHandler.handleError(response);
            }
        }

        return stackResponse;
    }

}
