package com.devdigital.networklib.handler;

import android.content.Context;
import android.support.annotation.NonNull;
import android.view.View;

import com.devdigital.networklib.NetworkController;
import com.devdigital.networklib.entity.BaseEntity;
import com.devdigital.networklib.json.JSONFactory;
import com.devdigital.networklib.listener.IErrorDialogClickListener;
import com.devdigital.networklib.listener.IWebResponseListener;
import com.devdigital.networklib.stack.INetworkStack;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.Collections;
import java.util.Map;

/**
 * Model class to pass data to {@link IWebRequestExecutor} or {@link INetworkStack}
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 21 November 2017
 */
public class WebRequestBuilder {

    /**
     * Supported Request method.
     */
    public enum RequestMethod {
        GET,
        POST,
        PUT,
        DELETE,
        MULTI_PART
    }

    /**
     * Application Context.
     * <p>
     * If showProgressDialog or showErrorDialog is true then context must be instance of {@link android.app.Activity}
     */
    private Context context;

    /**
     * Request URL
     */
    private String url;

    /**
     * Request method.
     * currently supporting GET and POST only.
     */
    private RequestMethod requestMethod;

    /**
     * Request Header
     */
    private Map<String, Object> header;

    /**
     * JSON body for POST request
     */
    private JSONObject requestJSON;

    /**
     * Request param for GET or POST request
     */
    private Map<String, Object> requestParams;

    /**
     * File Param for multipart Request
     */
    private Map<String, File> multipartFile;

    /**
     * String body for POST request
     */
    private String requestBody;

    /**
     * If set to true then progress dialog will be visible
     * <p>
     * Field will be ignored for getSync request call
     */
    private boolean showProgressDialog;

    /**
     * Set View to be visible while executing request.
     * This has higher priority then progress dialog
     * <p>
     * Field will be ignored for getSync request call
     */
    private View progressView;

    /**
     * Message to be show on Progress dilog
     */
    private String progressMessage;

    /**
     * If set to true then only error will be handled
     * else error will be ignored.
     */
    private boolean handleError;

    /**
     * handleError will handle the error and it will perform action based on the error code.
     * <p>
     * But if this field is set then only Error dialog will be visible event if handle error is true.s
     */
    private boolean showErrorDialog;

    /**
     * Provide network stack on which HTTP request needs to be executed.
     */
    private Class<? extends INetworkStack> networkStack;

    /**
     * If true the debug log will be enabled
     */
    private boolean debug;

    /**
     * Request Identifier
     */
    private String tag;

    /**
     * Add Error Dialog click listener
     */
    private IErrorDialogClickListener errorDialogClickListener;

    public WebRequestBuilder(Context context, String url) {
        this.context = context;
        this.requestMethod = RequestMethod.GET;
        this.networkStack  = NetworkController.NETWORK_STACK_CLASS;
        this.requestParams = Collections.emptyMap();
        this.header = Collections.emptyMap();
        this.url = url;
    }

    public WebRequestBuilder setHeader(Map<String, Object> header) {
        this.header = header;
        return this;
    }

    public WebRequestBuilder setRequestMethod(RequestMethod requestMethod) {
        this.requestMethod = requestMethod;
        return this;
    }

    public WebRequestBuilder setRequestJSON(Map<String, Object> requestParams) {
        try {
            this.requestJSON = new JSONObject(JSONFactory.getInstance().getJson(requestParams));
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return this;
    }

    public WebRequestBuilder setRequestParams(Map<String, Object> requestParams) {
        this.requestParams = requestParams;
        return this;
    }

    public WebRequestBuilder setHandleError(boolean handleError) {
        this.handleError = handleError;
        return this;
    }

    public WebRequestBuilder setShowProgressDialog(boolean showProgressDialog) {
        this.showProgressDialog = showProgressDialog;
        return this;
    }

    public WebRequestBuilder setShowErrorDialog(boolean showErrorDialog) {
        this.showErrorDialog = showErrorDialog;
        return this;
    }

    public WebRequestBuilder setErrorDialogClickListener(IErrorDialogClickListener errorDialogClickListener) {
        this.errorDialogClickListener = errorDialogClickListener;
        return this;
    }

    public View getProgressView() {
        return progressView;
    }

    public WebRequestBuilder setProgressView(View progressView) {
        this.progressView = progressView;
        return this;
    }

    public String getProgressMessage() {
        return progressMessage;
    }

    public WebRequestBuilder setProgressMessage(String progressMessage) {
        this.progressMessage = progressMessage;
        return this;
    }

    public boolean isDebug() {
        return debug;
    }

    public WebRequestBuilder setDebug(boolean debug) {
        this.debug = debug;
        return this;
    }

    public Map<String, File> getMultipartFile() {
        return multipartFile;
    }

    public WebRequestBuilder setMultipartFile(Map<String, File> multipartFile) {
        this.multipartFile = multipartFile;
        return this;
    }

    public Context getContext() {
        return context;
    }

    public String getUrl() {
        return url;
    }

    public RequestMethod getRequestMethod() {
        return requestMethod;
    }

    public Map<String, Object> getHeader() {
        return header;
    }

    public JSONObject getRequestJSON() {
        return requestJSON;
    }

    public Map<String, Object> getRequestParams() {
        return requestParams;
    }

    public boolean isHandleError() {
        return handleError;
    }

    public boolean isShowProgressDialog() {
        return showProgressDialog;
    }

    public boolean isShowErrorDialog() {
        return showErrorDialog;
    }

    public String getRequestBody() {
        return requestBody;
    }

    public WebRequestBuilder setRequestBody(String requestBody) {
        this.requestBody = requestBody;
        return this;
    }

    public Class<? extends INetworkStack> getNetworkStack() {
        return networkStack;
    }

    public WebRequestBuilder setNetworkStack(@NonNull Class<? extends INetworkStack> networkStack) {
        this.networkStack = networkStack;
        return this;
    }

    public IErrorDialogClickListener getErrorDialogClickListener() {
        return errorDialogClickListener;
    }

    public String getTag() {
        return tag;
    }

    public WebRequestBuilder setTag(String tag) {
        this.tag = tag;
        return this;
    }

    public WebRequestExecutor build() {
        //URL must start with http.
        if (!url.startsWith("http")) {
            throw new RuntimeException("URL must be valid to perform any action");
        }
        return new WebRequestExecutor(this);
    }

    @Override
    public String toString() {
        return "WebRequestBuilder{" +
                "context=" + context +
                ", url='" + url + '\'' +
                ", requestMethod=" + requestMethod +
                ", header=" + header +
                ", requestJSON=" + requestJSON +
                ", requestParams=" + requestParams +
                ", multipartFile=" + multipartFile +
                ", requestBody='" + requestBody + '\'' +
                ", showProgressDialog=" + showProgressDialog +
                ", progressView=" + progressView +
                ", progressMessage='" + progressMessage + '\'' +
                ", handleError=" + handleError +
                ", showErrorDialog=" + showErrorDialog +
                ", networkStack=" + networkStack +
                ", debug=" + debug +
                ", tag='" + tag + '\'' +
                ", errorDialogClickListener=" + errorDialogClickListener +
                '}';
    }
}
