package com.devdigital.networklib.handler;

import android.text.TextUtils;
import android.util.Log;

import com.devdigital.networklib.NetworkController;
import com.devdigital.networklib.entity.BaseEntity;
import com.devdigital.networklib.exception.NetworkError;
import com.devdigital.networklib.exception.WebRequestException;
import com.devdigital.networklib.json.JSONFactory;
import com.devdigital.networklib.listener.IWebResponseListener;
import com.devdigital.networklib.stack.AndroidNetworkingStack;
import com.devdigital.networklib.stack.INetworkStack;
import com.devdigital.networklib.stack.NetworkStackResponse;
import com.devdigital.networklib.utils.NetworkUtils;

import java.lang.reflect.Constructor;

/**
 * Implementation for {@link IWebRequestExecutor}
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 21 November 2017
 */
public class WebRequestExecutor implements IWebRequestExecutor, IWebResponseListener {

    /**
     * Default Network Stack
     */
    private static final Class<? extends INetworkStack> NETWORK_STACK_CLASS = AndroidNetworkingStack.class;

    private WebRequestBuilder mWebRequestBuilder;
    private IWebResponseListener mIWebResponseListener;
    private INetworkStack mNetworkStack;
    private BaseWebResponseHandler mWebResponseHandler;

    public WebRequestExecutor(WebRequestBuilder builder) {
        this.mWebRequestBuilder = builder;

        Class<? extends INetworkStack> networkStackCls = NETWORK_STACK_CLASS;
        if(mWebRequestBuilder.getNetworkStack()!=null){
            networkStackCls = mWebRequestBuilder.getNetworkStack();
        }

        //Swap response listener. inorder to listen to response in this class.
        mIWebResponseListener = mWebRequestBuilder.getResponseListener();
        mWebRequestBuilder.setResponseListener(this);

        try {
            //Using reflection get Network Instance class.
            Constructor<?> ctor = networkStackCls.getConstructor(WebRequestBuilder.class);
            mNetworkStack = (INetworkStack) ctor.newInstance(mWebRequestBuilder);
        } catch (Exception e) {
            e.printStackTrace();
        }

        try {
            mWebResponseHandler = NetworkController.getWebResponseHandler().newInstance();
            mWebResponseHandler.setWebRequestBuilder(mWebRequestBuilder);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void getAsync() {
        if(!NetworkUtils.isNetworkAvailable(mWebRequestBuilder.getContext())){
            //Show error message if network not available
            NetworkStackResponse response = new NetworkStackResponse();
            response.setError(NetworkError.NO_NETWORK_ERROR);
            response.setSuccess(false);

            if(mWebRequestBuilder.isHandleError()){
                mWebResponseHandler.handleNetworkError();
            }

            if(mIWebResponseListener!=null){
                mIWebResponseListener.onResponse(ResponseStatus.HTTP_ERROR, response);
            }

            return;
        }

        if(mWebRequestBuilder.isDebug()) {
            Log.w("Web_Service", mWebRequestBuilder.toString());
        }

        mWebResponseHandler.showProgressDialog();

        mNetworkStack.getAsync();
    }

    @Override
    public BaseEntity get() throws WebRequestException {
        if(!NetworkUtils.isNetworkAvailable(mWebRequestBuilder.getContext())){
            //Show error message if network not available
            mWebResponseHandler.handleNetworkError();
            throw new WebRequestException(NetworkError.NO_NETWORK_ERROR);
        }

        if(mWebRequestBuilder.isDebug()) {
            Log.w("Web_Service", mWebRequestBuilder.toString());
        }

        NetworkStackResponse networkResponse =  mNetworkStack.getSync();

        Object entity = handleResponse(networkResponse, true);
        return (BaseEntity) entity;
    }

    /**
     *  Handle response of getAsync Method
     *
     * @param responseStatus ResponseStatus object
     * @param result NetworkStackResponse object
     */
    @Override
    public void onResponse(ResponseStatus responseStatus, Object result) {
        NetworkStackResponse networkResponse = (NetworkStackResponse) result;
        mWebResponseHandler.hideProgressDialog();
        try {
            handleResponse(networkResponse, false);
        } catch (WebRequestException e) {
            e.printStackTrace();
        }
    }

    private Object handleResponse(NetworkStackResponse networkResponse, boolean sync) throws WebRequestException {
        if(mWebRequestBuilder.isDebug()) {
            Log.i("Response", "HTTP Response is " + networkResponse.toString());
        }
        Object baseEntity = null;
        if(networkResponse.isSuccess()) {
            String response = networkResponse.getResult();
            boolean isJsonError  = false;
            if (!TextUtils.isEmpty(response)) {
                try {
                    //Get Entity from json
                    baseEntity = JSONFactory.getInstance().fromJson(response, mWebRequestBuilder.getEntityClass());
                }catch (Exception e){
                    isJsonError = true;
                    e.printStackTrace();
                    //If Exception occure in above call than json might be of error. Parse it again in BaseEnity
                    //Because above error can be occured as error json doesn't have "values" field.
                    try {
                        //Get Entity from json
                        baseEntity = JSONFactory.getInstance().fromJson(response, BaseEntity.class);
                    }catch (Exception ex){
                        ex.printStackTrace();
                    }
                }
            }

            boolean status;
            if (mWebRequestBuilder.isHandleError()) {
                if(isJsonError){
                    status = false;
                    networkResponse.setSuccess(false);
                    networkResponse.setError(NetworkError.INVALID_JSON_ERROR);
                    networkResponse.setResult(response);
                    mWebResponseHandler.handleResponse(networkResponse);
                } else {
                    status = mWebResponseHandler.handleResponse((BaseEntity) baseEntity);
                }
            }else if(isJsonError){
                status = false;
                networkResponse.setSuccess(false);
                networkResponse.setResult(response);
                networkResponse.setError(NetworkError.INVALID_JSON_ERROR);
                mWebResponseHandler.handleResponse(networkResponse);
            }else {
                status = BaseWebResponseHandler.isValidResponse((BaseEntity) baseEntity);
            }

            if (mIWebResponseListener != null) {
                mIWebResponseListener.onResponse(status ? IWebResponseListener.ResponseStatus.SUCCESS
                        : IWebResponseListener.ResponseStatus.SERVER_ERROR , baseEntity);
            }

            if(isJsonError || (sync &&!status)){
                throw new WebRequestException(NetworkError.SERVER_ERROR);
            }
        }else{
            if(mWebRequestBuilder.isHandleError()){
                mWebResponseHandler.handleResponse(networkResponse);
            }

            if(mIWebResponseListener!=null){
                mIWebResponseListener.onResponse(IWebResponseListener.ResponseStatus.HTTP_ERROR, networkResponse);
            }
            if(sync){
                throw new WebRequestException(NetworkError.HTTP_ERROR);
            }
        }
        return baseEntity;
    }

}
