package com.devdigital.networklib;

import android.content.Context;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.gsonparserfactory.GsonParserFactory;
import com.devdigital.networklib.handler.BaseWebResponseHandler;
import com.devdigital.networklib.ssl.Tls12SocketFactory;
import com.facebook.stetho.Stetho;
import com.facebook.stetho.okhttp3.StethoInterceptor;

import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;

/**
 * Network Lib module dependency injector
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 22 November 2017
 */
public class NetworkController {

    private static Class<? extends BaseWebResponseHandler> mWebResponseHandlerClass;

    /**
     * @param cls BaseWebResponseHandler implementation
     */
    public static void setWebResponseHandler(Class<? extends BaseWebResponseHandler> cls) {
        mWebResponseHandlerClass = cls;
    }

    public static Class<? extends BaseWebResponseHandler> getWebResponseHandler() {
        return mWebResponseHandlerClass;
    }

    /**
     * Call this method once from main {@link android.app.Application} call once to
     * initialize Stetho and AndroidNetworking library
     * <p>
     * To Use Stetho open  chrome://inspect/#devices in Chrome Browser
     *
     * @param context {@link Context} object
     * @param isDebuggable boolean, if true then Stetho is enable else Stetho is disabled.
     */
    public static void initAndroidNetworking(Context context, boolean isDebuggable) {
        if(isDebuggable) {
            //Initialize Stetho
            Stetho.initializeWithDefaults(context);
        }

        OkHttpClient.Builder mClientBuilder = new OkHttpClient.Builder()
                .followRedirects(true)
                .followSslRedirects(true)
                .retryOnConnectionFailure(true)
                .cache(null)
                .connectTimeout(5, TimeUnit.SECONDS)
                .writeTimeout(5, TimeUnit.SECONDS)
                .readTimeout(5, TimeUnit.SECONDS);

        OkHttpClient.Builder builder = Tls12SocketFactory.enableTls12OnPreLollipop(mClientBuilder);
        if(isDebuggable) {
            //Intercept Network Request with Stetho
            builder.addNetworkInterceptor(new StethoInterceptor());
        }
        //Set Read Timeout as 1 Minute
        builder.readTimeout(1, TimeUnit.MINUTES);
        //Set Connection Timeout as 10 Second
        builder.connectTimeout(10, TimeUnit.SECONDS);
        OkHttpClient okHttpClient = builder.build();

        //Set OkHttpClient as HTTP stack
        AndroidNetworking.initialize(context, okHttpClient);

        //Add GSON as JSON Parser
        AndroidNetworking.setParserFactory(new GsonParserFactory());
    }

}
