package com.devdigital.networklib.stack;

import android.os.AsyncTask;
import android.util.Log;

import com.devdigital.networklib.exception.NetworkError;
import com.devdigital.networklib.handler.WebRequestBuilder;
import com.devdigital.networklib.listener.IWebResponseListener;

import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.SocketException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.util.List;
import java.util.Map;

/**
 * {@link INetworkStack} implementation on {@link java.net.HttpURLConnection}
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 22 November 2017
 */
public class HUrlConnectionStack extends AsyncTask<Void, Void, String> implements INetworkStack {

    private static final String TAG = HUrlConnectionStack.class.getSimpleName();
    private WebRequestBuilder mWebRequestBuilder;

    /**
     * HTTP Status code
     */
    private int mStatusCode;
    /**
     * HTTP Response
     */
    private String mResponse;
    /**
     * HTTP Response Header
     */
    private Map<String, List<String>> mHeaders;
    private Exception mError;

    public HUrlConnectionStack(WebRequestBuilder fields) {
        mWebRequestBuilder = fields;
    }

    @Override
    public void getAsync() {
        execute();
    }

    @Override
    public NetworkStackResponse getSync() {
        doInBackground();
        NetworkStackResponse response = new NetworkStackResponse();
        response.setHeaders(mHeaders);
        response.setResult(mResponse);
        response.setStatus(mStatusCode);

        if(mError!=null){
            if(mError instanceof ConnectException || mError instanceof SocketException){
                response.setError(NetworkError.CONNECTION_ERROR);
            }else {
                response.setError(NetworkError.UNKNOWN_ERROR);
            }
        }else{
            response.setSuccess(mStatusCode == 200);
        }

        return response;
    }

    @Override
    protected String doInBackground(Void... voids) {
        try {
            WebRequestBuilder.RequestMethod requestMethod = mWebRequestBuilder.getRequestMethod();

            String url = getURL();
            showLog("HTTP URL is " + url);
            URL obj = new URL(url);

            HttpURLConnection connection = (HttpURLConnection) obj.openConnection();

            //Set Headers
            setRequestHeaders(connection);

            switch (requestMethod) {
                case POST:
                case PUT:
                case DELETE:
                    connection.setRequestMethod(requestMethod.name());
                    connection.setDoOutput(true);
                    writePostBody(connection);
                    break;
                case GET:
                    connection.setRequestMethod(requestMethod.name());
                    break;
                case MULTI_PART:
                    connection.setRequestMethod("POST");
                    connection.setDoInput(true);
                    connection.setDoOutput(true);
                    connection.setRequestProperty("Connection", "Keep-Alive");
                    connection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + BOUNDARY);
                    writeMultipartBody(connection);
                    break;
                default:
                     throw new RuntimeException("Request Method not supported");
            }

            mStatusCode = connection.getResponseCode();

            if (mStatusCode == HttpURLConnection.HTTP_OK) {
                //success
                mResponse = getResponseBody(connection.getInputStream());
            } else {
                //error
                mResponse = getResponseBody(connection.getErrorStream());
            }

            mHeaders = connection.getHeaderFields();
        } catch (Exception e){
            e.printStackTrace();
            mError = e;
        }
        return mResponse;
    }

    @Override
    protected void onPostExecute(String s) {
        super.onPostExecute(s);
        if(mWebRequestBuilder.getResponseListener()!=null){
            NetworkStackResponse response = new NetworkStackResponse();
            response.setStatus(mStatusCode);
            response.setHeaders(mHeaders);
            response.setResult(mResponse);
            if(mError!=null){
                if(mError instanceof ConnectException
                        || mError instanceof SocketException){
                    response.setError(NetworkError.CONNECTION_ERROR);
                }else {
                    response.setError(NetworkError.UNKNOWN_ERROR);
                }
            }else{
                response.setSuccess(mStatusCode == 200);
                if(mStatusCode!=200){
                    response.setError(NetworkError.HTTP_ERROR);
                }
            }
            mWebRequestBuilder.getResponseListener().onResponse(mStatusCode==200? IWebResponseListener.ResponseStatus.SUCCESS:
                    IWebResponseListener.ResponseStatus.HTTP_ERROR, response);
        }
    }

    /**
     * Get Request URL based on GET or POST request
     *
     * @return request URL
     */
    private String getURL() {
        if (mWebRequestBuilder.getRequestMethod() == WebRequestBuilder.RequestMethod.GET) {
            if(mWebRequestBuilder.getRequestParams()==null){
                return mWebRequestBuilder.getUrl();
            }else{
                return mWebRequestBuilder.getUrl() +"?"+getBodyParams();
            }
        } else {
            return mWebRequestBuilder.getUrl();
        }
    }

    /**
     * Set Header for http request
     *
     * @param connection {@link HttpURLConnection} object
     */
    private void setRequestHeaders(HttpURLConnection connection) {
        if(mWebRequestBuilder.getHeader()!=null){
            for (Map.Entry<String, Object> entries : mWebRequestBuilder.getHeader().entrySet()) {
                if(entries.getKey()!=null && entries.getValue()!=null) {
                    connection.setRequestProperty(entries.getKey(), entries.getValue().toString());
                }
            }
        }
    }

    /**
     * Send data to post request
     *
     * @param connection {@link HttpURLConnection} object
     * @throws IOException
     */
    private void writePostBody(HttpURLConnection connection) throws IOException {
        OutputStreamWriter os = new OutputStreamWriter(connection.getOutputStream());
        if (mWebRequestBuilder.getRequestJSON() != null) {
            String json = mWebRequestBuilder.getRequestJSON().toString();
            showLog("HTTP Post Body is " + json);
            os.write(json);
        } else if (mWebRequestBuilder.getRequestParams() != null && !mWebRequestBuilder.getRequestParams().isEmpty()) {
            String body = getBodyParams();
            showLog("HTTP Post Body is " + body);
            os.write(body);
        } else if (mWebRequestBuilder.getRequestBody() != null) {
            showLog("HTTP Post Body is " + mWebRequestBuilder.getRequestBody());
            os.write(mWebRequestBuilder.getRequestBody());
        }
        os.flush();
        os.close();
    }

    /**
     * Send multipart data to  request
     *
     * @param connection {@link HttpURLConnection} object
     * @throws IOException
     */
    private void writeMultipartBody(HttpURLConnection connection) throws IOException {
        DataOutputStream outputStream = new DataOutputStream(connection.getOutputStream());
        if(mWebRequestBuilder.getRequestParams()!=null){
            addMultipartFormField(outputStream);
        }
        if(mWebRequestBuilder.getMultipartFile()!=null){
            addMultipartFormFile(outputStream);
        }

        outputStream.writeBytes(LINE_FEED);
        outputStream.flush();
        outputStream.writeBytes("--" + BOUNDARY + "--");
        outputStream.writeBytes(LINE_FEED);

        outputStream.flush();
        outputStream.close();
    }

    /**
     * Get response body
     *
     * @param inputStream {@link InputStream} object
     * @return get Response body
     * @throws IOException
     */
    private String getResponseBody(InputStream inputStream) throws IOException {
        BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));
        String inputLine;
        StringBuilder response = new StringBuilder();
        while ((inputLine = in.readLine()) != null) {
            response.append(inputLine);
        }
        in.close();
        return response.toString();
    }

    /**
     * encode simple string to UTF-8
     *
     * @param text to convert to UTF 8
     * @return utf-8 encoded string
     */
    public static String encodeToUTF8(String text) {
        try {
            return URLEncoder.encode(text, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return text;
        }
    }

    /**
     * Get Request Body as params
     *
     * @return Request Body
     */
    private String getBodyParams(){
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, Object> entry : mWebRequestBuilder.getRequestParams().entrySet()) {
            if (sb.length() > 0) {
                sb.append("&");
            }
            sb.append(String.format("%s=%s",
                    entry.getKey(), entry.getValue().toString()
                    //encodeToUTF8(entry.getKey()), encodeToUTF8(entry.getValue().toString())
            ));
            /*sb.append(String.format("%s=%s", entry.getKey(), entry.getValue().toString()));*/
        }
        return sb.toString();
    }


    private static final String LINE_FEED = "\r\n";
    private final String BOUNDARY =  "*****";

    /**
     * Adds a form field to the multipart request
     *
     * @param outputStream DataOutputStream, output stream object
     * @throws IOException
     */
    public void addMultipartFormField(DataOutputStream outputStream) throws IOException {
        for (Map.Entry<String, Object> entry : mWebRequestBuilder.getRequestParams().entrySet()) {
            outputStream.writeBytes("--" + BOUNDARY);
            outputStream.writeBytes(LINE_FEED);
            outputStream.writeBytes("Content-Disposition: form-data; name=\"" + entry.getKey() + "\"");
            outputStream.writeBytes(LINE_FEED);
            outputStream.writeBytes("Content-Type: text/plain; charset=UTF-8");
            outputStream.writeBytes(LINE_FEED);
            outputStream.writeBytes(LINE_FEED);
            outputStream.writeBytes(entry.getValue().toString());
            outputStream.writeBytes(LINE_FEED);
            outputStream.flush();
        }
    }

    /**
     * Adds a file field to the multipart request
     *
     * @param outputStream DataOutputStream, output stream object
     * @throws IOException
     */
    public void addMultipartFormFile(DataOutputStream outputStream) throws IOException {
        File file;
        for (Map.Entry<String, File> entry: mWebRequestBuilder.getMultipartFile().entrySet()) {
            file = entry.getValue();
            String fileName = file.getName();
            outputStream.writeBytes("--" + BOUNDARY);
            outputStream.writeBytes(LINE_FEED);
            outputStream.writeBytes(
                    "Content-Disposition: form-data; name=\"" + entry.getKey()
                            + "\"; filename=\"" + fileName + "\"");
            outputStream.writeBytes(LINE_FEED);
            outputStream.writeBytes(
                    "Content-Type: "
                            + URLConnection.guessContentTypeFromName(fileName));
            outputStream.writeBytes(LINE_FEED);
            outputStream.writeBytes("Content-Transfer-Encoding: binary");
            outputStream.writeBytes(LINE_FEED);
            outputStream.writeBytes(LINE_FEED);
            outputStream.flush();

            FileInputStream inputStream = new FileInputStream(file);
            byte[] buffer = new byte[8192];
            int bytesRead = -1;
            while ((bytesRead = inputStream.read(buffer)) != -1) {
                outputStream.write(buffer, 0, bytesRead);
            }
            outputStream.writeBytes(LINE_FEED);
            outputStream.flush();

            inputStream.close();
        }
    }

    /**
     * Show log if isDebug flag is true
     *
     * @param message String message
     */
    private void showLog(String message){
        if(mWebRequestBuilder.isDebug()) {
            Log.i(TAG, message);
        }
    }


}
