package com.devdigital.networklib.stack;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.common.ANRequest;
import com.androidnetworking.common.ANResponse;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.OkHttpResponseAndStringRequestListener;
import com.devdigital.networklib.exception.NetworkError;
import com.devdigital.networklib.handler.WebRequestBuilder;

import java.io.IOException;

import okhttp3.Response;

import static com.devdigital.networklib.listener.IWebResponseListener.ResponseStatus.HTTP_ERROR;
import static com.devdigital.networklib.listener.IWebResponseListener.ResponseStatus.SUCCESS;

/**
 * INetworkStack implementation on com.androidnetworking.AndroidNetworking
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 22 November 2017
 */
public class AndroidNetworkingStack implements INetworkStack, OkHttpResponseAndStringRequestListener {

    private WebRequestBuilder mWebRequestBuilder;

    public AndroidNetworkingStack(WebRequestBuilder fields) {
        mWebRequestBuilder = fields;
    }

    @Override
    public void getAsync() {
        ANRequest anRequest = getANRRequest();
        anRequest.getAsOkHttpResponseAndString(this);
    }

    @Override
    public NetworkStackResponse getSync() {
        ANRequest anRequest = getANRRequest();
        ANResponse anResponse = anRequest.executeForOkHttpResponse();

        NetworkStackResponse nResponse;
        if (anResponse.isSuccess()) {
            nResponse = getSuccessResponse(anResponse.getOkHttpResponse());
        } else {
            nResponse = getErrorResponse(anResponse.getError());
        }

        return nResponse;
    }

    /**
     * @return {@link ANRequest} based on GET or POST requests
     */
    private ANRequest getANRRequest() {
        ANRequest anRequest;
        if (mWebRequestBuilder.getRequestMethod() == WebRequestBuilder.RequestMethod.GET) {
            anRequest = getGetANRRequest();
        } else if (mWebRequestBuilder.getRequestMethod() == WebRequestBuilder.RequestMethod.MULTI_PART) {
            anRequest = getMultiPartANRRequest();
        } else {
            anRequest = getPostANRRequest();
        }
        return anRequest;
    }

    /**
     * @return {@link ANRequest} for POST/PUT method
     */
    private ANRequest getPostANRRequest() {
        ANRequest.PostRequestBuilder builder;
        if (mWebRequestBuilder.getRequestMethod() == WebRequestBuilder.RequestMethod.PUT) {
            builder = AndroidNetworking.put(mWebRequestBuilder.getUrl());
        } else if (mWebRequestBuilder.getRequestMethod() == WebRequestBuilder.RequestMethod.DELETE) {
            builder = AndroidNetworking.delete(mWebRequestBuilder.getUrl());
        } else {
            builder = AndroidNetworking.post(mWebRequestBuilder.getUrl());
        }

        builder.addHeaders(mWebRequestBuilder.getHeader());
        if (mWebRequestBuilder.getRequestJSON() != null) {
            builder.addJSONObjectBody(mWebRequestBuilder.getRequestJSON());
        } else if (mWebRequestBuilder.getRequestParams() != null && !mWebRequestBuilder.getRequestParams().isEmpty()) {
            builder.addBodyParameter(mWebRequestBuilder.getRequestParams());
        } else if (mWebRequestBuilder.getRequestBody() != null) {
            builder.addStringBody(mWebRequestBuilder.getRequestBody());
        }

        if(mWebRequestBuilder.getHeader()!=null && mWebRequestBuilder.getHeader().containsKey("Content-Type")){
            // custom ContentType
            builder.setContentType(mWebRequestBuilder.getHeader().get("Content-Type").toString());
        }
        return builder.setTag(this)
                .build();
    }

    public ANRequest getMultiPartANRRequest() {
        ANRequest.MultiPartBuilder builder = AndroidNetworking.upload(mWebRequestBuilder.getUrl())
                .addHeaders(mWebRequestBuilder.getHeader());

        if (mWebRequestBuilder.getRequestParams() != null) {
            builder.addMultipartParameter(mWebRequestBuilder.getRequestParams());
        }

        if (mWebRequestBuilder.getMultipartFile() != null) {
            builder.addMultipartFile(mWebRequestBuilder.getMultipartFile());
        }

        if(mWebRequestBuilder.getHeader()!=null && mWebRequestBuilder.getHeader().containsKey("Content-Type")){
            builder.setContentType(mWebRequestBuilder.getHeader().get("Content-Type").toString());
        }

        return builder.setTag(this)
                .build();
    }

    /**
     * @return {@link ANRequest} for GET method
     */
    private ANRequest getGetANRRequest() {
        ANRequest.GetRequestBuilder builder = AndroidNetworking.get(mWebRequestBuilder.getUrl())
                .addHeaders(mWebRequestBuilder.getHeader());

        if (mWebRequestBuilder.getRequestParams() != null) {
            builder.addQueryParameter(mWebRequestBuilder.getRequestParams());
        }

        return builder.setTag(this)
                .build();
    }

    @Override
    public void onResponse(Response okHttpResponse, String response) {
        if (mWebRequestBuilder.getResponseListener() != null) {
            NetworkStackResponse nResponse = getSuccessResponse(okHttpResponse, response);
            mWebRequestBuilder.getResponseListener().onResponse(SUCCESS, nResponse);
        }
    }

    /**
     * @param anError Handle error for getAsync method call
     */
    @Override
    public void onError(ANError anError) {
        NetworkStackResponse nResponse = getErrorResponse(anError);
        mWebRequestBuilder.getResponseListener().onResponse(HTTP_ERROR, nResponse);
    }

    /**
     * Convert Response object to NetworkStackResponse object
     *
     * @param response Response object
     * @return NetworkStackResponse object
     */
    private NetworkStackResponse getSuccessResponse(Response response) {
        NetworkStackResponse nResponse = new NetworkStackResponse();
        nResponse.setSuccess(true);
        nResponse.setStatus(response.code());
        try {
            nResponse.setResult(response.body().source().readUtf8());
        } catch (IOException e) {
            e.printStackTrace();
        }

        nResponse.setHeaders(response.headers().toMultimap());
        return nResponse;
    }

    private NetworkStackResponse getSuccessResponse(Response response, String parsed) {
        NetworkStackResponse nResponse = new NetworkStackResponse();
        nResponse.setSuccess(true);
        nResponse.setStatus(response.code());
        nResponse.setResult(parsed);
        nResponse.setHeaders(response.headers().toMultimap());
        return nResponse;
    }

    /**
     * Convert ANError object to NetworkStackResponse object
     *
     * @param anError ANError object
     * @return NetworkStackResponse object
     */
    private NetworkStackResponse getErrorResponse(ANError anError) {
        NetworkStackResponse nResponse = new NetworkStackResponse();
        nResponse.setSuccess(false);
        if (anError.getErrorCode() != 0) {
            // received error from server
            // error.getErrorCode() - the error code from server
            // error.getErrorBody() - the error body from server
            // error.getErrorDetail() - just an error detail
            nResponse.setStatus(anError.getErrorCode());
            nResponse.setResult(anError.getErrorBody());
            nResponse.setError(NetworkError.HTTP_ERROR);
        } else {
            // error.getErrorDetail() : connectionError, parseError, requestCancelledError
            nResponse.setResult(anError.getErrorDetail());
            switch (anError.getErrorDetail()){
                case "connectionError":
                    nResponse.setError(NetworkError.CONNECTION_ERROR);
                    break;
                case "parseError":
                    nResponse.setError(NetworkError.INVALID_JSON_ERROR);
                    break;
                default:
                    nResponse.setError(NetworkError.UNKNOWN_ERROR);
                    break;
            }
        }
        if(anError.getResponse()!=null) {
            nResponse.setHeaders(anError.getResponse().headers().toMultimap());
        }
        return nResponse;
    }
}
