package com.devdigital.networklib.handler;

import android.app.Activity;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.os.Build;
import android.support.v7.app.AlertDialog;
import android.util.Log;
import android.view.View;

import com.devdigital.networklib.R;
import com.devdigital.networklib.entity.BaseEntity;
import com.devdigital.networklib.stack.NetworkStackResponse;

/**
 * Abstract class to handle Web Service Response
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 21 November 2017
 */
public abstract class BaseWebResponseHandler {

    protected WebRequestBuilder mWebRequestBuilder;
    protected Object mProgressDialog;

    public void setWebRequestBuilder(WebRequestBuilder builder) {
        this.mWebRequestBuilder = builder;
    }

    /**
     * Handle Response in case server responded with HTTP 200
     *
     * @param baseEntity {@link BaseEntity} class
     * @return boolean true for success response.
     */
    public abstract boolean handleResponse(BaseEntity baseEntity);

    /**
     * Handle Response in case server responded with HTTP status code other than 200.
     *
     * @param response {@link NetworkStackResponse} response
     * @return boolean false always
     */
    public abstract boolean handleResponse(NetworkStackResponse response);

    /**
     * Handle Network Error
     */
    public void handleNetworkError(){
        Log.e("NetworkError", "handleNetworkError");
        showError(R.string.error_network_error);
    }

    /**
     * @param resId Show Error message using given string resource id
     */
    public void showError(int resId){
        Context context = mWebRequestBuilder.getContext();
        showError(context.getString(resId));
    }

    /**
     * @param message Show Error message using given string message
     */
    public void showError(String message){
        if(mWebRequestBuilder.isShowErrorDialog()){
            Context context = mWebRequestBuilder.getContext();
            if(!(context instanceof Activity)){
                throw new RuntimeException("Require Activity Instance");
            }

            AlertDialog.Builder builder;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                builder = new AlertDialog.Builder(context, android.R.style.Theme_Material_Dialog_Alert);
            } else {
                builder = new AlertDialog.Builder(context);
            }
            builder.setTitle(R.string.app_name)
                    .setMessage(message)
                    .setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int which) {
                            dialog.cancel();
                            if(mWebRequestBuilder.getErrorDialogClickListener()!=null){
                                mWebRequestBuilder.getErrorDialogClickListener().onDialogClick();
                            }
                        }
                    })
                    .setCancelable(false)
                    .setIcon(android.R.drawable.ic_dialog_alert)
                    .show();
        }
    }

    /**
     * Check if server response is success or failure
     * @param entity BaseEntity object
     * @return boolean true or false
     */
    public static boolean isValidResponse(BaseEntity entity){
        return entity!=null && StatusConstant.SUCCESS.name().equalsIgnoreCase(entity.getStatus());
    }


    /**
     * Show progress dialog on async request. Override this method to show custom progress dialog.
     */
    public void showProgressDialog() {
        if(!mWebRequestBuilder.isShowProgressDialog()) return;

        if(mWebRequestBuilder.getProgressView()!=null) {
            mWebRequestBuilder.getProgressView().setVisibility(View.VISIBLE);
        }else{
            String message = mWebRequestBuilder.getProgressMessage();
            if (message == null) {
                message = mWebRequestBuilder.getContext().getString(R.string.progress_dialog_message);
            }

            ProgressDialog progressDialog  = new ProgressDialog(mWebRequestBuilder.getContext());
            progressDialog .setMessage(message);
            progressDialog .show();
            this.mProgressDialog = progressDialog ;
        }
    }

    /**
     * Hide progress dialog.
     */
    public void hideProgressDialog() {
        if(!mWebRequestBuilder.isShowProgressDialog()) return;

        if(mWebRequestBuilder.getProgressView()!=null) {
            mWebRequestBuilder.getProgressView().setVisibility(View.GONE);
        } else {
            ProgressDialog progressDialog  = (ProgressDialog) this.mProgressDialog;
            if(progressDialog !=null) progressDialog .hide();
        }
    }

}
