package com.devdigital.networklib.stack;

import android.os.AsyncTask;

import com.devdigital.networklib.exception.NetworkError;
import com.devdigital.networklib.handler.WebRequestBuilder;
import com.devdigital.networklib.listener.IWebResponseListener;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.SocketException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.List;
import java.util.Map;

/**
 * {@link INetworkStack} implementation on {@link java.net.HttpURLConnection}
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 22 November 2017
 */
public class HUrlConnectionStack extends AsyncTask<Void, Void, String> implements INetworkStack {

    private WebRequestBuilder mWebRequestBuilder;

    /**
     * HTTP Status code
     */
    private int mStatusCode;
    /**
     * HTTP Response
     */
    private String mResponse;
    /**
     * HTTP Response Header
     */
    private Map<String, List<String>> mHeaders;
    private Exception mError;

    public HUrlConnectionStack(WebRequestBuilder fields) {
        mWebRequestBuilder = fields;
    }

    @Override
    public void getAsync() {
        execute();
    }

    @Override
    public NetworkStackResponse getSync() {
        doInBackground();
        NetworkStackResponse response = new NetworkStackResponse();
        response.setHeaders(mHeaders);
        response.setResult(mResponse);
        response.setStatus(mStatusCode);

        if(mError!=null){
            if(mError instanceof ConnectException || mError instanceof SocketException){
                response.setError(NetworkError.CONNECTION_ERROR);
            }else {
                response.setError(NetworkError.UNKNOWN_ERROR);
            }
        }else{
            response.setSuccess(mStatusCode == 200);
        }

        return response;
    }

    @Override
    protected String doInBackground(Void... voids) {
        try {
            WebRequestBuilder.RequestMethod requestMethod = mWebRequestBuilder.getRequestMethod();

            URL obj = new URL(getURL());
            HttpURLConnection connection = (HttpURLConnection) obj.openConnection();

            //Set Headers
            setRequestHeaders(connection);

            connection.setRequestMethod(requestMethod.name());
            switch (requestMethod) {
                case POST:
                case PUT:
                case DELETE:
                    connection.setDoOutput(true);
                    writePostBody(connection);
                    break;
            }

            mStatusCode = connection.getResponseCode();

            if (mStatusCode == HttpURLConnection.HTTP_OK) {
                //success
                mResponse = getResponseBody(connection.getInputStream());
            } else {
                //error
                mResponse = getResponseBody(connection.getErrorStream());
            }

            mHeaders = connection.getHeaderFields();
        } catch (Exception e){
            mError = e;
        }
        return mResponse;
    }

    @Override
    protected void onPostExecute(String s) {
        super.onPostExecute(s);
        if(mWebRequestBuilder.getResponseListener()!=null){
            NetworkStackResponse response = new NetworkStackResponse();
            response.setStatus(mStatusCode);
            response.setHeaders(mHeaders);
            response.setResult(mResponse);

            if(mError!=null){
                if(mError instanceof ConnectException
                        || mError instanceof SocketException){
                    response.setError(NetworkError.CONNECTION_ERROR);
                }else {
                    response.setError(NetworkError.UNKNOWN_ERROR);
                }
            }else{
                response.setSuccess(mStatusCode == 200);
            }

            mWebRequestBuilder.getResponseListener().onResponse(mStatusCode==200? IWebResponseListener.ResponseStatus.SUCCESS:
                    IWebResponseListener.ResponseStatus.HTTP_ERROR, response);
        }
    }

    /**
     * Get Request URL based on GET or POST request
     *
     * @return request URL
     */
    private String getURL() {
        if (mWebRequestBuilder.getRequestMethod() == WebRequestBuilder.RequestMethod.GET) {
            if(mWebRequestBuilder.getRequestParams()==null){
                return mWebRequestBuilder.getUrl();
            }else{
                return mWebRequestBuilder.getUrl() +"?"+getBodyParams();
            }
        } else {
            return mWebRequestBuilder.getUrl();
        }
    }

    /**
     * Set Header for http request
     *
     * @param connection {@link HttpURLConnection} object
     */
    private void setRequestHeaders(HttpURLConnection connection) {
        if(mWebRequestBuilder.getHeader()!=null){
            for (Map.Entry<String, Object> entries : mWebRequestBuilder.getHeader().entrySet()) {
                if(entries.getKey()!=null && entries.getValue()!=null) {
                    connection.setRequestProperty(entries.getKey(), entries.getValue().toString());
                }
            }
        }
    }

    /**
     * Send data to post request
     *
     * @param connection {@link HttpURLConnection} object
     * @throws IOException
     */
    private void writePostBody(HttpURLConnection connection) throws IOException {
        OutputStreamWriter os = new OutputStreamWriter(connection.getOutputStream());
        if (mWebRequestBuilder.getRequestJSON() != null) {
            String json = mWebRequestBuilder.getRequestJSON().toString();
            os.write(json);
        } else if (mWebRequestBuilder.getRequestParams() != null) {
            os.write(getBodyParams());
        } else if (mWebRequestBuilder.getRequestBody() != null) {
            os.write(mWebRequestBuilder.getRequestBody());
        }
        os.flush();
        os.close();
    }

    /**
     * Get response body
     *
     * @param inputStream {@link InputStream} object
     * @return get Response body
     * @throws IOException
     */
    private String getResponseBody(InputStream inputStream) throws IOException {
        BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));
        String inputLine;
        StringBuilder response = new StringBuilder();
        while ((inputLine = in.readLine()) != null) {
            response.append(inputLine);
        }
        in.close();
        return response.toString();
    }

    /**
     * encode simple string to UTF-8
     *
     * @param text to convert to UTF 8
     * @return utf-8 encoded string
     */
    public static String encodeToUTF8(String text) {
        try {
            return URLEncoder.encode(text, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return text;
        }
    }

    /**
     * Get Request Body as params
     *
     * @return Request Body
     */
    private String getBodyParams(){
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, Object> entry : mWebRequestBuilder.getRequestParams().entrySet()) {
            if (sb.length() > 0) {
                sb.append("&");
            }
            sb.append(String.format("%s=%s",
                    encodeToUTF8(entry.getKey()),
                    encodeToUTF8(entry.getValue().toString())
            ));
            /*sb.append(String.format("%s=%s", entry.getKey(), entry.getValue().toString()));*/
        }
        return sb.toString();
    }


}
