package com.devdigital.networklib.handler;

import android.text.TextUtils;
import android.util.Log;

import com.awesomedialog.blennersilva.awesomedialoglibrary.AwesomeProgressDialog;
import com.devdigital.networklib.NetworkController;
import com.devdigital.networklib.R;
import com.devdigital.networklib.entity.BaseEntity;
import com.devdigital.networklib.exception.NetworkError;
import com.devdigital.networklib.exception.WebRequestException;
import com.devdigital.networklib.json.JSONFactory;
import com.devdigital.networklib.listener.IWebResponseListener;
import com.devdigital.networklib.stack.HUrlConnectionStack;
import com.devdigital.networklib.stack.INetworkStack;
import com.devdigital.networklib.stack.NetworkStackResponse;
import com.devdigital.networklib.utils.NetworkUtils;

import java.lang.reflect.Constructor;

/**
 * Implementation for {@link IWebRequestExecutor}
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 21 November 2017
 */
public class WebRequestExecutor implements IWebRequestExecutor, IWebResponseListener {

    /**
     * Default Network Stack
     */
    private static final Class<? extends INetworkStack> NETWORK_STACK_CLASS = HUrlConnectionStack.class;

    private WebRequestBuilder mWebRequestBuilder;
    private IWebResponseListener mIWebResponseListener;
    private INetworkStack mNetworkStack;
    private BaseWebResponseHandler mWebResponseHandler;

    public WebRequestExecutor(WebRequestBuilder builder) {
        this.mWebRequestBuilder = builder;

        Class<? extends INetworkStack> networkStackCls = NETWORK_STACK_CLASS;
        if(mWebRequestBuilder.getNetworkStack()!=null){
            networkStackCls = mWebRequestBuilder.getNetworkStack();
        }

        //Swap response listener. inorder to listen to response in this class.
        mIWebResponseListener = mWebRequestBuilder.getResponseListener();
        mWebRequestBuilder.setResponseListener(this);

        try {
            //Using reflection get Network Instance class.
            Constructor<?> ctor = networkStackCls.getConstructor(WebRequestBuilder.class);
            mNetworkStack = (INetworkStack) ctor.newInstance(mWebRequestBuilder);
        } catch (Exception e) {
            e.printStackTrace();
        }

        mWebResponseHandler = NetworkController.getWebResponseHandler();
        mWebResponseHandler.setWebRequestBuilder(mWebRequestBuilder);
    }

    @Override
    public void getAsync() {
        if(!NetworkUtils.isNetworkAvailable(mWebRequestBuilder.getContext())){
            //Show error message if network not available
            mWebResponseHandler.handleNetworkError();
            return;
        }

        Log.w("Web_Service", mWebRequestBuilder.toString());

        showProgressDialog();

        mNetworkStack.getAsync();
    }

    @Override
    public BaseEntity get() throws WebRequestException {
        if(!NetworkUtils.isNetworkAvailable(mWebRequestBuilder.getContext())){
            //Show error message if network not available
            mWebResponseHandler.handleNetworkError();
            throw new WebRequestException(NetworkError.NO_NETWORK_ERROR);
        }

        NetworkStackResponse networkResponse =  mNetworkStack.getSync();

        Object entity = handleResponse(networkResponse, true);
        return (BaseEntity) entity;
    }

    /**
     *  Handle response of getAsync Method
     *
     * @param responseStatus ResponseStatus object
     * @param result NetworkStackResponse object
     */
    @Override
    public void onResponse(ResponseStatus responseStatus, Object result) {
        NetworkStackResponse networkResponse = (NetworkStackResponse) result;
        try {
            handleResponse(networkResponse, false);
        } catch (WebRequestException e) {
            e.printStackTrace();
        }

        hideProgressDialog();
    }

    private Object handleResponse(NetworkStackResponse networkResponse, boolean sync) throws WebRequestException {
        Log.i("Response", "HTTP Response is "+networkResponse.toString());
        Object baseEntity = null;
        if(networkResponse.isSuccess()) {
            String response = networkResponse.getResult();
            if (!TextUtils.isEmpty(response)) {
                try {
                    //Get Entity from json
                    baseEntity = JSONFactory.getInstance().fromJson(response, mWebRequestBuilder.getEntityClass());
                }catch (Exception e){
                    //If Exception occure in above call than json might be of error. Parse it again in BaseEnity
                    //Because above error can be occured as error json doesn't have "values" field.
                    baseEntity = JSONFactory.getInstance().fromJson(response, BaseEntity.class);
                }
            }

            boolean status;
            if (mWebRequestBuilder.isHandleError()) {
                status = mWebResponseHandler.handleResponse((BaseEntity) baseEntity);
            } else {
                status = BaseWebResponseHandler.isValidResponse((BaseEntity) baseEntity);
            }

            if (mIWebResponseListener != null) {
                mIWebResponseListener.onResponse(status ? IWebResponseListener.ResponseStatus.SUCCESS
                        : IWebResponseListener.ResponseStatus.SERVER_ERROR , baseEntity);
            }

            if(sync &&!status){
                throw new WebRequestException(NetworkError.SERVER_ERROR);
            }
        }else{
            if(mWebRequestBuilder.isHandleError()){
                mWebResponseHandler.handleResponse(networkResponse);
            }

            if(mIWebResponseListener!=null){
                mIWebResponseListener.onResponse(IWebResponseListener.ResponseStatus.HTTP_ERROR, networkResponse);
            }
            if(sync){
                throw new WebRequestException(NetworkError.HTTP_ERROR);
            }
        }
        return baseEntity;
    }

    /**
     * Show progress dialog on async request
     */
    private AwesomeProgressDialog mProgressDialog;
    private void showProgressDialog() {
        if(!mWebRequestBuilder.isShowProgressDialog()) return;

        mProgressDialog = new AwesomeProgressDialog(mWebRequestBuilder.getContext())
                .setTitle(R.string.app_name)
                .setMessage(R.string.progress_dialog_message)
                .setColoredCircle(R.color.colorPrimary)
                .setProgressBarColor(R.color.white)
                .setCancelable(false);
        mProgressDialog.show();
    }

    /**
     * Hide progress dialog
     */
    private void hideProgressDialog() {
        if(!mWebRequestBuilder.isShowProgressDialog()) return;
        if(mProgressDialog!=null) mProgressDialog.hide();
    }

}
