package com.devdigital.networklib.handler;

import android.app.Activity;
import android.content.Context;
import android.util.Log;

import com.awesomedialog.blennersilva.awesomedialoglibrary.AwesomeErrorDialog;
import com.awesomedialog.blennersilva.awesomedialoglibrary.interfaces.Closure;
import com.devdigital.networklib.R;
import com.devdigital.networklib.entity.BaseEntity;
import com.devdigital.networklib.stack.NetworkStackResponse;

/**
 * Abstract class to handle Web Service Response
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 21 November 2017
 */
public abstract class BaseWebResponseHandler {

    protected WebRequestBuilder mWebRequestBuilder;

    public void setWebRequestBuilder(WebRequestBuilder builder) {
        this.mWebRequestBuilder = builder;
    }

    /**
     * Handle Response in case server responded with HTTP 200
     *
     * @param baseEntity {@link BaseEntity} class
     * @return boolean true for success response.
     */
    public abstract boolean handleResponse(BaseEntity baseEntity);

    /**
     * Handle Response in case server responded with HTTP status code other than 200.
     *
     * @param response {@link NetworkStackResponse} response
     * @return boolean false always
     */
    public abstract boolean handleResponse(NetworkStackResponse response);

    /**
     * Handle Network Error
     */
    public void handleNetworkError(){
        Log.e("NetworkError", "handleNetworkError");
        showError(R.string.error_network_error);
    }

    /**
     * @param resId Show Error message using given string resource id
     */
    public void showError(int resId){
        Context context = mWebRequestBuilder.getContext();
        showError(context.getString(resId));
    }

    /**
     * @param message Show Error message using given string message
     */
    public void showError(String message){
        if(mWebRequestBuilder.isShowErrorDialog()){
            Context context = mWebRequestBuilder.getContext();
            if(!(context instanceof Activity)){
                throw new RuntimeException("Require Activity Instance");
            }

            final AwesomeErrorDialog dialog = new AwesomeErrorDialog(context)
                    .setTitle(R.string.app_name)
                    .setMessage(message)
                    .setColoredCircle(R.color.dialogErrorBackgroundColor)
                    .setDialogIconAndColor(R.drawable.ic_dialog_error, R.color.white)
                    .setCancelable(true)
                    .setButtonText(context.getString(R.string.dialog_ok_button))
                    .setButtonBackgroundColor(R.color.dialogErrorBackgroundColor);

            dialog.setErrorButtonClick(new Closure() {
                        @Override
                        public void exec() {
                            if(mWebRequestBuilder.getErrorDialogClickListener()!=null){
                                mWebRequestBuilder.getErrorDialogClickListener().onDialogClick();
                            }
                            dialog.hide();
                        }
                    });

            dialog.show();
        }
    }

    /**
     * Check if server response is success or failure
     * @param entity BaseEntity object
     * @return boolean true or false
     */
    public static boolean isValidResponse(BaseEntity entity){
        return entity!=null && StatusConstant.SUCCESS.name().equalsIgnoreCase(entity.getStatus());
    }

}
