package com.devdigital.gallerylib.utils;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;

/**
 * Resize Image
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 07 June 2018
 */
public class ImageResizer {

    private final float mFileSizeLimit;
    private final Context context;

    /**
     * @param context Application context class to get Real path from Uri
     * @param fileSizeLimit Resize image file until file size < fileSizeLimit.
     */
    public ImageResizer(Context context, float fileSizeLimit){
        this.mFileSizeLimit = fileSizeLimit;
        this.context = context;
    }

    /**
     *
     * @param imageUri Image URI
     * @return Uri file
     * @throws FileNotFoundException if file not exist
     */
    public Uri getResizedImage(Uri imageUri) throws FileNotFoundException {
        String path = UriFileUtils.getPathFromUri(context, imageUri);
        if(path==null) return imageUri;

        File file = new File(path);
        long fileLength = file.length();
        float fileSize = fileLength/1024.0f;

        if (fileSize <= mFileSizeLimit) {
            return imageUri;
        }
        float attempt = 1;
        while ((file.length()/1024.0f) > (mFileSizeLimit)) {
            file = resizeImageSize(file, attempt);
            attempt += 0.25f;
        }
        return Uri.fromFile(file);
    }

    private static File resizeImageSize(File file, float attempt) throws FileNotFoundException {
        BitmapFactory.Options bmOptions = new BitmapFactory.Options();
        if (file.length() > 10 * 1024 * 1024) {
            bmOptions.inSampleSize = 8;
        } else if (file.length() > 6 * 1024 * 1024) {
            bmOptions.inSampleSize = 4;
        } else if (file.length() > 2 * 1024 * 1024) {
            bmOptions.inSampleSize = 2;
        }

        Bitmap bitmap = BitmapFactory.decodeFile(file.getAbsolutePath(), bmOptions);

        int newWidth = bitmap.getWidth();
        int newHeight = bitmap.getHeight();
        int maxSize = (int) (2400/attempt);
        if (bitmap.getWidth() > maxSize || bitmap.getHeight() > maxSize) {
            if (bitmap.getWidth() > bitmap.getHeight()) {
                newWidth = maxSize;
                newHeight = (int) (bitmap.getHeight() * (((double) newWidth) / bitmap.getWidth()));
            } else {
                newHeight = maxSize;
                newWidth = (int) (bitmap.getWidth() * (((double) newHeight) / bitmap.getHeight()));
            }
        }
        Bitmap scaledBitmap = Bitmap.createScaledBitmap(bitmap, newWidth, newHeight, true);
        //Release resource
        bitmap.recycle();

        FileOutputStream out = new FileOutputStream(file);
        // compress bitmap
        scaledBitmap.compress(Bitmap.CompressFormat.JPEG, 90, out);

        //Release resource
        scaledBitmap.recycle();

        return file;
    }

}
