package com.devdigital.gallerylib.activities;

import android.Manifest;
import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.provider.MediaStore;
import android.support.annotation.Nullable;
import android.support.v4.content.FileProvider;
import android.support.v7.app.AppCompatActivity;
import android.util.Log;

import com.devdigital.gallerylib.R;
import com.devdigital.gallerylib.utils.FileUtil;
import com.devdigital.gallerylib.utils.IntentUtils;
import com.devdigital.gallerylib.utils.MediaUtils;
import com.devdigital.gallerylib.utils.PermissionManager;
import com.gun0912.tedpermission.PermissionListener;
import com.gun0912.tedpermission.TedPermission;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import life.knowledge4.videotrimmer.utils.PathToContentURI;

/**
 * Pick Video Using Camera or Gallery. This class will also Trim video if EXTRAS_TRIM_VIDEO intent param is set to true.
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 07 June 2018
 */
public class VideoPickerActivity extends AppCompatActivity {

    private static final String TAG = VideoPickerActivity.class.getSimpleName();

    private static final int INTENT_CHOOSER_REQ_CODE = 1011;
    private static final int INTENT_TRIM_REQ_CODE = 1012;

    /**
     * Pick Video Using Camera Only
     */
    public static final String EXTRAS_CAMERA_ONLY = "CAMERA_ONLY";
    /**
     * Pick Video Using Gallery Only
     */
    public static final String EXTRAS_GALLERY_ONLY = "GALLERY_ONLY";
    /**
     * Trim Picked Video
     */
    public static final String EXTRAS_TRIM_VIDEO = "TRIM_VIDEO";

    /**
     * max trim video duration
     */
    public static final String EXTRA_DURATION_LIMIT = "DURATION_LIMIT";
    /**
     * For Camera provide file path to write image
     */
    public static final String EXTRAS_CAMERA_VIDEO_FILE_PATH = "CAMERA_VIDEO_FILE_PATH";

    private boolean mTrimVideo;
    private int mMaxTrimLength;
    private boolean mPickGalleryVideoOnly;
    private boolean mPickCameraVideoOnly;
    private File mCameraFile;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        Intent intent = getIntent();
        mTrimVideo = intent.getBooleanExtra(EXTRAS_TRIM_VIDEO, false);
        mMaxTrimLength = intent.getIntExtra(EXTRA_DURATION_LIMIT, 0);
        mPickGalleryVideoOnly = intent.getBooleanExtra(EXTRAS_GALLERY_ONLY, false);
        mPickCameraVideoOnly = intent.getBooleanExtra(EXTRAS_CAMERA_ONLY, false);

        String cameraPath = intent.getStringExtra(EXTRAS_CAMERA_VIDEO_FILE_PATH);
        if(cameraPath!=null)
            mCameraFile = new File(cameraPath);

        checkPermission();
    }


    /**
     * Check Camera and Storage Permission.
     */
    private void checkPermission() {
        TedPermission.Builder builder = TedPermission.with(this)
                .setPermissionListener(new PermissionListener() {
                    @Override
                    public void onPermissionGranted() {
                        dispatchIntent();
                    }

                    @Override
                    public void onPermissionDenied(List<String> deniedPermissions) {
                        finish();
                    }
                })
                .setDeniedMessage(R.string.message_permission_denied);

        if (mPickGalleryVideoOnly) {
            builder.setPermissions(Manifest.permission.READ_EXTERNAL_STORAGE);
        } else {
            if(!PermissionManager.hasCameraPermission(this)){
                Log.e(TAG, "Check if camera permission is set in manifest.!");
            }
            builder.setPermissions(Manifest.permission.CAMERA,
                    Manifest.permission.WRITE_EXTERNAL_STORAGE);
        }

        builder.check();
    }

    public void dispatchIntent() {
        Intent galleryIntent = IntentUtils.getGalleryIntent(false);
        Intent cameraIntent = getCameraIntent();
        if (mPickGalleryVideoOnly) {
            startActivityForResult(galleryIntent, INTENT_CHOOSER_REQ_CODE);
        } else if (mPickCameraVideoOnly) {
            startActivityForResult(cameraIntent, INTENT_CHOOSER_REQ_CODE);
        } else {
            Intent chooserIntent = Intent.createChooser(cameraIntent, getString(R.string.txt_select));
            chooserIntent.putExtra(Intent.EXTRA_INITIAL_INTENTS, new Intent[]{galleryIntent});
            startActivityForResult(chooserIntent, INTENT_CHOOSER_REQ_CODE);
        }
    }

    /**
     * @return Camera Intent
     */
    private Intent getCameraIntent() {
        Intent intent = new Intent(MediaStore.ACTION_VIDEO_CAPTURE);
        if(mMaxTrimLength>0) {
            intent.putExtra(MediaStore.EXTRA_DURATION_LIMIT, mMaxTrimLength);
        }
        if (intent.resolveActivity(getPackageManager()) != null) {
            // Create the File where the photo should go
            if(mCameraFile==null) {
                mCameraFile = FileUtil.getCameraFile(false);
            }

            // Continue only if the File was successfully created
            if (mCameraFile != null) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    Uri photoURI = FileProvider.getUriForFile(this, getPackageName()+getString(R.string.file_provider_authority_suffix), mCameraFile);
                    intent.putExtra(MediaStore.EXTRA_OUTPUT, photoURI);
                } else {
                    intent.putExtra(MediaStore.EXTRA_OUTPUT, Uri.fromFile(mCameraFile));
                }
                return intent;
            }else{
                onError();
            }
        }
        return null;
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode == INTENT_CHOOSER_REQ_CODE) {
            if (resultCode == RESULT_CANCELED) {
                onError();
            } else {
                if (data != null && data.getData() != null) {
                    //Delete Camera Image File, if image is picked using gallery
                    Uri uri = data.getData();
                    String pathFromUri = PathToContentURI.getPathFromUri(this, uri);
                    if(!mCameraFile.getAbsolutePath().equals(pathFromUri) || mCameraFile.length()==0){
                        mCameraFile.delete();
                    }
                    long duration = MediaUtils.getVideoDuration(this, uri);
                    long durationInSec = duration/1000;
                    if((mMaxTrimLength>0 && durationInSec>mMaxTrimLength)  || mTrimVideo) {
                        Intent intent = new Intent(this, TrimmerActivity.class);
                        intent.putExtra(VideoPickerActivity.EXTRA_DURATION_LIMIT, mMaxTrimLength);
                        intent.setData(uri);
                        startActivityForResult(intent, INTENT_TRIM_REQ_CODE);
                    }else{
                        onSuccess(uri);
                    }
                } else {
                    if(mCameraFile.length()!=0) {
                        Uri uri = Uri.fromFile(mCameraFile);
                        if (mTrimVideo) {
                            Intent intent = new Intent(this, TrimmerActivity.class);
                            intent.setData(uri);
                            intent.putExtra(VideoPickerActivity.EXTRA_DURATION_LIMIT, mMaxTrimLength);
                            startActivityForResult(intent, INTENT_TRIM_REQ_CODE);
                        } else {
                            onSuccess(uri);
                        }
                    }else{
                        mCameraFile.delete();
                        onError();
                    }
                }
            }
        } else if (requestCode == INTENT_TRIM_REQ_CODE) {
            if (resultCode == RESULT_OK) {
                mCameraFile.delete();
                Intent intent = new Intent();
                intent.setData(data.getData());
                setResult(Activity.RESULT_OK, intent);
                finish();
            } else {
                onError();
            }
        } else {
            onError();
        }
    }
    /**
     * Handle Success
     */
    private void onSuccess(Uri data) {
        Intent intent = new Intent();
        intent.setData(data);
        setResult(Activity.RESULT_OK, intent);
        finish();
    }

    /**
     * Handle Error
     */
    private void onError() {
        if (mCameraFile != null)
            mCameraFile.delete();

        Intent intent = new Intent();
        setResult(Activity.RESULT_CANCELED, intent);
        finish();
    }

}
