package com.devdigital.gallerylib.activities;

import android.Manifest;
import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.provider.MediaStore;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v4.content.FileProvider;
import android.support.v7.app.AppCompatActivity;
import android.util.Log;

import com.devdigital.gallerylib.R;
import com.devdigital.gallerylib.utils.FileUtil;
import com.devdigital.gallerylib.utils.ImageResizer;
import com.devdigital.gallerylib.utils.IntentUtils;
import com.devdigital.gallerylib.utils.PermissionManager;
import com.gun0912.tedpermission.PermissionListener;
import com.gun0912.tedpermission.TedPermission;
import com.yalantis.ucrop.UCrop;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;

import life.knowledge4.videotrimmer.utils.PathToContentURI;

/**
 * Pick Image Using Camera or Gallery. This class will also crop image if EXTRAS_CROP_IMAGE intent param is set to true.
 *
 * @author Dhaval Patel
 * @author Rahul
 * @version 1.1
 * @since 08 January 2018
 */
public class ImagePickerActivity extends AppCompatActivity {

    private static final String TAG = ImagePickerActivity.class.getSimpleName();

    private static final int INTENT_CHOOSER_REQ_CODE = 1001;

    /**
     * Crop Picked Image
     */
    public static final String EXTRAS_CROP_IMAGE = "CROP_IMAGE";
    /**
     * Pick Image Using Camera Only
     */
    public static final String EXTRAS_CAMERA_ONLY = "CAMERA_ONLY";
    /**
     * Pick Image Using Gallery Only
     */
    public static final String EXTRAS_GALLERY_ONLY = "GALLERY_ONLY";

    /**
     * Pick Image Using Gallery Only
     * Crop Image as per the aspect ratio
     */
    public static final String EXTRAS_CROP_ASPECT_X = "CROP_ASPECT_X";
    public static final String EXTRAS_CROP_ASPECT_Y = "CROP_ASPECT_Y";

    /**
     * For Camera provide file path to write image
     */
    public static final String EXTRAS_CAMERA_IMAGE_FILE_PATH = "CAMERA_IMAGE_FILE_PATH";

    /**
     * Resize file if file size is greater than limit.
     * File Size in KB
     */
    public static final String EXTRA_MAX_FILE_SIZE = "FILE_SIZE_LIMIT";

    private File mCameraFile;
    private boolean mCropImage;
    private boolean mPickGalleryImageOnly;
    private boolean mPickCameraImageOnly;
    private ImageResizer mImageResizer;
    private float mCropAspectX, mCropAspectY;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        Intent intent = getIntent();
        mCropImage = intent.getBooleanExtra(EXTRAS_CROP_IMAGE, false);
        mPickGalleryImageOnly = intent.getBooleanExtra(EXTRAS_GALLERY_ONLY, false);
        mPickCameraImageOnly = intent.getBooleanExtra(EXTRAS_CAMERA_ONLY, false);
        mCropAspectX = intent.getFloatExtra(EXTRAS_CROP_ASPECT_X, 1);
        mCropAspectY = intent.getFloatExtra(EXTRAS_CROP_ASPECT_Y, 1);
        float mFileSizeLimit = intent.getFloatExtra(EXTRA_MAX_FILE_SIZE, Float.MAX_VALUE);

        String cameraPath = intent.getStringExtra(EXTRAS_CAMERA_IMAGE_FILE_PATH);
        if(cameraPath!=null)
            mCameraFile = new File(cameraPath);

        mImageResizer = new ImageResizer(this, mFileSizeLimit);

        checkPermission();
    }

    /**
     * Check Camera and Storage Permission.
     */
    private void checkPermission() {
        TedPermission.Builder builder = TedPermission.with(this)
                .setPermissionListener(new PermissionListener() {
                    @Override
                    public void onPermissionGranted() {
                        dispatchIntent();
                    }

                    @Override
                    public void onPermissionDenied(ArrayList<String> deniedPermissions) {
                        finish();
                    }
                })
                .setDeniedMessage(R.string.message_permission_denied);

        if (mPickGalleryImageOnly) {
            builder.setPermissions(Manifest.permission.READ_EXTERNAL_STORAGE);
        } else {
            if(!PermissionManager.hasCameraPermission(this)){
                Log.e(TAG, "Check if camera permission is set in manifest.!");
            }
            builder.setPermissions(Manifest.permission.CAMERA,
                    Manifest.permission.WRITE_EXTERNAL_STORAGE);
        }

        builder.check();
    }

    public void dispatchIntent() {
        Intent galleryIntent = IntentUtils.getGalleryIntent(true);
        Intent cameraIntent = getCameraIntent();
        if (mPickGalleryImageOnly) {
            startActivityForResult(galleryIntent, INTENT_CHOOSER_REQ_CODE);
        } else if (mPickCameraImageOnly) {
            startActivityForResult(cameraIntent, INTENT_CHOOSER_REQ_CODE);
        } else {
            Intent chooserIntent = Intent.createChooser(cameraIntent, getString(R.string.txt_select));
            chooserIntent.putExtra(Intent.EXTRA_INITIAL_INTENTS, new Intent[]{galleryIntent});
            startActivityForResult(chooserIntent, INTENT_CHOOSER_REQ_CODE);
        }
    }

    /**
     * @return Camera Intent
     */
    private Intent getCameraIntent() {
        Intent intent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);
        if (intent.resolveActivity(getPackageManager()) != null) {
            // Create the File where the photo should go
            if(mCameraFile==null) {
                mCameraFile = FileUtil.getCameraFile(true);
            }

            // Continue only if the File was successfully created
            if (mCameraFile != null) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    Uri photoURI = FileProvider.getUriForFile(this, getPackageName()+getString(R.string.file_provider_authority_suffix), mCameraFile);
                    intent.putExtra(MediaStore.EXTRA_OUTPUT, photoURI);
                } else {
                    intent.putExtra(MediaStore.EXTRA_OUTPUT, Uri.fromFile(mCameraFile));
                }
                return intent;
            }else{
                onError();
            }
        }
        return null;
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode == INTENT_CHOOSER_REQ_CODE) {
            if (resultCode == RESULT_CANCELED) {
                onError();
            } else {
                if (data != null && data.getData() != null) {
                    //Delete Camera Image File, if image is picked using gallery
                    /*String pathFromUri = PathToContentURI.getPathFromUri(this, data.getData());
                    if(!mCameraFile.getAbsolutePath().equals(pathFromUri)){
                        mCameraFile.delete();
                    }*/
                    onSuccess(data.getData());
                } else {
                    onSuccess(Uri.fromFile(mCameraFile));
                }
            }
        } else if (requestCode == UCrop.REQUEST_CROP) {
            if (resultCode == RESULT_OK) {
                mCameraFile.delete();
                Intent intent = new Intent();
                try {
                    intent.setData(mImageResizer.getResizedImage(UCrop.getOutput(data)));
                    setResult(Activity.RESULT_OK, intent);
                    finish();
                } catch (FileNotFoundException e) {
                    e.printStackTrace();
                    onError();
                }
            } else {
                onError();
            }
        } else {
            onError();
        }
    }

    @Override
    public void onBackPressed() {
        super.onBackPressed();
        onError();
    }

    /**
     * Handle Success, Crop image if EXTRAS_CROP_IMAGE flag is set to true
     */
    private void onSuccess(Uri path) {
        if (mCropImage) {
            try {
                cropImage(path);
            } catch (IOException e) {
                e.printStackTrace();
                Intent intent = new Intent();
                setResult(Activity.RESULT_CANCELED, intent);
                finish();
            }
        } else {
            Intent intent = new Intent();
            try {
                intent.setData(mImageResizer.getResizedImage(path));
                setResult(Activity.RESULT_OK, intent);
                finish();
            } catch (FileNotFoundException e) {
                e.printStackTrace();
                onError();
            }
        }
    }

    /**
     * Handle Error
     */
    private void onError() {
        if (mCameraFile != null)
            mCameraFile.delete();
        if (mCropImageFile != null)
            mCropImageFile.delete();
        Intent intent = new Intent();
        setResult(Activity.RESULT_CANCELED, intent);
        finish();
    }

    private File mCropImageFile;

    /**
     * @param path Image to be cropped
     * @throws IOException if failed to crop image
     */
    private void cropImage(Uri path) throws IOException {
        UCrop.Options options = new UCrop.Options();
        options.setToolbarColor(ContextCompat.getColor(this, R.color.colorPrimary));
        options.setStatusBarColor(ContextCompat.getColor(this, R.color.colorPrimaryDark));
        mCropImageFile = FileUtil.getCameraFile(true);
        UCrop uCrop = UCrop.of(path, Uri.fromFile(mCropImageFile))
                .withOptions(options);

        if (mCropAspectX>0 && mCropAspectY>0) {
            uCrop.withAspectRatio(mCropAspectX, mCropAspectY);
        }

        try {
            uCrop.start(this, UCrop.REQUEST_CROP);
        }catch (ActivityNotFoundException ex){
            Log.e(TAG, "uCrop not specified in manifest file.");
            ex.printStackTrace();
            onError();
        }
    }

}
