package com.devdigital.customtabs.webview;

import android.app.Activity;
import android.content.Intent;
import android.graphics.Color;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.AnimRes;
import android.support.annotation.DrawableRes;
import android.support.v4.app.ActivityOptionsCompat;
import android.support.v4.content.ContextCompat;

import com.devdigital.customtabs.webview.ui.CustomWebViewActivity;


public class WebViewIntent {

    public static final String EXTRA_TOOLBAR_TITLE = "com.devdigital.customtab.extra.toolbar_title";

    /**
     * Extra that changes the background color for the toolbar. colorRes is an int that specifies a
     * {@link Color}, not a resource id.
     */
    public static final String EXTRA_TOOLBAR_COLOR ="com.devdigital.customtab.extra.toolbar_color";

    public static final String EXTRA_TOOLBAR_TITLE_COLOR ="com.devdigital.customtab.extra.toolbar_title_color";
    public static final String EXTRA_TOOLBAR_SUBTITLE_COLOR ="com.devdigital.customtab.extra.toolbar_subtitle_color";
    public static final String EXTRA_TOOLBAR_ICON_TINT ="com.devdigital.customtab.extra.toolbar_icon_tint";
    public static final String EXTRA_TOOLBAR_CLOSE_ICON ="com.devdigital.customtab.extra.toolbar_close_icon";

    public static final String EXTRA_PROGRESS_TINT ="com.devdigital.customtab.extra.progress_tint";

    /**
     * Bundle constructed out of {@link ActivityOptionsCompat} that will be running when the
     * {@link Activity} that holds the custom tab gets finished. A similar ActivityOptions
     * for creation should be constructed and given to the startActivity() call that
     * launches the custom tab.
     */
    public static final String EXTRA_EXIT_ANIMATION_BUNDLE =
            "android.support.customtabs.extra.EXIT_ANIMATION_BUNDLE";

    private WebViewIntent() {
    }

    public static class Builder {
        private Activity activity;

        private final Intent mIntent;

        private Bundle startAnimationBundle;

        /**
         * @param activity Activity instance
         */
        public Builder(Activity activity) {
            this.activity = activity;
            mIntent = new Intent(activity, CustomWebViewActivity.class);
        }

        public Builder setToolbarTitle(String title) {
            mIntent.putExtra(EXTRA_TOOLBAR_TITLE, title);
            return this;
        }

        public Builder setToolbarTitle(int resId) {
            mIntent.putExtra(EXTRA_TOOLBAR_TITLE, activity.getString(resId));
            return this;
        }

        public Builder setToolbarColor(int colorRes) {
            mIntent.putExtra(EXTRA_TOOLBAR_COLOR, getColor(colorRes));
            return this;
        }

        public Builder setToolbarTitleColor(int colorRes) {
            mIntent.putExtra(EXTRA_TOOLBAR_TITLE_COLOR, getColor(colorRes));
            return this;
        }

        public Builder setToolbarSubtitleColor(int colorRes) {
            mIntent.putExtra(EXTRA_TOOLBAR_SUBTITLE_COLOR, getColor(colorRes));
            return this;
        }

        public Builder setToolbarIconTint(int colorRes) {
            mIntent.putExtra(EXTRA_TOOLBAR_ICON_TINT, getColor(colorRes));
            return this;
        }

        public Builder setCloseIcon(@DrawableRes int iconResId) {
            mIntent.putExtra(EXTRA_TOOLBAR_CLOSE_ICON, iconResId);
            return this;
        }

        public Builder setProgressTint(int progressTint) {
            mIntent.putExtra(EXTRA_PROGRESS_TINT, getColor(progressTint));
            return this;
        }

        /**
         * Sets the start animations.
         *
         * @param enterResId Resource ID of the "enter" animation for the browser.
         * @param exitResId Resource ID of the "exit" animation for the application.
         */
        public Builder setStartAnimations(@AnimRes int enterResId, @AnimRes int exitResId) {
            startAnimationBundle = ActivityOptionsCompat.makeCustomAnimation(activity, enterResId, exitResId).toBundle();
            return this;
        }

        /**
         * Sets the exit animations.
         *
         * @param enterResId Resource ID of the "enter" animation for the application.
         * @param exitResId Resource ID of the "exit" animation for the browser.
         */
        public Builder setExitAnimations(@AnimRes int enterResId, @AnimRes int exitResId) {
            int[] array = {enterResId, exitResId};
            mIntent.putExtra(EXTRA_EXIT_ANIMATION_BUNDLE, array);
            return this;
        }

        public void launchUrl(String url) {
            mIntent.setData(Uri.parse(url));
            if (Build.VERSION.SDK_INT >= 16) {
                activity.startActivity(mIntent, startAnimationBundle);
            } else {
                activity.startActivity(mIntent);
            }
        }

        private int getColor(int colorRes) {
            return ContextCompat.getColor(activity, colorRes);
        }
    }
}
