package com.devdigital.customtabs.chrome;

import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.net.Uri;
import android.os.Build;
import android.support.annotation.AnimRes;
import android.support.annotation.ColorRes;
import android.support.annotation.DrawableRes;
import android.support.customtabs.CustomTabsIntent;
import android.support.v4.content.ContextCompat;

import com.devdigital.customtabs.R;
import com.devdigital.customtabs.util.PackageUtil;

import java.util.HashMap;
import java.util.Map;

/**
 * Open Url in Chrome Custom Tabs
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 27 March 2018
 */
public class ChromeIntent {
    public static class Builder{
        private final Context context;

        @ColorRes
        private int toolbarColor;

        @DrawableRes
        private int closeIcon;

        @AnimRes
        private int startAnimEnter, startAnimExit;
        @AnimRes
        private int exitAnimEnter, exitAnimExit;

        private Map<String, PendingIntent> menuItemMap;
        private Map<Integer, Object[]> actionButtonMap;
        private int closeIconColor;

        private boolean chromeOnly = false;

        /**
         * @param context Application Context
         */
        public Builder(Context context){
            this.context = context;
            this.menuItemMap = new HashMap<>();
            this.actionButtonMap = new HashMap<>();
        }

        /**
         * Set Toolbar color
         *
         * @param toolbarColor Toolbar color
         */
        public Builder setToolbarColor(int toolbarColor) {
            this.toolbarColor = toolbarColor;
            return this;
        }

        /**
         * Set Back Navigation Icon
         *
         * @param closeIcon Close Button icon
         */
        public Builder setCloseIcon(int closeIcon) {
            this.closeIcon = closeIcon;
            return this;
        }

        /**
         * Set Back Navigation Icon Tint
         *
         * @param closeIconColor Close Button icon Tint color
         */
        public Builder setCloseIconColor(int closeIconColor) {
            this.closeIconColor = closeIconColor;
            return this;
        }

        /**
         * Set Activity Start animation
         *
         * @param startAnimEnter Enter Anim Resource
         * @param startAnimExit Exit Anim Resource
         */
        public Builder setStartAnimations(int startAnimEnter, int startAnimExit) {
            this.startAnimEnter = startAnimEnter;
            this.startAnimExit = startAnimExit;
            return this;
        }

        /**
         * Set Activity Finish animation
         *
         * @param exitAnimEnter Enter Anim Resource
         * @param exitAnimExit Exit Anim Resource
         */
        public Builder setExitAnimations(int exitAnimEnter, int exitAnimExit) {
            this.exitAnimEnter = exitAnimEnter;
            this.exitAnimExit = exitAnimExit;
            return this;
        }

        /**
         * Add Menu Item
         *
         * @param title Menu Item title
         * @param cls  Class to notify Menu Item Click
         * @param action String to identify Action for Menu Item Class
         */
        public Builder addMenuItem(String title, Class cls, String action){
            menuItemMap.put(title, getPendingIntent(cls, action));
            return this;
        }

        public Builder chromeOnly(){
            chromeOnly = true;
            return this;
        }

        /**
         * Add Action Button
         *
         * @param icon Action Button Icon
         * @param description   Action Button Description for accessibility
         * @param cls  Class to notify Menu Item Click
         * @param action String to identify Action for Menu Item Class
         */
        public Builder addActionButton(int icon, String description,  Class cls, String action){
            Object[] objects = new Object[2];
            objects[0] = description;
            objects[1] = getPendingIntent(cls, action);
            actionButtonMap.put(icon, objects);
            return this;
        }

        private PendingIntent getPendingIntent(Class cls, String action) {
            Intent actionIntent = new Intent(context, cls);
            actionIntent.setAction(action);
            return PendingIntent.getBroadcast(context, 0, actionIntent, 0);
        }

        /**
         * Open URL
         */
        public void launchUrl(String url){
            CustomTabsIntent.Builder builder = new CustomTabsIntent.Builder();

            if(toolbarColor==0) {
                toolbarColor = R.color.colorPrimary;
            }

            //Change Toolbar Color
            builder.setToolbarColor(ContextCompat.getColor(context, toolbarColor));

            //Show Website Title, In case of google webpage it will show Google as title
            builder.setShowTitle(true);

            //Add Custom Close Button
            if(closeIcon!=0) {
                builder.setCloseButtonIcon(BitmapFactory.decodeResource(context.getResources(), closeIcon));
            }else if(closeIconColor!=0){
                Bitmap icon = BitmapFactory.decodeResource(context.getResources(), R.drawable.ic_arrow_back_black_24dp);
                Bitmap bitmap1 = tintImage(icon, ContextCompat.getColor(context, closeIconColor));
                builder.setCloseButtonIcon(bitmap1);
            }

            //Add Activity Start/Exit animation
            if(startAnimEnter!=0&& startAnimExit!=0)
                builder.setStartAnimations(context, startAnimEnter, startAnimExit);
            if(exitAnimEnter!=0&& exitAnimExit!=0)
                builder.setExitAnimations(context, exitAnimEnter, exitAnimExit);

            //Add Share Menu Item, This will auto handle all the task
            builder.addDefaultShareMenuItem();

            //Add Share Action Button, Manually handle action button click
            for (Integer iconResId : actionButtonMap.keySet()) {
                Bitmap icon = BitmapFactory.decodeResource(context.getResources(), iconResId);
                Object[] objects = actionButtonMap.get(iconResId);
                builder.setActionButton(icon, (String) objects[0], (PendingIntent) objects[1], true);
            }

            //Add Copy Link Menu Item, Manually handle menu item click
            for (String title : menuItemMap.keySet()) {
                builder.addMenuItem(title, menuItemMap.get(title));
            }

            CustomTabsIntent customTabsIntent = builder.build();

            if(chromeOnly && PackageUtil.isPackageExisted(context, PackageUtil.CHROME_PACKAGE_NAME)) {
                customTabsIntent.intent.setPackage(PackageUtil.CHROME_PACKAGE_NAME);
                customTabsIntent.intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
                //It's usually very important for websites to track where their traffic is coming from.
                //Make sure you let them know you are sending them users by setting the referrer when launching your Custom Tab
                customTabsIntent.intent.putExtra(Intent.EXTRA_REFERRER, Uri.parse(Intent.URI_ANDROID_APP_SCHEME + "//" + context.getPackageName()));
            }

            //Launch URL
            customTabsIntent.launchUrl(context, Uri.parse(url));
        }
    }

    private static Bitmap tintImage(Bitmap bitmap, int color) {
        Paint paint = new Paint();
        paint.setColorFilter(new PorterDuffColorFilter(color, PorterDuff.Mode.SRC_IN));
        Bitmap bitmapResult = Bitmap.createBitmap(bitmap.getWidth(), bitmap.getHeight(), Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmapResult);
        canvas.drawBitmap(bitmap, 0, 0, paint);
        return bitmapResult;
    }


}
