package com.devdigital.customtabs;

import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.support.annotation.AnimRes;
import android.support.annotation.ColorRes;
import android.support.annotation.DrawableRes;
import android.support.customtabs.CustomTabsIntent;
import android.support.v4.content.ContextCompat;
import android.support.v4.graphics.drawable.DrawableCompat;

import java.util.HashMap;
import java.util.Map;

/**
 * Open Url in Chrome Custom Tabs
 *
 * @author Dhaval Patel
 * @version 1.0
 * @since 27 March 2018
 */
public class CustomTabsManager {

    private final Context context;

    @ColorRes
    private int toolbarColor;

    @DrawableRes
    private int closeIcon;

    private String url;

    @AnimRes
    private int startAnimEnter, startAnimExit;
    @AnimRes
    private int exitAnimEnter, exitAnimExit;

    private Map<String, PendingIntent> menuItemMap;
    private Map<Integer, Object[]> actionButtonMap;
    private int closeIconColor;

    /**
     * @param context Application Context
     * @param url URL to open in chrome
     */
    public CustomTabsManager(Context context, String url){
        this.context = context;
        this.url = url;
        this.menuItemMap = new HashMap<>();
        this.actionButtonMap = new HashMap<>();
    }

    /**
     * Set Toolbar color
     *
     * @param toolbarColor Toolbar color
     */
    public void setToolbarColor(int toolbarColor) {
        this.toolbarColor = toolbarColor;
    }

    /**
     * Set Back Navigation Icon
     *
     * @param closeIcon Close Button icon
     */
    public void setCloseIcon(int closeIcon) {
        this.closeIcon = closeIcon;
    }

    /**
     * Set Back Navigation Icon Tint
     *
     * @param closeIconColor Close Button icon Tint color
     */
    public void setCloseIconColor(int closeIconColor) {
        this.closeIconColor = closeIconColor;
    }

    /**
     * Set Activity Start animation
     *
     * @param startAnimEnter Enter Anim Resource
     * @param startAnimExit Exit Anim Resource
     */
    public void setStartAnimations(int startAnimEnter, int startAnimExit) {
        this.startAnimEnter = startAnimEnter;
        this.startAnimExit = startAnimExit;
    }

    /**
     * Set Activity Finish animation
     *
     * @param exitAnimEnter Enter Anim Resource
     * @param exitAnimExit Exit Anim Resource
     */
    public void setExitAnimations(int exitAnimEnter, int exitAnimExit) {
        this.exitAnimEnter = exitAnimEnter;
        this.exitAnimExit = exitAnimExit;
    }

    /**
     * Add Menu Item
     *
     * @param title Menu Item title
     * @param cls  Class to notify Menu Item Click
     * @param action String to identify Action for Menu Item Class
     */
    public void addMenuItem(String title, Class cls, String action){
        menuItemMap.put(title, getPendingIntent(cls, action));
    }

    /**
     * Add Action Button
     *
     * @param icon Action Button Icon
     * @param description   Action Button Description for accessibility
     * @param cls  Class to notify Menu Item Click
     * @param action String to identify Action for Menu Item Class
     */
    public void addActionButton(int icon, String description,  Class cls, String action){
        Object[] objects = new Object[2];
        objects[0] = description;
        objects[1] = getPendingIntent(cls, action);
        actionButtonMap.put(icon, objects);
    }

    /**
     * Open URL
     */
    public void launchUrl(){
        CustomTabsIntent.Builder builder = new CustomTabsIntent.Builder();

        if(toolbarColor==0) {
            toolbarColor = R.color.colorPrimary;
        }

        //Change Toolbar Color
        builder.setToolbarColor(ContextCompat.getColor(context, toolbarColor));

        //Show Website Title, In case of google webpage it will show Google as title
        builder.setShowTitle(true);

        //Add Custom Close Button
        if(closeIcon!=0) {
            builder.setCloseButtonIcon(BitmapFactory.decodeResource(context.getResources(), closeIcon));
        }else if(closeIconColor!=0){
            Bitmap icon = BitmapFactory.decodeResource(context.getResources(), R.drawable.ic_arrow_back_black_24dp);
            Bitmap bitmap1 = tintImage(icon, ContextCompat.getColor(context, closeIconColor));
            builder.setCloseButtonIcon(bitmap1);
        }

        //Add Activity Start/Exit animation
        if(startAnimEnter!=0&& startAnimExit!=0)
            builder.setStartAnimations(context, startAnimEnter, startAnimExit);
        if(exitAnimEnter!=0&& exitAnimExit!=0)
            builder.setExitAnimations(context, exitAnimEnter, exitAnimExit);

        //Add Share Menu Item, This will auto handle all the task
        builder.addDefaultShareMenuItem();

        //Add Share Action Button, Manually handle action button click
        for (Integer iconResId : actionButtonMap.keySet()) {
            Bitmap icon = BitmapFactory.decodeResource(context.getResources(), iconResId);
            Object[] objects = actionButtonMap.get(iconResId);
            builder.setActionButton(icon, (String) objects[0], (PendingIntent) objects[1], true);
        }

        //Add Copy Link Menu Item, Manually handle menu item click
        for (String title : menuItemMap.keySet()) {
            builder.addMenuItem(title, menuItemMap.get(title));
        }

        CustomTabsIntent customTabsIntent = builder.build();

        //Launch URL
        customTabsIntent.launchUrl(context, Uri.parse(url));
    }

    private PendingIntent getPendingIntent(Class cls, String action) {
        Intent actionIntent = new Intent(context, cls);
        actionIntent.setAction(action);
        return PendingIntent.getBroadcast(context, 0, actionIntent, 0);
    }

    public static Bitmap tintImage(Bitmap bitmap, int color) {
        Paint paint = new Paint();
        paint.setColorFilter(new PorterDuffColorFilter(color, PorterDuff.Mode.SRC_IN));
        Bitmap bitmapResult = Bitmap.createBitmap(bitmap.getWidth(), bitmap.getHeight(), Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmapResult);
        canvas.drawBitmap(bitmap, 0, 0, paint);
        return bitmapResult;
    }


}
