/***************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 * Copyright 2017 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.jcr.vault.maven.mgr;

import java.io.File;
import java.io.FileNotFoundException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.ArtifactRepositoryPolicy;
import org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout;
import org.apache.maven.artifact.resolver.AbstractArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.eclipse.aether.impl.VersionRangeResolver;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.VersionRangeRequest;
import org.eclipse.aether.resolution.VersionRangeResolutionException;
import org.eclipse.aether.resolution.VersionRangeResult;
import org.eclipse.aether.version.Version;

/**
 * Install a Content Package on a remote CRX or Communique 5 system.
 */
@Mojo(
        name = "install",
        defaultPhase = LifecyclePhase.INSTALL,
        requiresProject = false
)
public class PackageInstallMojo extends AbstractPackageManagerMojo {

    /**
     * The Maven project.
     */
    @Parameter(property = "project", required = true, readonly = true)
    private MavenProject project;

    /**
     * The name of the content package file to install on the target system.
     * If not set, the primary artifact of the project is considered the
     * content package to be installed.
     */
    @Parameter(property = "vault.file", defaultValue = "${project.build.directory}/${project.build.finalName}.zip")
    private File packageFile;

    /**
     * The name of the content package
     */
    @Parameter(property = "vault.name", defaultValue = "${project.artifactId}")
    private String name;

    /**
     * Whether to install (unpack) the uploaded package automatically or not.
     */
    @Parameter(property = "vault.install", defaultValue = "true")
    private boolean install;

    /**
     * The groupId of the artifact to install
     */
    @Parameter(property = "vault.groupId")
    private String groupId;

    /**
     * The artifactId of the artifact to install
     */
    @Parameter(property = "vault.artifactId")
    private String artifactId;

    /**
     * The version of the artifact to install
     */
    @Parameter(property = "vault.version")
    private String version;

    /**
     * The packaging of the artifact to install
     */
    @Parameter(property = "vault.packaging", defaultValue = "zip")
    private String packaging = "zip";

    /**
     * The maven coordinates; A string of the form groupId:artifactId:version[:packaging].
     */
    @Parameter(property = "vault.artifact")
    private String artifact;

    /**
     */
    @Parameter(property = "project.remoteArtifactRepositories", readonly = true, required = true)
    private List pomRemoteRepositories;

    /**
     * The id of the repository from which we'll download the artifact
     */
    @Parameter(property = "vault.repoId", defaultValue="temp")
    private String repositoryId;

    /**
     * The url of the repository from which we'll download the artifact
     */
    @Parameter(property = "vault.repoUrl")
    private String repositoryUrl;

    /**
     */
    @Component
    private ArtifactFactory artifactFactory;

    /**
     */
    @Component
    private ArtifactResolver artifactResolver;
    
    /**
     */
    @Component
    private VersionRangeResolver vRangeResolver;

    /**
     */
    @Component
    private ArtifactRepositoryFactory artifactRepositoryFactory;

    /**
     */
    @Component(hint = "default")
    private ArtifactRepositoryLayout repositoryLayout;

    /**
     */
    @Component
    private ArtifactMetadataSource source;

    /**
     */
    @Parameter(property = "localRepository", readonly = true)
    private ArtifactRepository localRepository;
    
    /**
     */
    @Parameter(property = "session", readonly = true, required = true)
    private MavenSession mavenSession;

    /**
     * If true, fail the build if there is an error while installing.
     */
    @Parameter(property = "vault.failOnError", defaultValue="false")
    private boolean failOnError;

    /**
     * The number of max attempts of upload on AEM.
     */
    @Parameter(property = "vault.maxUploadAttemps", defaultValue="5")
    private short maxUploadAttemps;

    /**
     * The seconds to wait before querying AEM the packages list to check the uploaded package is installed.
     */
    @Parameter(property = "vault.waitingTimeForPackagesService", defaultValue="3")
    private short waitingTimeForPackagesService;

    public void execute() throws MojoExecutionException, MojoFailureException {
        super.execute();

        packageFile = getPackageFile();

        if (packageFile == null) {
            throw new MojoExecutionException("Missing artifact to install.");
        }

        if (!packageFile.canRead()) {
            throw new MojoExecutionException("Cannot access package file " + packageFile);
        }

        uploadPackage();
    }

    protected File getPackageFile() throws MojoFailureException, MojoExecutionException {
        File file = null;

        // if package file was provided in plugin configuration or through the vault.file parameter
        if (packageFile != null) {
            return packageFile;
        }
        // first check if the artifact was configured as parameters
        if (artifact != null || (artifactId != null && groupId != null && version != null)) {
            file = resolvePackageFileFromArtifact();
        }
        // if not, then check if we have a primary artifact
        if (file == null) {
            file = getProjectArtifactFile();
        }
        return file;
    }

    protected File getProjectArtifactFile() {
        return project.getArtifact().getFile();
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private File resolvePackageFileFromArtifact() throws MojoFailureException, MojoExecutionException {
        if (artifactId == null && artifact == null) {
            return null;
        }
        if (artifactId == null) {
            String[] tokens = StringUtils.split(artifact, ":");
            if (tokens.length != 3 && tokens.length != 4) {
                throw new MojoFailureException("Invalid artifact, you must specify "
                        + "groupId:artifactId:version[:packaging] " + artifact);
            }
            groupId = tokens[0];
            artifactId = tokens[1];
            version = tokens[2];
            if (tokens.length == 4)
                packaging = tokens[3];
        }
        Artifact packageArtifact = artifactFactory.createBuildArtifact(groupId, artifactId, version, packaging);

        if (pomRemoteRepositories == null) {
            pomRemoteRepositories = new ArrayList();
        }

        List repoList = new ArrayList(pomRemoteRepositories);

        if (repositoryUrl != null) {
            ArtifactRepositoryPolicy policy =
                new ArtifactRepositoryPolicy( true, ArtifactRepositoryPolicy.UPDATE_POLICY_ALWAYS,
                                              ArtifactRepositoryPolicy.CHECKSUM_POLICY_WARN );
            ArtifactRepository remoteRepo = artifactRepositoryFactory.createArtifactRepository(repositoryId, repositoryUrl,
                    repositoryLayout, policy, policy);

            repoList.add(remoteRepo);
        }

        try {
            VersionRangeRequest rangeRequest = new VersionRangeRequest();
            rangeRequest.setArtifact(new org.eclipse.aether.artifact.DefaultArtifact(packageArtifact.getGroupId(),packageArtifact.getArtifactId(),packageArtifact.getClassifier(),packageArtifact.getType(),packageArtifact.getVersion()));
            for (Object repo : repoList) {
                ArtifactRepository aRepo = (ArtifactRepository) repo;
                RemoteRepository remoteRepo = new RemoteRepository.Builder(aRepo.getId(),
                        "default", aRepo.getUrl()).build(); //$NON-NLS-1$
                rangeRequest.addRepository(remoteRepo);
            }
            VersionRangeResult result;
            try {
                result = vRangeResolver.resolveVersionRange(mavenSession.getRepositorySession(), rangeRequest);
                List<Version> vList = result.getVersions();
                Version toUse = vList.get(vList.size()-1);
                packageArtifact.setVersion(toUse.toString());                                
            } catch (VersionRangeResolutionException e) {
                throw new MojoExecutionException(e.getMessage(), e);
            }

            
            artifactResolver.resolve(packageArtifact, repoList, localRepository);
            getLog().info("Resolved artifact to " + packageArtifact.getFile().getAbsolutePath());
        } catch (AbstractArtifactResolutionException e) {
            throw new MojoExecutionException("Couldn't download artifact: " + e.getMessage(), e);
        }

        return packageArtifact.getFile();
    }

    private void uploadPackage() throws MojoExecutionException, MojoFailureException {
        final String task = install ? "Installing" : "Uploading";
        getLog().info(task + " " + name + " (" + packageFile + ") to " + getTargetURL());
        Path relPath = project.getBasedir().toPath().relativize(packageFile.toPath());

        boolean installed = false;
        short attempts = 0;
        while (!installed && attempts < maxUploadAttemps) {
            if (attempts > 0) {
                getLog().info("Retry upload (" + (attempts + 1) + "/" + maxUploadAttemps + ")");

            }
            ArrayList<Part> parts = new ArrayList<Part>();
            try {
                parts.add(new FilePart("file", packageFile));
            } catch (FileNotFoundException fnfe) {
                // this can be ignored since we are sure packageFile exists
            }

            if (install) {
                parts.add(new StringPart("install", "true"));
            }

            Xpp3Dom result = postRequest(null, parts);
            installed = checkStatus(result);
            if (installed) {
                final Xpp3Dom pkg = getNestedChild(result, "response/data/package");
                logPackage(pkg);

                if (isVerbose()) {
                    final Xpp3Dom log = getNestedChild(result, "response/data/log");
                    if (log != null) {
                        getLog().info(getText(log, null));
                    }
                }
            } else {
                getLog().info("Looks like the " + relPath + " package failed to installed, retrieving the package list...");

                try {
                    Thread.sleep(waitingTimeForPackagesService * 1000);
                } catch (InterruptedException e) {
                    // just ignore it, go to packages list directly
                }

                result = postRequest("ls", (Collection<Part>) null);
                if (checkStatus(result)) {
                    Xpp3Dom packages = getNestedChild(result, "response/data/packages");
                    if (packages != null) {
                        Xpp3Dom installedPackages[] = packages.getChildren("package");
                        if (installedPackages != null) {
                            for (Xpp3Dom installedPackage : installedPackages) {
                                if (packageFile.getName().equals(getText(installedPackage, "downloadName"))) {
                                    installed = true;
                                    getLog().info(relPath + " package successfully installed and found in the packages list.");
                                    logPackage(installedPackage);
                                }
                            }
                        } else {
                            getLog().warn("Impossible to access to the installed packages list, data not available on server");
                        }
                    } else {
                        getLog().warn("Impossible to access to the installed packages list, data not available on server");
                    }
                }
            }

            if (!installed) {
                getLog().warn(relPath + " package not installed.");
                attempts++;
            }
        }

        if (!installed && failOnError) {
            throw new MojoFailureException("Error while installing package. Check log for details.");
        }
    }

}
