/***************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 * Copyright 2017 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.jcr.vault.maven.mgr;

import java.io.InputStreamReader;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.settings.Proxy;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.codehaus.plexus.util.xml.Xpp3DomBuilder;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcher;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcherException;

import com.day.jcr.vault.maven.mgr.ssl.EasySSLProtocolSocketFactory;

abstract class AbstractPackageManagerMojo extends AbstractMojo {

    /**
     * The Maven settings in vigor to get at the configured proxies
     */
    @Parameter(property = "settings", required = true, readonly = true)
    private Settings settings;

    /**
     * The URL to the HTTP service API of the CRX package manager. See <a href=
     * "http://wiki.day.com/CrxPackageManager.html#HTTP_Service_Interface">HTTP
     * Service Interface</a> for details on this interface.
     */
    @Parameter(
            property = "vault.targetURL",
            defaultValue = "http://localhost:4502/crx/packmgr/service.jsp",
            required = true)
    private String targetURL;
    
    /**
     */
    @Parameter(
            property = "vault.serviceURL",
            defaultValue="http://localhost:4502/crx/packmgr/service/.json",
            required = true)
    private String serviceURL;

    /**
     * The user name to authenticate as against the remote CRX system.
     */
    @Parameter(property = "vault.userId",
            defaultValue="admin",
            required = true)
    private String userId;

    /**
     * The password to authenticate against the remote CRX system.
     */
    @Parameter(property = "vault.password",
            defaultValue="admin",
            required = true)
    private String password;

    /**
     * The server id with which to get the username and password from
     * the user's settings file.
     */
    @Parameter(property = "vault.serverId")
    private String serverId;

    /**
     * Enable verbose logging when set to <code>true</code>.
     */
    @Parameter(
            property = "vault.verbose",
            defaultValue="false")
    private boolean verbose;

    /**
     * The connection timeout to set when communicating with the package manager
     * service. Default value is 5 seconds. Value is specified in seconds.
     */
    @Parameter(
            property = "vault.timeout",
            defaultValue="5")
    private int timeout;

    /**
     * Setting this to <code>false</code> disables considering the use of any of
     * the active proxies configured in the Maven Settings. By default the first
     * active proxy configuration in the Maven Settings is used to proxy any
     * request to the package manager.
     */
    @Parameter(
            property = "vault.useProxy",
            defaultValue="true")
    private boolean useProxy;
    
    /**
     */
    @Parameter(
            property = "vault.relaxedSSLCheck",
            defaultValue="false")
    private boolean relaxedSSLCheck;

    /**
     * Component which decrypts passwords.
     */
    @Component(hint = "mojo")
    private SecDispatcher securityDispatcher;

    public void execute() throws MojoExecutionException, MojoFailureException {
        if (relaxedSSLCheck) {
            Protocol.registerProtocol("https",  new Protocol("https", new EasySSLProtocolSocketFactory(), 443));
        }
    }

    public String getTargetURL() {
        return targetURL;
    }

    public String getUserId() {
        if (serverId != null) {
            Server server = settings.getServer(serverId);
            if (server == null) {
                throw new IllegalArgumentException(String.format("Server ID %s does not exist in settings.xml.", serverId));
            }
            return server.getUsername();
        } else {
            return userId;
        }
    }

    public String getPassword() throws SecDispatcherException {
        if (serverId != null) {
            Server server = settings.getServer(serverId);
            if (server == null) {
                throw new IllegalArgumentException(String.format("Server ID %s does not exist in settings.xml.", serverId));
            }

            return securityDispatcher.decrypt(server.getPassword());

        } else {
            return password;
        }
    }

    public boolean isVerbose() {
        return verbose;
    }

    public int getTimoutMs() {
        return 1000 * timeout;
    }

    protected Xpp3Dom postRequest(String command, Collection<Part> parameters)
            throws MojoExecutionException {

        // ensure parameters
        if (parameters == null) {
            parameters = new ArrayList<Part>();
        }
        if (command != null) {
            parameters.add(new StringPart("cmd", command));
        }
        if (parameters.isEmpty()) {
            getLog().error(
                "Not POSTing to " + getTargetURL() + ": Missing parameters");
            return null;
        }

        PostMethod filePost = new PostMethod(getTargetURL());
        filePost.getParams().setBooleanParameter(
            HttpMethodParams.USE_EXPECT_CONTINUE, true);
        filePost.setRequestHeader("referer", "about:blank");
        try {

            Part[] parts = new Part[parameters.size()];
            parameters.toArray(parts);

            filePost.setRequestEntity(new MultipartRequestEntity(parts,
                filePost.getParams()));

            HttpClient client = new HttpClient();
            client.getParams().setAuthenticationPreemptive(true);
            UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(
                getUserId(), getPassword());
            client.getState().setCredentials(AuthScope.ANY, credentials);

            setupProxy(client);

            client.getHttpConnectionManager().getParams().setConnectionTimeout(
                getTimoutMs());
            int status = client.executeMethod(filePost);
            if (status == HttpStatus.SC_OK) {
                StringWriter out = new StringWriter();
                InputStreamReader in = new InputStreamReader(
                        filePost.getResponseBodyAsStream(), filePost.getRequestCharSet());
                IOUtil.copy(in, out);
                String ret = out.toString();
                Xpp3Dom dom;
                try {
                    dom = Xpp3DomBuilder.build(new StringReader(ret));
                } catch (XmlPullParserException e) {
                    // assume malformed XML due to missing escaping in <log></log> section
                    Matcher m = Pattern.compile("<log>(.*)</log>", Pattern.DOTALL).matcher(ret);
                    StringBuffer result = new StringBuffer();
                    while (m.find()) {
                        String log = "<log><![CDATA[" + m.group(1) + "]]></log>";
                        m.appendReplacement(result, Matcher.quoteReplacement(log));
                    }
                    m.appendTail(result);
                    ret = result.toString();
                    dom = Xpp3DomBuilder.build(new StringReader(ret));
                }
                return dom;
            }

            getLog().error(
                "Request to " + getTargetURL() + " failed, response="
                    + HttpStatus.getStatusText(status));

            return null;

        } catch (Exception ex) {

            throw new MojoExecutionException(ex.getMessage(), ex);

        } finally {

            filePost.releaseConnection();

        }
    }
    

    protected JSONObject postRequest(Command command, String path) throws MojoExecutionException {
        if (command == null) {
            getLog().error("command was null. Not POSTing.");
            return null;
        }
        
        if (path == null) {
            getLog().error("path was null. Not POSTing.");
            return null;
        }
        
        String url = getTargetURL(path, command);
        getLog().info("POSTing to " + url);
        final PostMethod post = new PostMethod(url);
        post.getParams().setBooleanParameter(
            HttpMethodParams.USE_EXPECT_CONTINUE, true);
        post.setRequestHeader("referer", "about:blank");
        try {

            //Part[] parts = new Part[parameters.size()];
            //parameters.toArray(parts);

            //filePost.setRequestEntity(new MultipartRequestEntity(parts,
            //    filePost.getParams()));

            HttpClient client = new HttpClient();
            client.getParams().setAuthenticationPreemptive(true);
            UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(
                getUserId(), getPassword());
            client.getState().setCredentials(AuthScope.ANY, credentials);

            setupProxy(client);

            client.getHttpConnectionManager().getParams().setConnectionTimeout(
                getTimoutMs());
            int status = client.executeMethod(post);
            if (status == HttpStatus.SC_OK) {
                StringWriter out = new StringWriter();
                InputStreamReader in = new InputStreamReader(
                        post.getResponseBodyAsStream(), post.getRequestCharSet());
                IOUtil.copy(in, out);
                String ret = out.toString();
                return new JSONObject(ret);
            }

            getLog().error(
                "Request to " + url + " failed, response="
                    + HttpStatus.getStatusText(status));

            return null;

        } catch (Exception ex) {

            throw new MojoExecutionException(ex.getMessage(), ex);

        } finally {

            post.releaseConnection();

        }
    }

    protected String getTargetURL(String path, Command command) {
        return String.format("%s%s?cmd=%s", this.serviceURL, path, command.getQueryParamValue());
    }

    protected void setupProxy(HttpClient client) {
        List<Proxy> proxies = settings.getProxies();
        if (useProxy && proxies != null && !proxies.isEmpty()) {
            getLog().debug("Considering " + proxies.size() + " proxies");
            for (Proxy proxy : proxies) {
                if (proxy.isActive()) {
                    getLog().debug(
                        "Using proxy " + proxy.getHost() + ":"
                            + proxy.getPort());
                    client.getHostConfiguration().setProxy(proxy.getHost(),
                        proxy.getPort());
                    if (proxy.getUsername() != null) {
                        UsernamePasswordCredentials proxyCredentials = new UsernamePasswordCredentials(
                            proxy.getUsername(), proxy.getPassword());
                        client.getState().setProxyCredentials(AuthScope.ANY,
                            proxyCredentials);
                    }
                    break;
                }
            }
        } else {
            getLog().debug(
                "Proxying disabled (useProxy=" + useProxy
                    + ") or no proxies configured");
        }
    }

    // ---------- XPP3 DOM support

    protected Xpp3Dom getNestedChild(final Xpp3Dom element, final String path) {
        String[] parts = path.split("/");

        Xpp3Dom current = element;
        for (int i = 0; i < parts.length && current != null; i++) {
            current = current.getChild(parts[i]);
        }

        return current;
    }

    protected String getText(final Xpp3Dom parent, final String childName) {
        final Xpp3Dom child = (childName == null)
                ? parent
                : parent.getChild(childName);
        if (child != null) {
            return child.getValue();
        }

        return parent.getAttribute(childName);
    }

    protected boolean checkStatus(Xpp3Dom result) {

        // request failure has already been logged, return false quickly
        if (result == null) {
            return false;
        }

        final Xpp3Dom status = getNestedChild(result, "response/status");
        if (status == null) {
            getLog().error(
                "Missing response status information in response: " + result);
            return false;
        }

        final String code = getText(status, "code");
        if ("200".equals(code)) {
            getLog().debug("Request succeeded");
            return true;
        }

        getLog().error(
            "Request failed: " + getText(status, null) + " ("
                + getText(status, "code") + ")");
        return false;
    }

    protected boolean checkStatus(JSONObject result) throws MojoExecutionException {
        // request failure has already been logged, return false quickly
        if (result == null) {
            return false;
        }
        if (result.has("success")) {
            try {
                boolean success = result.getBoolean("success");
                if (success) {
                    getLog().debug("Request succeeded");
                    return true;
                } else {
                    String msg;
                    if (result.has("msg")) {
                        msg = result.getString("msg");
                    } else {
                        msg = "Unknown error. Full JSON response: " + result.toString(2);
                    }
                    getLog().error(
                        "Request failed: " + msg);
                    return false;
                }
            } catch (JSONException e) {
                throw new MojoExecutionException("Usparsable JSON Object", e);
            }
        } else {
            getLog().error(
                    "Missing response status information in response: " + result);
                return false;
        }
    }

    protected void logPackage(Xpp3Dom pkg) {
        if (pkg != null) {
            StringBuilder buf = new StringBuilder();

            buf.append(getText(pkg, "group"));
            buf.append(":");
            buf.append(getText(pkg, "name"));
            String version = getText(pkg, "version");
            if (version != null && version.length() > 0) {
                buf.append(", ");
                buf.append(version);
            }
            buf.append(" (");
            buf.append(getText(pkg, "size"));
            buf.append(" bytes)");

            getLog().info(buf);

            if (isVerbose()) {
                logPackageDate(pkg, "Created ", "created");
                logPackageDate(pkg, "Modified", "lastModified");
                logPackageDate(pkg, "Unpacked", "lastUnpacked");
                getLog().info("");
            }
        }
    }

    private void logPackageDate(final Xpp3Dom pkg, final String label,
            final String kind) {
        final String user = getText(pkg, kind + "By");

        StringBuilder builder = new StringBuilder();
        builder.append("    ").append(label).append(": ");
        if (user != null && !"null".equals(user)) {
            builder.append(getText(pkg, kind)).append(" by ").append(user);
        } else {
            builder.append("-");
        }
        getLog().info(builder);
    }

}
