/***************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 * Copyright 2017 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.jcr.vault.maven.mgr;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.sling.commons.json.JSONObject;
import org.codehaus.plexus.util.xml.Xpp3Dom;

/**
 * Abstract class which supports mojos which can either either a group/name
 * combination or a path.
 */
abstract class AbstractNameOrPathPackageManagerMojo extends AbstractPackageManagerMojo {

    private static final String PATH_FORMAT = "/etc/packages/%s/%s-%s.zip";

    /**
     * The group of the package on the target system.
     */
    @Parameter(property = "vault.group")
    private String group;

    /**
     * The name of the package on the target system.
     */
    @Parameter(property = "vault.name")
    private String name;

    /**
     * The path of the package to build on the target system.
     */
    @Parameter(property = "vault.path")
    private String path;

    /**
     * The version of the package on the target system.
     */
    @Parameter(property = "vault.version")
    private String version;

    /**
     * If true, fail the build if there is an error while executing.
     */
    @Parameter(property = "vault.failOnError", defaultValue = "false")
    private boolean failOnError;

    protected abstract String getLegacyCommandName();

    protected abstract Command getCommand();

    protected abstract String getInfoMessageFormat();

    protected abstract String getErrorMessage();

    protected void logSuccess(String description, String targetUrl) {
        getLog().info(String.format(getInfoMessageFormat(), description, targetUrl));
    }

    protected static String constructPath(String group, String name, String version) {
        return String.format(PATH_FORMAT, group, name, version);
    }

    public void execute() throws MojoExecutionException, MojoFailureException {
        super.execute();

        if (name != null) {
            if (group != null && version !=  null) {
                executeOnPath(constructPath(group, name, version));
            } else {
                List<Part> parts = new ArrayList<Part>();
                String displayName = name;
                parts.add(new StringPart("name", name));
                if (group != null) {
                    displayName = group + ":" + displayName;
                    parts.add(new StringPart("group", group));
                }
                Xpp3Dom result = postRequest(getLegacyCommandName(), parts);
        
                if (checkStatus(result)) {
                    logSuccess(displayName, getTargetURL());
                } else if (failOnError) {
                    throw new MojoFailureException(getErrorMessage());
                }
            }
        } else if (path != null) {
            executeOnPath(path);
        } else {
            throw new MojoExecutionException("Either name or path must be specified.");
        }
    }

    private void executeOnPath(String packagePath) throws MojoExecutionException, MojoFailureException {
        JSONObject result = postRequest(getCommand(), packagePath);

        if (checkStatus(result)) {
            logSuccess(packagePath, getTargetURL(packagePath, getCommand()));
        } else if (failOnError) {
            throw new MojoFailureException(getErrorMessage());
        }
    }
}
