/*
 * Decompiled with CFR 0.152.
 */
package com.day.crx.persistence.bundle;

import com.day.crx.persistence.bundle.ConnectionFactory;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import javax.jcr.RepositoryException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionRecoveryManager {
    private static Logger log = LoggerFactory.getLogger(ConnectionRecoveryManager.class);
    private final String driver;
    private final String url;
    private final String user;
    private final String password;
    private Connection connection;
    private boolean autoReconnect = true;
    private final boolean block;
    private static final int SLEEP_BEFORE_RECONNECT = 500;
    public static final int TRIALS = 20;
    private HashMap<String, PreparedStatement> preparedStatements = new HashMap();
    private boolean isClosed;

    public ConnectionRecoveryManager(boolean block, String driver, String url, String user, String password) throws RepositoryException {
        this.block = block;
        this.driver = driver;
        this.url = url;
        this.user = user;
        this.password = password;
        try {
            this.setupConnection();
            this.isClosed = false;
        }
        catch (SQLException e) {
            this.logException("could not setup connection", e);
            this.close();
        }
    }

    public synchronized Connection getConnection() throws SQLException, RepositoryException {
        if (this.isClosed) {
            if (this.autoReconnect) {
                this.reestablishConnection();
            } else {
                throw new SQLException("connection has been closed and autoReconnect == false");
            }
        }
        return this.connection;
    }

    public synchronized void setAutoReconnect(boolean autoReconnect) {
        this.autoReconnect = autoReconnect;
    }

    public synchronized ResultSet executeQuery(String sql) throws SQLException, RepositoryException {
        int trials = 2;
        SQLException lastException = null;
        while (true) {
            --trials;
            try {
                return this.executeQueryInternal(sql);
            }
            catch (SQLException e) {
                lastException = e;
                if (this.autoReconnect && (this.block || trials > 0)) continue;
                throw lastException;
            }
            break;
        }
    }

    private ResultSet executeQueryInternal(String sql) throws SQLException, RepositoryException {
        PreparedStatement stmt = null;
        try {
            stmt = this.preparedStatements.get(sql);
            if (stmt == null) {
                stmt = this.getConnection().prepareStatement(sql);
                this.preparedStatements.put(sql, stmt);
            }
            ResultSet resultSet = stmt.executeQuery();
            this.resetStatement(stmt);
            return resultSet;
        }
        catch (SQLException e) {
            try {
                this.logException("could not execute statement", e);
                this.close();
                throw e;
            }
            catch (Throwable throwable) {
                this.resetStatement(stmt);
                throw throwable;
            }
        }
    }

    public PreparedStatement executeStmt(String sql, Object[] params) throws SQLException, RepositoryException {
        return this.executeStmt(sql, params, false, 0);
    }

    public synchronized PreparedStatement executeStmt(String sql, Object[] params, boolean returnGeneratedKeys, int maxRows) throws SQLException, RepositoryException {
        int trials = 2;
        SQLException lastException = null;
        while (true) {
            --trials;
            try {
                return this.executeStmtInternal(sql, params, returnGeneratedKeys, maxRows);
            }
            catch (SQLException e) {
                lastException = e;
                if (this.autoReconnect && (this.block || trials > 0)) continue;
                throw lastException;
            }
            break;
        }
    }

    private PreparedStatement executeStmtInternal(String sql, Object[] params, boolean returnGeneratedKeys, int maxRows) throws SQLException, RepositoryException {
        try {
            PreparedStatement stmt;
            String key = sql;
            if (returnGeneratedKeys) {
                key = key + " RETURN_GENERATED_KEYS";
            }
            if ((stmt = this.preparedStatements.get(key)) == null) {
                stmt = returnGeneratedKeys ? this.getConnection().prepareStatement(sql, 1) : this.getConnection().prepareStatement(sql);
                this.preparedStatements.put(key, stmt);
            }
            stmt.setMaxRows(maxRows);
            return this.executeStmtInternal(params, stmt);
        }
        catch (SQLException e) {
            this.logException("could not execute statement", e);
            this.close();
            throw e;
        }
    }

    public synchronized void close() {
        this.preparedStatements.clear();
        try {
            if (this.connection != null) {
                if (!this.connection.getAutoCommit()) {
                    this.connection.rollback();
                }
                this.connection.close();
            }
        }
        catch (SQLException e) {
            this.logException("failed to close connection", e);
        }
        this.connection = null;
        this.isClosed = true;
    }

    private void setupConnection() throws SQLException, RepositoryException {
        try {
            this.connection = ConnectionFactory.getConnection(this.driver, this.url, this.user, this.password);
        }
        catch (SQLException e) {
            log.warn("Could not connect; driver: " + this.driver + " url: " + this.url + " user: " + this.user + " error: " + e.toString(), (Throwable)e);
            throw e;
        }
        if (!this.connection.getAutoCommit()) {
            this.connection.setAutoCommit(true);
        }
        try {
            DatabaseMetaData meta = this.connection.getMetaData();
            log.info("Database: " + meta.getDatabaseProductName() + " / " + meta.getDatabaseProductVersion());
            log.info("Driver: " + meta.getDriverName() + " / " + meta.getDriverVersion());
        }
        catch (SQLException e) {
            log.warn("Can not retrieve database and driver name / version", (Throwable)e);
        }
    }

    private PreparedStatement executeStmtInternal(Object[] params, PreparedStatement stmt) throws SQLException {
        for (int i = 0; params != null && i < params.length; ++i) {
            Object p = params[i];
            if (p instanceof StreamWrapper) {
                StreamWrapper wrapper = (StreamWrapper)p;
                stmt.setBinaryStream(i + 1, wrapper.stream, (int)wrapper.size);
                continue;
            }
            if (p instanceof InputStream) {
                InputStream stream = (InputStream)p;
                stmt.setBinaryStream(i + 1, stream, -1);
                continue;
            }
            stmt.setObject(i + 1, p);
        }
        stmt.execute();
        this.resetStatement(stmt);
        return stmt;
    }

    private void reestablishConnection() throws SQLException, RepositoryException {
        long trials = 20L;
        SQLException exception = null;
        this.close();
        if (this.block) {
            log.warn("blocking until database connection is up again...");
        }
        while (trials-- >= 0L || this.block) {
            exception = null;
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException ignore) {
                // empty catch block
            }
            try {
                this.setupConnection();
                this.isClosed = false;
                break;
            }
            catch (SQLException e) {
                exception = e;
                this.close();
            }
        }
        if (exception != null) {
            throw exception;
        }
        if (this.block) {
            log.warn("database connection is up again!");
        }
    }

    private void resetStatement(PreparedStatement stmt) {
        if (stmt != null) {
            try {
                stmt.clearParameters();
                stmt.clearWarnings();
            }
            catch (SQLException se) {
                this.logException("Failed resetting PreparedStatement", se);
            }
        }
    }

    private void logException(String message, SQLException se) {
        message = message == null ? "" : message;
        log.error(message + ", reason: " + se.getMessage() + ", state/code: " + se.getSQLState() + "/" + se.getErrorCode());
        log.debug("   dump:", (Throwable)se);
    }

    public static class StreamWrapper {
        private final InputStream stream;
        private final long size;

        public StreamWrapper(InputStream in, long size) {
            this.stream = in;
            this.size = size;
        }
    }
}

