/*
 * Decompiled with CFR 0.152.
 */
package com.day.crx.core.cluster;

import EDU.oswego.cs.dl.util.concurrent.SynchronizedBoolean;
import com.day.crx.core.cluster.ClusterLifecycleListener;
import com.day.crx.core.cluster.ClusterMaster;
import com.day.crx.core.cluster.ClusterNodeInfo;
import com.day.crx.core.cluster.ClusterProperties;
import com.day.crx.core.cluster.ClusterSkeleton;
import com.day.crx.core.cluster.ClusterSlave;
import com.day.crx.core.cluster.IncomingCall;
import com.day.crx.core.cluster.OutgoingCall;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class ClusterController {
    static final int OP_LOGIN = 0;
    static final int OP_JOIN = 1;
    static final int OP_PASSIVE = 2;
    static final int OP_STOP_LISTENER = 3;
    static final int OP_RESTART = 4;
    static final int OP_MASTER_INFO = 5;
    static final int OP_SLAVE_INFOS = 6;
    static final int OP_GET_FILE_CONTENTS = 7;
    static final int OP_SLAVE_JOINED = 8;
    static final int OP_SYNC_MEMBERS = 9;
    public static final String CLUSTER_NODE_ID_FILE = "cluster_node.id";
    public static final String CLUSTERED_FILE_NAME = "clustered.txt";
    protected static final int CONNECT_TIMEOUT_MS = Integer.getInteger("socket.connectTimeout", 1000);
    private static final int RECEIVE_TIMEOUT_MS = Integer.getInteger("socket.receiveTimeout", 60000);
    private final int waitForMasterRetries = Integer.getInteger("com.day.crx.core.cluster.WaitForMasterRetries", 60);
    private static final int MAX_RETRIES = 3;
    private static final int MAX_RETRY_DELAY = 1000;
    private static Logger log = LoggerFactory.getLogger(ClusterController.class);
    private static final HashMap<File, ClusterController> controllers = new HashMap();
    private final Map<String, ClusterSkeleton> skeletons = Collections.synchronizedMap(new HashMap());
    private final Set<ClusterLifecycleListener> listeners = Collections.synchronizedSet(new HashSet());
    private final SynchronizedBoolean stopped = new SynchronizedBoolean(false);
    private final SynchronizedBoolean paused = new SynchronizedBoolean(false);
    private final Random random = new Random();
    private final File repositoryHome;
    private final File clustered;
    private boolean exclusiveMode;
    private InetAddress bindaddr;
    private int[] candidatePorts = new int[]{8088, 8089, 8090, 8091, 8092, 8093};
    private ClusterProperties cprops;
    private final Properties props = new Properties();
    private String clusterNodeId;
    private ClusterMaster master;
    private ClusterSlave slave;
    private int connectTimeout = CONNECT_TIMEOUT_MS;
    private int receiveTimeout = RECEIVE_TIMEOUT_MS;
    private boolean autoStartStop = true;
    private final Object stateChangeLock = new Object();
    private boolean masterListenerStarted;
    private boolean becomeMasterOnTimeout;
    private boolean disableRestart;

    public static ClusterController getInstance(File repositoryHome) throws IOException {
        return ClusterController.getInstance(repositoryHome, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static ClusterController getInstance(File repositoryHome, String clusterNodeId) throws IOException {
        HashMap<File, ClusterController> hashMap = controllers;
        synchronized (hashMap) {
            File key = repositoryHome.getCanonicalFile();
            ClusterController c = controllers.get(key);
            if (c == null) {
                c = new ClusterController(key, clusterNodeId);
                controllers.put(key, c);
            }
            return c;
        }
    }

    ClusterController(File repositoryHome, String clusterNodeId) throws IOException {
        this.repositoryHome = repositoryHome;
        this.clusterNodeId = clusterNodeId;
        this.clustered = new File(repositoryHome, CLUSTERED_FILE_NAME);
        this.init();
    }

    protected void init() throws IOException {
        if (this.clusterNodeId == null) {
            File f = new File(this.repositoryHome, CLUSTER_NODE_ID_FILE);
            if (f.canRead()) {
                this.clusterNodeId = FileUtils.readFileToString((File)f).trim();
            } else {
                throw new IOException("File does not exist or is not readable: " + f.getPath());
            }
        }
        this.cprops = new ClusterProperties(new File(this.repositoryHome, "cluster.properties"));
        this.cprops.load();
        String clusterId = this.cprops.getClusterId();
        if (clusterId == null) {
            clusterId = UUID.randomUUID().toString();
            this.cprops.setClusterId(clusterId);
            this.cprops.addMember(this.clusterNodeId);
            this.cprops.save();
            log.info("Generated new cluster id: {}", (Object)clusterId);
        }
    }

    public String getClusterId() {
        return this.cprops.getClusterId();
    }

    String[] getClusterMembers() {
        return this.cprops.getMembers();
    }

    public String getClusterNodeId() {
        return this.clusterNodeId;
    }

    public File getRepositoryHome() {
        return this.repositoryHome;
    }

    public InetAddress getMasterInetAddress() {
        if (this.slave != null) {
            return this.slave.getMasterInetAddress();
        }
        if (this.master != null) {
            return this.master.getInetAddress();
        }
        return null;
    }

    public InetSocketAddress getMasterSocketAddress() {
        if (this.slave != null) {
            return this.slave.getMasterSocketAddress();
        }
        if (this.master != null) {
            return this.master.getSocketAddress();
        }
        return null;
    }

    public ClusterNodeInfo getMasterInfo() throws UnknownHostException {
        if (this.slave != null) {
            try {
                return this.slave.getMasterInfo();
            }
            catch (IOException e) {
                log.warn("Unable to obtain master info.", (Throwable)e);
            }
        } else if (this.master != null) {
            String hostname = this.getBindAddrHostName();
            return new ClusterNodeInfo(this.clusterNodeId, this.getOS(), hostname != null ? hostname : "", this.getRepositoryHome().getAbsolutePath());
        }
        return null;
    }

    public ClusterNodeInfo[] getSlaveInfos() {
        if (this.slave != null) {
            try {
                return this.slave.getSlaveInfos();
            }
            catch (IOException e) {
                log.warn("Unable to obtain slave infos.", (Throwable)e);
            }
        } else if (this.master != null) {
            return this.master.getSlaveInfos();
        }
        return null;
    }

    public InetAddress getBindAddress() {
        return this.bindaddr;
    }

    public String getBindAddrHostName() {
        InetAddress bindaddr = this.bindaddr;
        if (bindaddr == null) {
            try {
                bindaddr = InetAddress.getLocalHost();
            }
            catch (UnknownHostException e) {
                log.warn("Unable to find IP address for local host.", (Throwable)e);
                return null;
            }
        }
        return bindaddr.getHostName();
    }

    public void setBindAddress(InetAddress bindaddr) {
        if (this.isStarted()) {
            log.warn("Controller already started: setting bind address has no effect.");
            return;
        }
        this.bindaddr = bindaddr;
    }

    public void setCandidatePorts(int[] candidatePorts) throws IllegalArgumentException {
        if (this.isStarted()) {
            log.warn("Controller already started: setting candidate ports has no effect.");
            return;
        }
        if (candidatePorts == null || candidatePorts.length == 0) {
            log.warn("candidatePorts must have non-zero length, ignored.");
            return;
        }
        this.candidatePorts = new int[candidatePorts.length];
        System.arraycopy(candidatePorts, 0, this.candidatePorts, 0, candidatePorts.length);
    }

    public void setConnectTimeout(int timeout) {
        if (this.isStarted()) {
            log.warn("Controller already started: setting connect timeout has no effect.");
            return;
        }
        this.connectTimeout = timeout;
    }

    public void setReceiveTimeout(int timeout) {
        if (this.isStarted()) {
            log.warn("Controller already started: setting receive timeout has no effect.");
            return;
        }
        this.receiveTimeout = timeout;
    }

    public void setBecomeMasterOnTimeout(boolean becomeMasterOnTimeout) {
        if (this.isStarted()) {
            log.warn("Controller already started: setting become on master timeout has no effect.");
            return;
        }
        this.becomeMasterOnTimeout = becomeMasterOnTimeout;
    }

    public void setAutoStartStop(boolean autoStartStop) {
        this.autoStartStop = autoStartStop;
    }

    public void setProperty(String key, String value) {
        this.props.setProperty(key, value);
    }

    public Properties getProperties() {
        Properties props = new Properties();
        Enumeration<?> names = this.props.propertyNames();
        while (names.hasMoreElements()) {
            String key = (String)names.nextElement();
            props.setProperty(key, this.props.getProperty(key));
        }
        this.cprops.addAll(props);
        return props;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void join(String clusterId, InetAddress addr) throws IOException, IllegalStateException {
        Object object = this.stateChangeLock;
        synchronized (object) {
            this.checkNotStopped();
            if (this.isStarted()) {
                throw new IllegalStateException("Already started.");
            }
            try {
                this.startSlave(addr, clusterId);
            }
            catch (IOException e) {
                String msg = MessageFormat.format("Slave was unable to connect to {0}: {1}", addr.getHostAddress(), e.getMessage());
                throw new IOException(msg);
            }
            String masterId = this.slave.getMasterId();
            this.cprops.addMember(masterId);
            this.cprops.setClusterId(clusterId);
            if (!this.isLocalIPAddress(addr)) {
                this.cprops.addAddress(addr.getHostAddress());
            }
            this.cprops.save();
            this.syncMembers(this.cprops.getMembers(), this.cprops.getAddresses());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void becomeMaster() throws IOException, IllegalStateException {
        Object object = this.stateChangeLock;
        synchronized (object) {
            this.checkNotStopped();
            if (this.master != null) {
                return;
            }
            if (this.slave == null) {
                throw new IllegalStateException("Not started.");
            }
            InetSocketAddress sa = this.slave.getMasterSocketAddress();
            log.info("Telling {} to stop listening.", (Object)sa);
            this.slave.newCall("", 3).execute();
            try {
                this.attempt(new IOOperation(){

                    public void execute() throws IOException {
                        ClusterController.this.startMasterListener(true);
                        ClusterController.this.activateSkeletons();
                        ClusterController.this.acceptConnections();
                    }
                }, null);
            }
            catch (IOException e) {
                throw e;
            }
            finally {
                try {
                    log.info("Telling {} to restart.", (Object)sa);
                    this.slave.newCall("", 4).execute();
                }
                catch (IOException e) {
                    log.warn("Unable to restart master on {}.", (Object)sa);
                }
                finally {
                    this.slave.stop();
                    this.slave = null;
                }
            }
            return;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void start() throws IOException, IllegalStateException {
        Object object = this.stateChangeLock;
        synchronized (object) {
            this.checkNotStopped();
            if (this.isStarted()) {
                return;
            }
            boolean startOnlyAsSlave = this.clustered.exists();
            if (this.exclusiveMode) {
                startOnlyAsSlave = false;
            }
            if (startOnlyAsSlave) {
                log.info("Trying to connect to a master, as the file " + this.clustered.getName() + " exists.");
                this.masterListenerStarted = false;
            } else {
                this.masterListenerStarted = this.startMasterListener(false);
            }
            int i = 0;
            while (true) {
                if (!this.exclusiveMode && this.startAsSlave()) {
                    String masterId;
                    if (this.masterListenerStarted) {
                        this.master.stop();
                        this.master = null;
                        this.masterListenerStarted = false;
                    }
                    if (this.cprops.addMember(masterId = this.slave.getMasterId())) {
                        this.cprops.saveQuietly();
                    }
                    this.syncMembers(this.cprops.getMembers(), this.cprops.getAddresses());
                    return;
                }
                if (startOnlyAsSlave) {
                    if (i > this.waitForMasterRetries) {
                        String msg = "To avoid becoming out-of-sync, this cluster node was not started as master.\nThe file " + this.clustered.getAbsolutePath() + " exists,\n" + "meaning another cluster node was running the last time. Could not connect to a master node now.\n" + "To start anyway, please delete the file and re-try.";
                        log.error(msg);
                        throw new IOException(msg);
                    }
                    if (i == 0) {
                        log.info("Re-trying for at most " + this.waitForMasterRetries + " seconds");
                    }
                    try {
                        Thread.sleep(1000L);
                    }
                    catch (InterruptedException e) {}
                } else {
                    if (!this.masterListenerStarted || !this.master.getPotentialMasterDetected().compareAndSet(true, false)) break;
                    if (i > 10) {
                        log.warn("Potential master detected, but can't connect as slave: " + i);
                    }
                }
                ++i;
            }
            if (!this.masterListenerStarted) {
                throw new IOException("Could not create a listener on any of the following ports: " + Arrays.toString(this.candidatePorts) + ", and could not connect to another cluster node");
            }
            if (!this.exclusiveMode) {
                this.acceptConnections();
            }
            this.masterListenerStarted = false;
            this.activateSkeletons();
        }
    }

    private boolean startAsSlave() {
        if (this.candidatePorts.length == 1 && this.candidatePorts[0] == 0) {
            return false;
        }
        for (String address : this.cprops.getAddresses()) {
            try {
                this.startSlave(InetAddress.getByName(address), this.getClusterId());
                return true;
            }
            catch (IOException e) {
                log.debug("Unable to connect as slave to {}: {}", (Object)address, (Object)e.getMessage());
            }
        }
        try {
            this.startSlave(InetAddress.getByName("localhost"), this.getClusterId());
            return true;
        }
        catch (IOException e) {
            log.debug("Unable to connect as slave to localhost: {}", (Object)e.getMessage());
            return false;
        }
    }

    private InetAddress[] getAllMachineIPAddresses() {
        try {
            return InetAddress.getAllByName(InetAddress.getLocalHost().getHostName());
        }
        catch (UnknownHostException e) {
            log.warn("Unable to find IP address for local host: {}", (Object)e.getMessage());
            return new InetAddress[0];
        }
    }

    boolean isLocalIPAddress(InetAddress addr) {
        if (addr.isAnyLocalAddress()) {
            return true;
        }
        if (addr.isLoopbackAddress()) {
            return true;
        }
        for (InetAddress local : this.getAllMachineIPAddresses()) {
            if (!local.equals(addr)) continue;
            return true;
        }
        return false;
    }

    private void startSlave(InetAddress addr, String clusterId) throws IOException {
        ClusterSlave slave = new ClusterSlave(this, addr, this.candidatePorts, this.clusterNodeId, clusterId, this.connectTimeout, this.receiveTimeout);
        slave.start();
        this.setClusteredFlag(true);
        this.slave = slave;
        log.info("Node {} started as: slave, connected to address: {}", (Object)this.clusterNodeId, (Object)slave.getMasterSocketAddress());
    }

    boolean startMasterListener(boolean force) throws IOException {
        ClusterMaster master = new ClusterMaster(this, this.bindaddr, this.candidatePorts, this.clusterNodeId, this.getClusterId(), this.receiveTimeout);
        if (!master.startListener(force)) {
            return false;
        }
        this.master = master;
        InetSocketAddress sa = master.getSocketAddress();
        if (this.candidatePorts.length == 1 && this.candidatePorts[0] == 0) {
            this.candidatePorts[0] = sa.getPort();
        }
        log.info("Node {} started the master listener, on address: {} force: {}", new Object[]{this.clusterNodeId, sa, force});
        return true;
    }

    void acceptConnections() throws IOException {
        this.setClusteredFlag(false);
        this.master.acceptConnections();
        log.info("Node {} started as: master", (Object)this.clusterNodeId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setClusteredFlag(boolean value) {
        if (value) {
            if (!this.clustered.exists()) {
                log.info("Not alone: creating " + this.clustered.getAbsolutePath());
                PrintWriter out = null;
                try {
                    out = new PrintWriter(new FileWriter(this.clustered));
                    out.println("This repository is or was running as part of a cluster.");
                    out.println("If this file exists, this cluster node will only start as slave,");
                    out.println("in order to prevent cluster nodes to become out of sync.");
                    out.println("Please do not delete this marker file, unless if want to run this cluster node");
                    out.println("in standalone mode or as the new master node (in which case");
                    out.println("other cluster nodes may need to be re-created as copies of this node).");
                }
                catch (IOException e) {
                    log.warn("The file " + this.clustered.getAbsolutePath() + " could not be created - ignored", (Throwable)e);
                }
                finally {
                    if (out != null) {
                        out.close();
                    }
                }
            }
        } else if (this.clustered.exists()) {
            log.info("Alone: deleting " + this.clustered.getAbsolutePath());
            if (!this.clustered.delete()) {
                log.warn("The file " + this.clustered.getAbsolutePath() + " could not be deleted - ignored");
            }
        }
    }

    public void register(String id, ClusterSkeleton skel) throws IllegalStateException, IOException {
        this.checkNotStopped();
        if (this.autoStartStop) {
            this.start();
        }
        if (this.skeletons.containsKey(id)) {
            throw new IllegalStateException("Already registered: " + id);
        }
        this.skeletons.put(id, skel);
        if (skel instanceof ClusterLifecycleListener) {
            this.listeners.add((ClusterLifecycleListener)((Object)skel));
        }
        log.debug("Registered skeleton: {}", (Object)id);
        if (this.isMaster()) {
            skel.activate();
        }
    }

    public boolean unregister(String id) {
        ClusterSkeleton skel = this.skeletons.remove(id);
        if (skel != null) {
            if (this.isMaster()) {
                skel.deactivate();
            }
            if (skel instanceof ClusterLifecycleListener) {
                this.listeners.remove((ClusterLifecycleListener)((Object)skel));
            }
            if (this.skeletons.isEmpty() && this.autoStartStop && this.isStarted()) {
                this.stop();
            }
        }
        return skel != null;
    }

    public OutgoingCall newCall(String obj, int op) throws IOException {
        this.checkNotStopped();
        if (!this.isStarted()) {
            throw new IOException("Not started.");
        }
        if (this.slave == null) {
            throw new IOException("Not a slave.");
        }
        return this.slave.newCall(obj, op);
    }

    public OutgoingCall broadcast(String obj, int op) throws IOException {
        this.checkNotStopped();
        if (!this.isStarted()) {
            throw new IOException("Not started.");
        }
        if (this.master == null) {
            throw new IOException("Not a master.");
        }
        return this.master.broadcast(obj, op, null);
    }

    public OutgoingCall broadcast(String obj, int op, Set<String> exclude) throws IOException {
        this.checkNotStopped();
        if (!this.isStarted()) {
            throw new IOException("Not started.");
        }
        if (this.master == null) {
            throw new IOException("Not a master.");
        }
        return this.master.broadcast(obj, op, exclude);
    }

    void dispatch(IncomingCall call) throws IOException {
        if (this.paused.get()) {
            return;
        }
        String target = call.getTarget();
        if (target.equals("")) {
            this.dispatchInternal(call);
        } else {
            ClusterSkeleton skel = this.skeletons.get(target);
            if (skel == null) {
                call.error("No skeleton found with id: " + target);
            } else {
                skel.dispatch(call);
            }
        }
    }

    private void dispatchInternal(IncomingCall call) throws IOException {
        DataInput in = call.getInput();
        DataOutput out = call.getOutput();
        switch (call.getOperation()) {
            case 3: {
                this.deactivateSkeletons();
                this.master.stopListener();
                call.ok();
                break;
            }
            case 4: {
                call.ok();
                this.stopAndRestart();
                break;
            }
            case 5: {
                out = call.getOutput();
                out.writeUTF(this.getOS());
                String hostname = this.getBindAddrHostName();
                out.writeUTF(hostname != null ? hostname : "");
                out.writeUTF(this.getRepositoryHome().getAbsolutePath());
                break;
            }
            case 6: {
                out = call.getOutput();
                ClusterNodeInfo[] infos = this.master.getSlaveInfos();
                out.writeInt(infos.length);
                for (ClusterNodeInfo info : infos) {
                    out.writeUTF(info.getId());
                    out.writeUTF(info.getOS());
                    out.writeUTF(info.getHostname());
                    out.writeUTF(info.getRepositoryHome());
                }
                break;
            }
            case 7: {
                String path = in.readUTF();
                byte[] contents = FileUtils.readFileToByteArray((File)new File(this.repositoryHome, path));
                out.writeInt(contents.length);
                out.write(contents);
                break;
            }
            case 8: {
                this.slaveJoined(call.getInput().readUTF());
                call.ok();
                break;
            }
            case 9: {
                int cmembers = in.readInt();
                String[] members = new String[cmembers];
                for (int i = 0; i < cmembers; ++i) {
                    members[i] = in.readUTF();
                }
                int caddresses = in.readInt();
                String[] addresses = new String[caddresses];
                for (int i = 0; i < caddresses; ++i) {
                    addresses[i] = in.readUTF();
                }
                this.onSyncMembers(members, addresses);
                break;
            }
            default: {
                call.error("Operation not implemented: " + call.getOperation());
            }
        }
    }

    private void stopAndRestart() {
        Thread t = new Thread(new Runnable(){

            public void run() {
                ClusterController.this.doStopAndRestart();
            }
        });
        String name = String.format(String.format("Restarter (%s)", this.clusterNodeId), new Object[0]);
        t.setName(name);
        t.setDaemon(true);
        t.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void doStopAndRestart() {
        Object object = this.stateChangeLock;
        synchronized (object) {
            if (this.stopped.get()) {
                return;
            }
            this.doStop();
            try {
                this.restart();
            }
            catch (IOException e) {
                log.error(String.format("Unable to restart cluster node %s.", this.clusterNodeId), (Throwable)e);
                this.stop();
            }
        }
    }

    public Collection<String> activeSkeletons() {
        return Collections.unmodifiableCollection(this.skeletons.keySet());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void remoteCopy(String path) throws IOException, IllegalStateException {
        this.checkNotStopped();
        if (!this.isStarted()) {
            throw new IllegalStateException("Not started.");
        }
        if (this.slave == null) {
            throw new IllegalStateException("Not a slave.");
        }
        OutgoingCall call = this.slave.newCall("", 7);
        try {
            DataOutput out = call.getOutput();
            out.writeUTF(path);
            DataInput in = call.getInput();
            byte[] contents = new byte[in.readInt()];
            in.readFully(contents);
            FileUtils.writeByteArrayToFile((File)new File(this.repositoryHome, path), (byte[])contents);
        }
        finally {
            call.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() {
        HashMap<File, ClusterController> hashMap = this.stateChangeLock;
        synchronized (hashMap) {
            if (this.stopped.set(true)) {
                return;
            }
            this.doStop();
            this.skeletons.clear();
            this.listeners.clear();
        }
        hashMap = controllers;
        synchronized (hashMap) {
            controllers.remove(this.repositoryHome);
        }
    }

    private void doStop() {
        if (this.master != null) {
            this.deactivateSkeletons();
            this.master.stop();
            this.master = null;
        } else if (this.slave != null) {
            this.slave.stop();
            this.slave = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void activateSkeletons() {
        Map<String, ClusterSkeleton> map = this.skeletons;
        synchronized (map) {
            for (ClusterSkeleton skel : this.skeletons.values()) {
                skel.activate();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void deactivateSkeletons() {
        Map<String, ClusterSkeleton> map = this.skeletons;
        synchronized (map) {
            for (ClusterSkeleton skel : this.skeletons.values()) {
                skel.deactivate();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void blockIncoming() {
        Object object = this.stateChangeLock;
        synchronized (object) {
            if (this.master != null) {
                this.master.blockIncomingCalls();
            } else if (this.slave != null) {
                this.slave.blockIncomingCalls();
            }
        }
    }

    void disableRestart() {
        this.disableRestart = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isStarted() {
        Object object = this.stateChangeLock;
        synchronized (object) {
            return this.master != null || this.slave != null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isMaster() {
        Object object = this.stateChangeLock;
        synchronized (object) {
            return this.master != null && this.master.isRunning();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isSlave() {
        Object object = this.stateChangeLock;
        synchronized (object) {
            return this.slave != null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void masterDisconnected(String masterId, boolean abnormalTermination, boolean restartAfterAbnormalTermination) {
        log.info("Master disconnected: " + masterId);
        Object object = this.stateChangeLock;
        synchronized (object) {
            if (this.stopped.get()) {
                return;
            }
            if (this.master != null) {
                return;
            }
            if (this.slave != null) {
                this.slave.stop();
                this.slave = null;
            }
            if (!(this.disableRestart || abnormalTermination && !this.becomeMasterOnTimeout)) {
                try {
                    this.setClusteredFlag(false);
                    this.restart();
                }
                catch (IOException e) {
                    log.error("Unable to restart cluster node: " + this.clusterNodeId, (Throwable)e);
                }
            } else if (restartAfterAbnormalTermination) {
                log.warn("Connection to master terminated abnormally, slave restarting.");
                try {
                    this.restart();
                }
                catch (IOException e) {
                    log.error("Unable to restart cluster node: " + this.clusterNodeId, (Throwable)e);
                }
            } else {
                log.error("Connection to master terminated abnormally, slave stopped.");
                this.stop();
            }
        }
    }

    private void restart() throws IOException {
        this.attempt(new IOOperation(){

            public void execute() throws IOException {
                ClusterController.this.start();
            }
        }, null);
    }

    void attempt(IOOperation op, String message) throws IOException {
        IOException last = null;
        int delayBound = 1000;
        for (int i = 0; i < 3; ++i) {
            try {
                op.execute();
                return;
            }
            catch (IOException e) {
                if (message != null) {
                    log.debug(String.format("%s failed (attempt %d)", message, i + 1), (Throwable)e);
                }
                last = e;
                int delay = this.random.nextInt(delayBound);
                if (delay < delayBound / 2) {
                    delay += delayBound / 2;
                }
                delayBound *= 2;
                try {
                    Thread.sleep(delay);
                }
                catch (InterruptedException e2) {
                    // empty catch block
                }
                continue;
            }
        }
        if (last != null) {
            throw last;
        }
    }

    void slaveJoined(String slaveId) {
        try {
            if (this.cprops.addMember(slaveId)) {
                this.cprops.save();
            }
        }
        catch (IOException e) {
            log.warn("Unable to save properties.", (Throwable)e);
        }
    }

    void slaveConnected(String slaveId, InetAddress bindaddr) {
        log.info("Slave connected: " + slaveId);
        if (bindaddr != null && !this.isLocalIPAddress(bindaddr)) {
            this.cprops.addAddress(bindaddr.getHostAddress());
        }
    }

    void slaveDisconnected(String slaveId) {
        log.info("Slave disconnected: " + slaveId);
        for (ClusterLifecycleListener l : new ArrayList<ClusterLifecycleListener>(this.listeners)) {
            l.slaveDisconnected(slaveId);
        }
    }

    void onSyncMembers(String[] members, String[] addresses) {
        boolean modified = false;
        for (String member : members) {
            modified |= this.cprops.addMember(member);
        }
        for (String address : addresses) {
            modified |= this.cprops.addAddress(address);
        }
        if (modified) {
            this.cprops.saveQuietly();
        }
        if ((modified || members.length != this.cprops.getMembers().length || addresses.length != this.cprops.getAddresses().length) && this.isMaster()) {
            try {
                this.syncMembers(this.cprops.getMembers(), this.cprops.getAddresses());
            }
            catch (IOException e) {
                // empty catch block
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void syncMembers(String[] members, String[] addresses) throws IOException {
        OutgoingCall call = this.isMaster() ? this.master.broadcast("", 9, null) : this.slave.newCall("", 9, true);
        try {
            DataOutput out = call.getOutput();
            out.writeInt(members.length);
            for (String s : members) {
                out.writeUTF(s);
            }
            out.writeInt(addresses.length);
            for (String s : addresses) {
                out.writeUTF(s);
            }
            call.execute();
        }
        finally {
            call.release();
        }
    }

    String getOS() {
        return System.getProperty("os.name") + " " + System.getProperty("os.version");
    }

    private void checkNotStopped() throws IllegalStateException {
        if (this.stopped.get()) {
            throw new IllegalStateException("Controller is stopped.");
        }
    }

    boolean isMasterListenerStarted() {
        return this.masterListenerStarted;
    }

    public void kill() {
        this.deactivateSkeletons();
        this.stop();
        this.skeletons.clear();
    }

    public void setExclusiveMode(boolean value) {
        this.exclusiveMode = value;
    }

    public boolean isExclusiveMode() {
        return this.exclusiveMode;
    }

    static interface IOOperation {
        public void execute() throws IOException;
    }
}

